"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.PollingMachine = void 0;
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
class PollingMachine {
    constructor(scope, lambdaFunction) {
        let statementFailed1 = new sfn.Fail(scope, 'StatementFailed_');
        let statementSucceeded1 = new sfn.Succeed(scope, 'StatementSucceeded_');
        let executeBeforePollingRsTaskProcedure = new util_1.RetryableLambdaInvoke(scope, 'executeBeforePolling', {
            lambdaFunction: lambdaFunction,
            payloadResponseOnly: true,
            payload: sfn.TaskInput.fromObject({
                sqlStatement: 'select public.f_slow(getdate()::varchar(50), 60)',
            }),
            heartbeat: cdk.Duration.seconds(300),
            resultPath: '$.executionDetails',
        });
        executeBeforePollingRsTaskProcedure.addCatch(statementFailed1, { errors: ['States.ALL'] });
        let waitBetweenPolls = new sfn.Wait(scope, 'WaitBetweenPolls', {
            time: sfn.WaitTime.duration(cdk.Duration.seconds(10)),
        });
        executeBeforePollingRsTaskProcedure.next(waitBetweenPolls);
        let checkExecutionStateRSTask = new util_1.RetryableLambdaInvoke(scope, 'checkExecutionStateRSTask', {
            lambdaFunction: lambdaFunction,
            payloadResponseOnly: true,
            payload: sfn.TaskInput.fromObject({
                'statementId.$': '$.executionDetails.Id',
                'action': 'describeStatement',
            }),
            heartbeat: cdk.Duration.seconds(300),
            resultPath: '$.executionDetails',
        });
        waitBetweenPolls.next(checkExecutionStateRSTask);
        let choiceExecutionResult = new sfn.Choice(scope, 'choiceExecutionResult', {}).when(sfn.Condition.stringEquals('$.executionDetails.Status', 'FINISHED'), statementSucceeded1).when(sfn.Condition.stringEquals('$.executionDetails.Status', 'ABORTED'), statementFailed1).when(sfn.Condition.stringEquals('$.executionDetails.Status', 'FAILED'), statementFailed1).otherwise(waitBetweenPolls);
        checkExecutionStateRSTask.next(choiceExecutionResult);
        this.definition = new sfn.StateMachine(scope, 'PollingRsTask', {
            definition: executeBeforePollingRsTaskProcedure,
        });
    }
}
exports.PollingMachine = PollingMachine;
//# sourceMappingURL=data:application/json;base64,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