"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.CancellingStatementMachine = void 0;
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const util_1 = require("./util");
class CancellingStatementMachine {
    constructor(scope, lambdaFunction) {
        this.create_udf_slow = `
      create or replace function public.f_slow(str_in varchar(50), int_in int)
      RETURNS varchar(50)
      stable AS $$
        import time
        time.sleep(int_in)
        return str_in
      $$ LANGUAGE plpythonu;
  `;
        let cancelingSucceeded = new sfn.Succeed(scope, 'cancelingSucceeded');
        let cancelingFailed = new sfn.Fail(scope, 'cancelingFailed');
        let setupPreReqsCancelRSStatement = new util_1.RetryableLambdaInvoke(scope, 'provideUdfSlowExecution', {
            lambdaFunction: lambdaFunction,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                'taskToken': sfn.JsonPath.taskToken,
                'executionArn.$': '$$.Execution.Id',
                'sqlStatement': this.create_udf_slow,
            }),
            heartbeat: cdk.Duration.seconds(3600),
            resultPath: '$.executionDetails',
        }).addCatch(cancelingFailed, { errors: ['States.ALL'] });
        let cancelSlowQuery = new util_1.RetryableLambdaInvoke(scope, 'cancelSlowQuery', {
            lambdaFunction: lambdaFunction,
            payloadResponseOnly: true,
            payload: sfn.TaskInput.fromObject({
                'statementId': 'LATEST',
                'executionArn.$': '$$.Execution.Id',
                'action': 'cancelStatement',
            }),
            resultPath: '$.executionCancelDetails',
        }).addCatch(cancelingFailed, { errors: ['States.ALL'] });
        let startSlowQuery = new util_1.RetryableLambdaInvoke(scope, 'startSlowQuery', {
            lambdaFunction: lambdaFunction,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                'taskToken': sfn.JsonPath.taskToken,
                'executionArn.$': '$$.Execution.Id',
                'sqlStatement': 'select public.f_slow(getdate()::varchar(50), 600)',
            }),
            heartbeat: cdk.Duration.seconds(10),
            resultPath: '$.executionDetails',
        }).addCatch(cancelSlowQuery, { errors: ['States.Timeout'] }).addCatch(cancelingFailed, { errors: ['States.ALL'] }).next(cancelingFailed); // We should not normally transition because heartbeat < statement duration
        setupPreReqsCancelRSStatement.next(startSlowQuery);
        let describeCanceledSlowQuery = new util_1.RetryableLambdaInvoke(scope, 'describeCanceledSlowQuery', {
            lambdaFunction: lambdaFunction,
            payloadResponseOnly: true,
            payload: sfn.TaskInput.fromObject({
                'statementId': 'LATEST',
                'executionArn.$': '$$.Execution.Id',
                'action': 'describeStatement',
            }),
            resultPath: '$.executionDetails',
        }).addCatch(cancelingFailed, { errors: ['States.ALL'] });
        cancelSlowQuery.next(describeCanceledSlowQuery);
        let verifyCanceledQuery = new sfn.Choice(scope, 'verifyCancelledQuery', {}).when(sfn.Condition.booleanEquals('$.executionCancelDetails.Status', false), cancelingFailed).when(
        //If query  still in started state then retry describe (better to have retry backoff mechanism)
        sfn.Condition.stringEquals('$.executionDetails.Status', 'STARTED'), describeCanceledSlowQuery).when(sfn.Condition.not(sfn.Condition.stringEquals('$.executionDetails.Status', 'ABORTED')), cancelingFailed).otherwise(cancelingSucceeded);
        describeCanceledSlowQuery.next(verifyCanceledQuery);
        this.definition = new sfn.StateMachine(scope, 'CancelRsTask', {
            definition: setupPreReqsCancelRSStatement,
        });
    }
}
exports.CancellingStatementMachine = CancellingStatementMachine;
//# sourceMappingURL=data:application/json;base64,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