"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.IntegTesting = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const redshift = require("@aws-cdk/aws-redshift");
const cdk = require("@aws-cdk/core");
const index_1 = require("./index");
const cancelling_statement_1 = require("./machines/cancelling_statement");
const chained_machines_1 = require("./machines/chained_machines");
const parallel_no_concurrency_1 = require("./machines/parallel_no_concurrency");
const polling_1 = require("./machines/polling");
const single_failure_1 = require("./machines/single_failure");
const single_success_1 = require("./machines/single_success");
const success_and_fail_1 = require("./machines/success_and_fail");
class IntegTesting {
    constructor() {
        const app = new cdk.App();
        const env = {
            region: process.env.CDK_DEFAULT_REGION,
            account: process.env.CDK_DEFAULT_ACCOUNT,
        };
        const stack = new cdk.Stack(app, 'integ-test-stack', { env });
        let redshiftUsername = 'admin';
        let redshiftDbName = 'dev';
        let cluster = new redshift.Cluster(stack, 'rsCluster', {
            vpc: new ec2.Vpc(stack, 'vpc', {
                subnetConfiguration: [
                    {
                        cidrMask: 28,
                        name: 'isolated',
                        subnetType: ec2.SubnetType.ISOLATED,
                    },
                ],
            }),
            masterUser: {
                masterUsername: redshiftUsername,
            },
            defaultDatabaseName: redshiftDbName,
            clusterType: redshift.ClusterType.SINGLE_NODE,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            vpcSubnets: { subnetType: ec2.SubnetType.ISOLATED },
        });
        let rs_task_helper = new index_1.SfnRedshiftTasker(stack, 'RSTask', {
            redshiftTargetProps: {
                dbUser: redshiftUsername,
                dbName: redshiftDbName,
                clusterIdentifier: cluster.clusterName,
            },
            logLevel: 'DEBUG',
        });
        //Deploying separate function to allow access to another user without duplicating infra or powertools layer.
        let rs_task_helper2 = new index_1.SfnRedshiftTasker(stack, 'RSTaskUser2', {
            redshiftTargetProps: {
                dbUser: 'user2',
                dbName: redshiftDbName,
                clusterIdentifier: cluster.clusterName,
            },
            existingTableObj: rs_task_helper.trackingTable,
            createCallbackInfra: false,
            powertoolsArn: rs_task_helper.powertoolsArn,
        });
        let chainedMachine = new chained_machines_1.ChainedMachine(stack);
        chainedMachine.push_front('singleFailure', new single_failure_1.SingleFailureMachine(stack, rs_task_helper2.lambdaFunction).definition);
        chainedMachine.push_front('singleSuccess', new single_success_1.SingleSuccessMachine(stack, rs_task_helper.lambdaFunction).definition);
        chainedMachine.push_front('parallelNoConcurrency', new parallel_no_concurrency_1.ParallelNoConcurrencyMachine(stack, rs_task_helper.lambdaFunction).definition);
        chainedMachine.push_front('successAndFail', new success_and_fail_1.SuccessAndFailMachine(stack, rs_task_helper.lambdaFunction).definition);
        chainedMachine.push_front('polling', new polling_1.PollingMachine(stack, rs_task_helper.lambdaFunction).definition);
        chainedMachine.push_front('cancelling', new cancelling_statement_1.CancellingStatementMachine(stack, rs_task_helper.lambdaFunction).definition);
        chainedMachine.build();
        this.stack = [stack];
    }
}
exports.IntegTesting = IntegTesting;
// run the integ testing
new IntegTesting();
//# sourceMappingURL=data:application/json;base64,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