"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SfnRedshiftTasker = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const assert = require("assert");
const path = require("path");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const logs = require("@aws-cdk/aws-logs");
const sam = require("@aws-cdk/aws-sam");
const cdk = require("@aws-cdk/core");
const aws_events_rule_sqs_1 = require("@aws-solutions-constructs/aws-events-rule-sqs");
const aws_lambda_dynamodb_1 = require("@aws-solutions-constructs/aws-lambda-dynamodb");
const aws_sqs_lambda_1 = require("@aws-solutions-constructs/aws-sqs-lambda");
/**
 * Create infrastructure to easily create tasks in a Stepfunction that run a SQL statement on a Redshift cluster and await completion.
 *
 * @stability stable
 */
class SfnRedshiftTasker extends cdk.Construct {
    /**
     * Creates the infrastructure to allow stepfunction tasks that execute SQL commands and await their completion.
     *
     * @param scope Scope within where this infrastructure is created.
     * @param id Identifier to name this building block.
     * @param props The configuration properties of the infrastructure.
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        let lambdaP = path.join(__dirname, '../lambda');
        let pythonP = path.join(lambdaP, 'python');
        let rsIntegrationFunctionP = path.join(pythonP, 'rs_integration_function');
        let ddbP = path.join(rsIntegrationFunctionP, 'ddb');
        let ddbInitP = path.join(ddbP, '__init__.py');
        let rsIntegrationFunctionEnvVarP = path.join(rsIntegrationFunctionP, 'environment_labels.py');
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const PropertiesReader = require('properties-reader');
        const stripSurroundingQuotes = (x) => x.replace(/^['"](.+)['"]$/, '$1');
        // Define helper function that return the string value from a Python string
        let ddbProps = new PropertiesReader(ddbInitP);
        const getDdbProp = (x) => stripSurroundingQuotes(ddbProps.get(x));
        /**
         * Make sure we set environment variable that our Lambda function expects.
         */
        let DDB_TTL = getDdbProp('DDB_TTL');
        let DDB_ID = getDdbProp('DDB_ID');
        let DDB_INVOCATION_ID = getDdbProp('DDB_INVOCATION_ID');
        let rsProcedureStarterEnvProps = new PropertiesReader(rsIntegrationFunctionEnvVarP);
        const getRsProcedureStarterEnvProp = (x) => stripSurroundingQuotes(rsProcedureStarterEnvProps.get(x));
        let CLUSTER_IDENTIFIER = getRsProcedureStarterEnvProp('CLUSTER_IDENTIFIER');
        let DATABASE = getRsProcedureStarterEnvProp('DATABASE');
        let DB_USER = getRsProcedureStarterEnvProp('DB_USER');
        if (props.powertoolsArn === undefined) {
            let powertools = new sam.CfnApplication(this, 'Powertools', {
                location: {
                    applicationId: 'arn:aws:serverlessrepo:eu-west-1:057560766410:applications/aws-lambda-powertools-python-layer',
                    semanticVersion: '1.11.0',
                },
            });
            this.powertoolsArn = powertools.getAtt('Outputs.LayerVersionArn').toString();
        }
        else {
            this.powertoolsArn = props.powertoolsArn;
        }
        let defaultDynamoTableProps = {
            partitionKey: { name: DDB_ID, type: dynamodb.AttributeType.STRING },
            sortKey: { name: DDB_INVOCATION_ID, type: dynamodb.AttributeType.STRING },
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            timeToLiveAttribute: DDB_TTL,
        };
        let defaultLambdaFunctionProps = {
            code: new lambda.AssetCode(rsIntegrationFunctionP),
            handler: 'index.handler',
            runtime: lambda.Runtime.PYTHON_3_8,
            environment: {
                // DynamoDB table environment variable gets automatically added by LambdaToDynamoDB
                [CLUSTER_IDENTIFIER]: props.redshiftTargetProps.clusterIdentifier,
                [DATABASE]: props.redshiftTargetProps.dbName,
                [DB_USER]: props.redshiftTargetProps.dbUser,
                [DDB_TTL]: '1',
                LOG_LEVEL: props.logLevel || 'INFO',
            },
            layers: [lambda.LayerVersion.fromLayerVersionArn(this, 'powertoolsVersion', this.powertoolsArn)],
            logRetention: logs.RetentionDays.ONE_YEAR,
            timeout: cdk.Duration.seconds(29),
            reservedConcurrentExecutions: 1,
        };
        const existingTableErr = 'Must pass existing helper table via "existingTableObj" if createCallBackInfra is set to false';
        assert(props.createCallbackInfra || props.createCallbackInfra === undefined || props.existingTableObj !== undefined, existingTableErr);
        // When an existing lambda function is provided re-use it otherwise create one using the provided properties
        let lambdaDetails;
        if (props.starterExistingLambdaObj === undefined) {
            lambdaDetails = { lambdaFunctionProps: { ...defaultLambdaFunctionProps, ...props.starterLambdaFunctionProps } };
        }
        else {
            lambdaDetails = { existingLambdaObj: props.starterExistingLambdaObj };
        }
        // When an existing DDB table is provided re-use it otherwise create one using the provided properties
        let ddbDetails;
        if (props.existingTableObj === undefined) {
            ddbDetails = { dynamoTableProps: { ...defaultDynamoTableProps, ...props.dynamoTableProps } };
        }
        else {
            ddbDetails = { existingTableObj: props.existingTableObj };
        }
        let lambda_ddb = new aws_lambda_dynamodb_1.LambdaToDynamoDB(this, 'RSInvoker', {
            ...lambdaDetails,
            ...ddbDetails,
            tablePermissions: props.tablePermissions || 'ReadWrite',
        });
        this.lambdaFunction = lambda_ddb.lambdaFunction;
        this.trackingTable = lambda_ddb.dynamoTable;
        let allowRedshiftDataApiExecuteStatement = new iam.PolicyStatement({
            actions: ['redshift-data:ExecuteStatement', 'redshift-data:DescribeStatement',
                'redshift-data:GetStatementResult', 'redshift-data:CancelStatement', 'redshift-data:ListStatements'],
            effect: iam.Effect.ALLOW,
            resources: ['*'],
        });
        let allowRedshiftGetCredentials = new iam.PolicyStatement({
            actions: ['redshift:GetClusterCredentials'],
            effect: iam.Effect.ALLOW,
            resources: [
                cdk.Fn.sub('arn:${AWS::Partition}:redshift:${AWS::Region}:${AWS::AccountId}:dbname:${ID}/${DB}', {
                    ID: props.redshiftTargetProps.clusterIdentifier,
                    DB: props.redshiftTargetProps.dbName,
                }),
                cdk.Fn.sub('arn:${AWS::Partition}:redshift:${AWS::Region}:${AWS::AccountId}:dbuser:${ID}/${DB_USER}', {
                    ID: props.redshiftTargetProps.clusterIdentifier,
                    DB_USER: props.redshiftTargetProps.dbUser,
                }),
            ],
        });
        this.lambdaFunction.addToRolePolicy(allowRedshiftDataApiExecuteStatement);
        this.lambdaFunction.addToRolePolicy(allowRedshiftGetCredentials);
        if (props.createCallbackInfra === undefined || props.createCallbackInfra) {
            let allowReportTaskOutcome = new iam.PolicyStatement({
                actions: ['states:SendTaskSuccess', 'states:SendTaskFailure'],
                effect: iam.Effect.ALLOW,
                resources: ['*'],
            });
            let completerLambdaDetails;
            if (props.completerExistingLambdaObj === undefined && props.completerLambdaFunctionProps === undefined) {
                //We fall back on re-using the function that starts execution.
                completerLambdaDetails = { existingLambdaObj: this.lambdaFunction };
            }
            else {
                if (props.completerExistingLambdaObj === undefined) {
                    completerLambdaDetails = { lambdaFunctionProps: { ...defaultLambdaFunctionProps, ...props.completerLambdaFunctionProps } };
                }
                else {
                    completerLambdaDetails = { existingLambdaObj: props.completerExistingLambdaObj };
                }
            }
            let completerIntegration = new aws_lambda_dynamodb_1.LambdaToDynamoDB(this, 'Completer', {
                ...completerLambdaDetails,
                existingTableObj: this.trackingTable,
            });
            let eventQueue = new aws_events_rule_sqs_1.EventsRuleToSqs(this, 'QueryFinished', {
                eventRuleProps: {
                    description: 'Monitor queries that have been issued by Redshift data API and that completed',
                    enabled: true,
                    eventPattern: {
                        source: ['aws.redshift-data'],
                        detailType: ['Redshift Data Statement Status Change'],
                    },
                },
                existingQueueObj: props.existingQueueObj,
                queueProps: props.queueProps,
                enableQueuePurging: props.enableQueuePurging,
                deadLetterQueueProps: props.deadLetterQueueProps,
                deployDeadLetterQueue: props.deployDeadLetterQueue,
                maxReceiveCount: props.maxReceiveCount,
                enableEncryptionWithCustomerManagedKey: props.enableEncryptionWithCustomerManagedKey,
                encryptionKey: props.encryptionKey,
                encryptionKeyProps: props.encryptionKeyProps,
            });
            new aws_sqs_lambda_1.SqsToLambda(this, 'SqsToCompleter', {
                existingLambdaObj: completerIntegration.lambdaFunction,
                existingQueueObj: eventQueue.sqsQueue,
            });
            completerIntegration.lambdaFunction.addToRolePolicy(allowReportTaskOutcome);
        }
        else {
            // No callback infrastructure needed
            let no_queue_err = 'Queue is part of SFN callback infra so cannot be provided if sfnCallbackSupport == false';
            assert(props.existingQueueObj === undefined, no_queue_err);
            assert(props.queueProps === undefined, no_queue_err);
            assert(props.enableQueuePurging === undefined, no_queue_err);
            assert(props.deadLetterQueueProps === undefined, no_queue_err);
            assert(props.deployDeadLetterQueue === undefined, no_queue_err);
            assert(props.maxReceiveCount === undefined, no_queue_err);
            assert(props.enableEncryptionWithCustomerManagedKey === undefined, no_queue_err);
            assert(props.encryptionKey === undefined, no_queue_err);
            assert(props.encryptionKeyProps === undefined, no_queue_err);
        }
    }
}
exports.SfnRedshiftTasker = SfnRedshiftTasker;
_a = JSII_RTTI_SYMBOL_1;
SfnRedshiftTasker[_a] = { fqn: "cdk-stepfunctions-redshift.SfnRedshiftTasker", version: "1.114.1" };
//# sourceMappingURL=data:application/json;base64,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