# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['sparkml_base_classes']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'sparkml-base-classes',
    'version': '0.1.6',
    'description': 'SparkML base classes for Transformers and Estimators',
    'long_description': 'Custom SparkML Pipelines\n========================\n\nThis document includes an example of how to build a custom Estimator and Transformer using the base classes in this package, and how to integrate them with SparkML Pipelines. For information about the SparkML Pipelines concepts and use of existing Estimators and Transformers within the SparkML module, please refer to the `Spark ML Pipelines <https://spark.apache.org/docs/latest/ml-pipeline.html>`__ documentation.\n\nInstallation\n------------\n\n.. code-block:: bash\n\n    pip install -U sparkml-base-classes\n\nBuild a custom Transformer\n--------------------------\n\nIn this section we build a Transformer that adds a constant to a column and updates the column\'s values in-place.\n\n.. note::\n\n    The arguments set in the `__init__` method will be accessible in the `_transform` method as attributes by adding an underscore prefix to the argument name. In the example below the arguments are `column_name` and `value`, and they are available as attributes in the `_transform` method as `self._column_name` and `self._value`.\n\n.. code-block:: python\n\n    import pyspark.sql.functions as F\n    from pyspark import keyword_only\n    from sparkml_base_classes import TransformerBaseClass\n\n\n    class AdditionColumnTransformer(TransformerBaseClass):\n\n        @keyword_only\n        def __init__(self, column_name=None, value=None):\n            super().__init__()\n\n        def _transform(self, ddf):\n            self._logger.info("AdditionColumn transform with column {self._column_name}")\n            ddf = ddf.withColumn(self._column_name, F.col(self._column_name) + self._value)\n            return ddf\n\nBuild a custom Estimator\n------------------------\n\nIn this section we build an Estimator that normalizes the values of a column by the mean. An Estimator\'s `_fit` method must return a Transformer because the use of an Estimator consists of 2 steps:\n\n1. Fitting the estimator.\n    This step consists of using the `_fit` method to calculate some value(s) from the DataFrame and return a Transformer that stores the calculated value(s) and use them in the `_transform` method to transform a DataFrame. In this example the Estimator calculates the mean and returns a Transformer that divides the column by this mean value.\n\n2. Transforming the DataFrame.\n    Once the Estimator has been fitted and a Transformer has been returned, then we use the returned Transformer to transform the DataFrame. In this case the Transformer divides the specified column by the mean and returns the transformed DataFrame.\n\n.. note::\n\n    The arguments set in the `__init__` method will be accessible in the `_transform` and `_fit` methods as attributes by adding an underscore prefix to the argument name. In the example below the arguments are `column_name` and `mean`, and they are available as attributes in the `_transform` and `_fit` method as `self._column_name` and `self._mean`.\n\n.. code-block:: python\n\n    import pyspark.sql.functions as F\n    from pyspark import keyword_only\n    from sparkml_base_classes import EstimatorBaseClass, TransformerBaseClass\n\n    class MeanNormalizerTransformer(TransformerBaseClass):\n\n        @keyword_only\n        def __init__(self, column_name=None, mean=None):\n            super().__init__()\n\n        def _transform(self, ddf):\n            # add your transformation logic here\n            self._logger.info("MeanNormalizer transform")\n            ddf = ddf.withColumn(self._column_name, F.col(self._column_name) / self._mean)\n            return ddf\n\n    class MeanNormalizerEstimator(EstimatorBaseClass):\n\n        @keyword_only\n        def __init__(self, column_name=None):\n            super().__init__()\n\n        def _fit(self, ddf):\n            # add your transformation logic here\n            self._logger.info("MeanNormalizer fit")\n            mean, = ddf.agg(F.mean(self._column_name)).head()\n            return MeanNormalizerTransformer(\n                column_name=self._column_name,\n                mean=mean\n            )\n\nBuild the Pipeline\n------------------\n\nIn this section we will build a Pipeline containing our custom Transformer and Estimator. We will first initialize both classes and then add them as stages to the Pipeline.\n\n.. note::\n    We can also use Transformers and Estimators individually by calling their respective `_transform` and `_fit` methods, the advantage of using a Pipeline is to chain them together therefore reducing the code maintenance needed. In addition, it is a good practice to always use them as part of a Pipeline.\n\n\n.. code-block:: python\n\n    from pyspark.ml import Pipeline\n\n    multiply_column_transformer = AdditionColumnTransformer(column_name="foo", value=2)\n    mean_normalizer_estimator = MeanNormalizerEstimator(column_name="foo")\n    my_pipeline = Pipeline(stages=[multiply_column_transformer, mean_normalizer_estimator])\n\nFit the Pipeline and transform the DataFrame\n--------------------------------------------\n\nIn this section we will fit the created Pipeline to a DataFrame and then use the fitted Pipeline (or PipelineModel in SparkML terms) to transform a DataFrame. Thus, after a Pipeline’s fit method runs, it produces a PipelineModel, which is a Transformer. This PipelineModel can be later used to transform any DataFrame. Please refer to the `Spark ML Pipelines <https://spark.apache.org/docs/latest/ml-pipeline.html#how-it-works>`__ documentation for an in-depth description.\n\n.. note::\n    After fitting a Pipeline, the stages containing an Estimator will now contain the Transformer returned in the Estimator\'s `_fit` method.\n\n.. note::\n    The returned object of fitting a Pipeline is not a Pipeline object but a PipelineModel.\n\n.. code-block:: python\n\n    from pyspark.sql import SparkSession\n    from pyspark.ml import Pipeline\n\n    spark = SparkSession.builder.getOrCreate()\n\n    ddf = spark.createDataFrame(\n        [[1], [2], [3]],\n        ["foo"],\n    )\n\n    # the returned object is of PipelineModel type\n    my_fitted_pipeline = my_pipeline.fit(ddf)\n    my_fitted_pipeline.transform(ddf).show()\n\n    +----+\n    | foo|\n    +----+\n    |0.75|\n    | 1.0|\n    |1.25|\n    +----+\n\nSave and load fitted Pipeline\n-----------------------------\n\nIn the previous section we transformed the DataFrame immediately after fitting the Pipeline, in this section we will use an intermediary saving mechanism that allows us to decouple the fitting of the Pipeline from the transforming of the DataFrame.\n\n.. note::\n    It is a good practice to save the Pipeline using the `.pipeline` extension.\n\n.. note::\n    If you are using Spark in an AWS service, like SageMaker, the path to save the model can be an S3 path. This will work out-of-the-box given that the correct permission to read/write to that path are set.\n\n.. code-block:: python\n\n    from pyspark.ml import PipelineModel\n    from pyspark.sql import SparkSession\n\n    spark = SparkSession.builder.getOrCreate()\n\n    ddf = spark.createDataFrame(\n        [[8], [10], [12]],\n        ["foo"],\n    )\n\n    my_fitted_pipeline.save(\'my_fitted_pipeline.pipeline\')\n    my_fitted_pipeline = PipelineModel.load(\'my_fitted_pipeline.pipeline\')\n    my_fitted_pipeline.transform(ddf).show()\n\n    +----+\n    | foo|\n    +----+\n    | 2.5|\n    |   3|\n    | 3.5|\n    +----+\n\n',
    'author': 'Rodrigo Agundez',
    'author_email': 'rragundez@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
