# -*- coding: utf-8 -*-
from distutils.core import setup

packages = \
['confect']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'confect',
    'version': '0.2.5',
    'description': 'Python configuration library that provides pleasant configuration definition and access interface, and it reads unrestricted python configuration file.',
    'long_description': "Confect\n=======\n\n``confect`` is a Python configuration library with the following features.\n\n- A unified and pleasant configuration definition and accessing interface\n- Immutable conf object for reducing the possibility of making errors\n- Configuration file written in Python. This makes it possible to\n  dynamically handle complicated logic in the configuration file, read\n  other TOML/YMAL/JSON files or even environment variables in it.\n\nInstall\n-------\n\n``confect`` is a Python package hosted on PyPI and works only on Python 3.6 up.\n\nJust like other Python package, install it by `pip\n<https://pip.pypa.io/en/stable/>`_ into a  `virtualenv <https://hynek.me/articles/virtualenv-lives/>`_, or use `poetry <https://poetry.eustace.io/>`_ to\nautomatically create and manage the virtualenv.\n\n.. code:: console\n\n   $ pip install confect\n\n\nBasic Usage\n-----------\n\nCalling ``conf = confect.Conf()`` creates a new configuration manager object.\nAll configuration properties reside in it and are accessable through get\nattribute like this ``conf.group_name.prop_name``. It is possible to create\nmultiple ``Conf`` objects, but normally we don't need it. In most cases,\ninitialize only one ``Conf`` object in some module, then import and use it\nanywhere in your application.\n\nPut following lines in your application package. For example, in\n``your_package.__init__.py`` or ``your_package.core.py``.\n\n.. code:: python\n\n   from confect import conf\n   conf = Conf()\n\nConfiguration properties should be declared before using it. Use\n``Conf.declare_group(group_name)`` context manager to declare a configuration\ngroup and all properties under it. It's nessasery to provide a default value for\neach properties. Default values can be any type as long as the value can be\ndeepcopy. Group names shuold be valid attribute names.\n\nPut your configuration group declaration code in modules where you need those\nproperties. And make sure that the declaration is before all the lines that\naccess these properties. Normally, the group name is your class name, module\nname or subpackage name.\n\n\n.. code:: python\n\n   from your_package import conf\n   with conf.declare_group('yummy') as yummy:\n       yummy.kind = 'seafood'\n       yummy.name = 'fish'\n       yummy.weight = 10\n\n   def print_yummy():\n       # get some configuration through `conf.group_name.prop_name`\n       print(f'{conf.yummy.kind} {conf.yummy.name} {conf.yummy.weight}')\n\n   class Yummy:\n       # keep a configuration group in a variable\n       yummy_conf = conf.yummy\n\n       def get_yummy(self):\n           if self.yummy_conf.kind == 'seafood':\n               fishing_on_the_sea()\n\nConfiguration properties and groups are immutable. They can only be globally\nchanged by loading configuration files. Otherwise, they are always default\nvalues.\n\n>>> conf.yummy.name = 'octopus'\nTraceback (most recent call last):\n   ...\nconfect.error.FrozenConfPropError: Configuration properties are frozen.\n\nConfiguration File\n------------------\n\nUse ``Conf.load_conf_file(path)`` or ``Conf.load_conf_module(module_name)`` to\nload configuration files. No matter it is loaded before or after\ngroups/properties declaration, property values in configuration file always\noverride default values. Loading multiple files is possible, the latter one\nwould replace old values.\n\nBe aware, you should access your configuration properties after load\nconfiguration files. If not, you might get wrong/default value. Therefore, we\nusually load configure file right after initilize the ``Conf``, and before\nloading all other modules that might access this ``Conf`` object.\n\nFor instance, have the following code in ``your_package/core.py``, and ``from\nyour_package.core import conf`` in other module.\n\n.. code:: python\n\n   import sys\n   from confect import Conf\n   conf = Conf()\n\n   if len(sys.argv) == 2:\n       conf.load_conf_file(sys.argv[1])\n   else:\n       conf.load_conf_file('default/path/to/conf.py')\n\nThe default configuration file is in Python. That makes your configuration file\nprogrammable and unrestricted. In configuration file, import ``confect.c``\nobject and set all properties on it. Here's an example of configuration file.\n\n.. code-block:: python\n\n   from confect import c\n\n   c.yummy.kind = 'poultry'\n   c.yummy.name = 'chicken'\n   c.yummy.weight = 25\n\n   import os\n   c.cache.expire = 60 * 60 # one hour\n   c.cache.key = os.environ['CACHE_KEY']\n\n   DEBUG = True\n   if DEBUG:\n       c.cache.disable = True\n\n   import json\n   with open('secret.json') as f:\n       secret = json.load(f)\n\n   c.secret.key = secret['key']\n   c.secret.token = secret['token']\n\nThe ``c`` object only exits when loading a python configuration file, it's not\npossible to import it in your source code. You can set any property in any\nconfiguration group onto the ``c`` object. However, they are only accessable if\nyou declared it in the source code with ``Conf.declare_group(group_name)``.\n\nIf it's hard for you to specify the path of configuration file. You can load it\nthrough the import system of Python. Put your configuration file somewhere under\nyour package or make ``PYTHONPATH`` pointing to the directory it resides. Then\nload it with ``Conf.load_conf_module(module_name)``.\n\n.. code:: console\n\n   $ edit my_conf.py\n   $ export PYTHONPATH=.\n   $ python your_application.py\n\n\n.. code:: python\n\n   from confect import Conf\n   conf = Conf()\n   conf.load_conf_module('my_conf')\n\nLoad Environment Variables\n---------------------------\n\n``Conf.load_envvars(prefix: str)`` automatically searches environment variables\nin ``<prefix>__<group>__<prop>`` format. All of these three identifier are case\nsensitive. If you have a configuration property ``conf.cache.expire_time`` and\nyou call ``Conf.load_envvars('proj_X')``. It will set that ``expire_time``\nproperty to the parsed value of ``proj_X__cache__expire_time`` environment\nvariable.\n\n>>> import os\n>>> os.environ['proj_X.cache.expire'] = '3600'\n\n>>> conf = confect.new_conf()\n>>> conf.load_envvars('proj_X')  # doctest: +SKIP\n\nConfect includes predefined parsers of these primitive types.\n\n- ``str``\n- ``int``\n- ``float``\n- ``bytes``\n- ``datetime.datetime``\n- ``datetime.date``\n- ``tuple``\n- ``dict``\n- ``list``\n\nMutable Environment\n-----------------\n\n``Conf.mutate_locally()`` context manager creates an environment that makes ``Conf``\nobject temporarily mutable. All changes would be restored when it leaves the\nblock. It is usaful on writing test case or testing configuration properties in Python REPL.\n\n>>> conf = Conf()\n>>> conf.declare_group(  # declare group through keyword arguments\n...      'dummy',\n...      prop1=3,\n...      prop2='some string')\n...\n>>> with conf.mutate_locally():\n...      conf.dummy.prop1 = 5\n...      print(conf.dummy.prop1)\n5\n...     call_some_function_use_this_property()\n>>> print(conf.dummy.prop1)  # all configuration restored\n3\n\n\nTo-Dos\n======\n\n- A function for loading dictionary into ``conflect.c``.\n- A function that loads command line arguments and overrides configuration properties.\n- Copy-on-write mechenism in ``conf.mutate_locally()`` for better performance and memory usage.\n- API reference page\n",
    'author': '顏孜羲',
    'author_email': 'joseph.yen@gmail.com',
    'url': 'https://github.com/d2207197/confect',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
