"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.WebApp = void 0;
const path = require("path");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const r53 = require("@aws-cdk/aws-route53");
const r53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("../backend/shared/constants");
const caching_1 = require("../caching");
const monitored_certificate_1 = require("../monitored-certificate");
const storage_1 = require("../s3/storage");
const temp_file_1 = require("../temp-file");
const config_1 = require("./config");
const response_function_1 = require("./response-function");
class WebApp extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(this);
        this.bucket = storageFactory.newBucket(this, 'WebsiteBucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        // generate a stable unique id for the cloudfront function and use it
        // both for the function name and the logical id of the function so if
        // it is changed the function will be recreated.
        // see https://github.com/aws/aws-cdk/issues/15523
        const functionId = `AddHeadersFunction${this.node.addr}`;
        const behaviorOptions = {
            compress: true,
            cachePolicy: cloudfront.CachePolicy.CACHING_OPTIMIZED,
            functionAssociations: [{
                    function: new response_function_1.ResponseFunction(this, functionId, {
                        functionName: functionId,
                    }),
                    eventType: cloudfront.FunctionEventType.VIEWER_RESPONSE,
                }],
        };
        this.distribution = new cloudfront.Distribution(this, 'Distribution', {
            defaultBehavior: { origin: new origins.S3Origin(this.bucket), ...behaviorOptions },
            domainNames: props.domain ? [props.domain.zone.zoneName] : undefined,
            certificate: props.domain ? props.domain.cert : undefined,
            defaultRootObject: 'index.html',
            errorResponses: [404, 403].map(httpStatus => ({
                httpStatus,
                responseHttpStatus: 200,
                responsePagePath: '/index.html',
            })),
            minimumProtocolVersion: cloudfront.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        // The base URL is currently the custom DNS if any was used, or the distribution domain name.
        // This needs changing in case, for example, we add support for a custom URL prefix.
        this.baseUrl = `https://${props.domain ? props.domain.zone.zoneName : this.distribution.distributionDomainName}`;
        const jsiiObjOrigin = new origins.S3Origin(props.packageData);
        this.distribution.addBehavior('/data/*', jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.CATALOG_KEY}`, jsiiObjOrigin, behaviorOptions);
        this.distribution.addBehavior(`/${constants_1.VERSION_TRACKER_KEY}`, jsiiObjOrigin, behaviorOptions);
        if (props.packageStats) {
            this.distribution.addBehavior(`/${props.packageStats.statsKey}`, jsiiObjOrigin, behaviorOptions);
        }
        // if we use a domain, and A records with a CloudFront alias
        if (props.domain) {
            // IPv4
            new r53.ARecord(this, 'ARecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // IPv6
            new r53.AaaaRecord(this, 'AaaaRecord', {
                zone: props.domain.zone,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this.distribution)),
                comment: 'Created by the AWS CDK',
            });
            // Monitor certificate expiration
            if ((_a = props.domain.monitorCertificateExpiration) !== null && _a !== void 0 ? _a : true) {
                const monitored = new monitored_certificate_1.MonitoredCertificate(this, 'ExpirationMonitor', {
                    certificate: props.domain.cert,
                    domainName: props.domain.zone.zoneName,
                });
                props.monitoring.addHighSeverityAlarm('ACM Certificate Expiry', monitored.alarmAcmCertificateExpiresSoon);
                props.monitoring.addHighSeverityAlarm('Endpoint Certificate Expiry', monitored.alarmEndpointCertificateExpiresSoon);
            }
        }
        // "website" contains the static react app
        const webappDir = path.join(__dirname, '..', '..', 'website');
        new s3deploy.BucketDeployment(this, 'DeployWebsite', {
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            sources: [s3deploy.Source.asset(webappDir)],
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        // Generate config.json to customize frontend behavior
        const config = new config_1.WebappConfig({
            packageLinks: props.packageLinks,
            packageTags: props.packageTags,
            featuredPackages: props.featuredPackages,
            showPackageStats: (_b = props.showPackageStats) !== null && _b !== void 0 ? _b : props.packageStats !== undefined,
            featureFlags: props.featureFlags,
            categories: props.categories,
        });
        // Generate preload.js
        const preloadScript = new temp_file_1.TempFile('preload.js', (_c = props.preloadScript) !== null && _c !== void 0 ? _c : '');
        new s3deploy.BucketDeployment(this, 'DeployWebsiteConfig', {
            sources: [s3deploy.Source.asset(config.file.dir), s3deploy.Source.asset(preloadScript.dir)],
            destinationBucket: this.bucket,
            distribution: this.distribution,
            prune: false,
            cacheControl: caching_1.CacheStrategy.default().toArray(),
        });
        new core_1.CfnOutput(this, 'DomainName', {
            value: this.distribution.domainName,
            exportName: 'ConstructHubDomainName',
        });
        // add a canary that pings our home page and alarms if it returns errors.
        props.monitoring.addWebCanary('Home Page', `https://${this.distribution.domainName}`);
    }
}
exports.WebApp = WebApp;
//# sourceMappingURL=data:application/json;base64,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