"""
Created on 18-February-2022 12:22
Code by Daniel Maldonado
This library allows the conversion of MedPC plain text files to .xlsx spreadsheets, and the extraction of relevant
measures with a simple syntax.
"""

from openpyxl import Workbook, load_workbook
from openpyxl.utils import get_column_letter
from os import listdir
from statistics import mean, median
import pandas
import re
from shutil import move
from copy import deepcopy


def to_list(item):
    if type(item) is list:
        return item
    else:
        return [item]


def fetch(sheet, origin_cell_row, origin_cell_column):
    """
    This function returns the value contained in a cell within the specified spreadsheet\n
    :param sheet: The chosen spreadsheet to extract the value from.
    :param origin_cell_column: The column in which the cell is found.
    :param origin_cell_row: The row in which the cell is found.
    :return: Value of the specified cell.
    """
    return sheet.cell(origin_cell_row, origin_cell_column).value


def count_resp(marks, trialStart, trialEnd, response):  # Count_per_trial
    """
    Counts responses between the beginning of the trial and the end of the trial. If a single mark identifies both the response to count and the
    response that marks the beginning of the trial, then there will be an extra response counted. This can be corrected with the
    "subtract" option in the analysis list.\n
    :param marks: List of marks.
    :param trialStart: Mark for start of trial or list of marks for different starts.
    :param trialEnd: Mark for end of trial or list of marks for different ends.
    :param response: Mark for response to count or list of marks for different responses.
    :return: List with the amount of responses per trial.
    """
    contadorTemp = 0
    inicio = 0
    resp = []
    for n in range(len(marks)):
        if marks[n].value in to_list(trialStart):
            inicio = 1
        elif marks[n].value in to_list(response) and inicio == 1:
            contadorTemp += 1
        elif marks[n].value in to_list(trialEnd) and inicio == 1:
            inicio = 0
            resp.append(contadorTemp)
            contadorTemp = 0
    if len(resp) == 0:
        resp = [0]
    return resp


def resp_dist(marks, time, trialStart, trialEnd, response, bin_size, bin_amount, unit):
    """
    Counts responses per time-bin in each trial of the session. The size and amount of bins can be specified. This
    function can deal with situations in which there is a trial-end mark (there is an inter-trial interval) as well
    as situations in which there is not. If there are responses beyond the last declared bin, then all the responses
    that occurred between the end of the last bin and the end of the trial will be aggregated in a single extra bin.\n
    :param marks: List of marks.
    :param time: List with session time.
    :param trialStart: Mark for start of trial or list of marks for different starts.
    :param trialEnd: Mark for end of trial or list of marks for different ends.
    :param response: Mark of the response of interest, or list of marks for different responses.
    :param bin_size: Bin size in seconds.
    :param bin_amount: Amount of bins per trial.
    :return: List made of sub-lists which contain all responses per-bin, per-trial.
    :param unit: Temporal resolution. The quantity by which the seconds are divided in your MedPc setup.
    """
    inicio = 0  # Supplementary variable. Keeps track of whether the trial has started or not.
    resp_por_ensayo = [0] * (bin_amount + 1)  # Generates a list with as many zeros as the parameter bin_amount dictates
    resp_totales = []  # List which will contain sublist with the responses per-bin.
    bin_tuples = []  # List of time pairs: bin-start and bin-stop.
    if trialStart == trialEnd:  # This route is taken if there is a single mark for both the start and the end of the trial.
        for index, mark in enumerate(marks):
            if mark.value in to_list(trialStart) and inicio == 0:
                tiempo_inicio = time[index].value
                # This loop creates a list with as many tuples as bin_amount dictates. Each tuple contains the start and
                # end time for each bin. The end of a bin corresponds with the start of the next bin.
                for i in range(bin_amount):
                    tiempo_fin = tiempo_inicio + (bin_size * unit)  # The end-time for each bin is generated by adding
                    # the value of bin_size multiplied by unit to the previous bin (or the trial-start for the first).
                    bin_tuples.append((tiempo_inicio, tiempo_fin))
                    tiempo_inicio = tiempo_fin
                inicio = 1

            elif mark.value in to_list(trialStart) and inicio == 1:
                # When the beginning of the next trial is found, the responses accumulated so far are added to the list
                # of total responses.
                resp_totales.append(resp_por_ensayo)
                resp_por_ensayo = [0] * (bin_amount + 1)  # resp_por_ensayo and bin_tuples are emptied again.
                bin_tuples = []
                tiempo_inicio = time[index].value
                # Maybe these 4 lines are redundant
                for i in range(bin_amount):
                    tiempo_fin = tiempo_inicio + (bin_size * unit)
                    bin_tuples.append((tiempo_inicio, tiempo_fin))
                    tiempo_inicio = tiempo_fin

            elif mark.value in to_list(response) and inicio == 1:
                # Check if the response or responses of interest are found at the current mark position
                for idx, bin_tuple in enumerate(bin_tuples):
                    if bin_tuple[0] <= time[index].value < bin_tuple[1]:
                        # If a response is found, the program cycles through the list of tuples. If the response time
                        # matches the time contained in a tuple interval, a unit is added to the resp_por_ensayo list in
                        # the corresponding position.
                        resp_por_ensayo[idx] += 1
                if time[index].value >= bin_tuples[-1][-1]:
                    # If the response time is beyond that dictated by the tuples, then a unit is added to the last
                    # position of resp_por_ensayo. This last position has an aggregate of all responses occured after
                    # the last bin but before the next trial.
                    resp_por_ensayo[-1] += 1
        resp_totales.append(resp_por_ensayo)

    else:  # This route is taken if there are separate marks for both trial-start and trial-end.
        for index, mark in enumerate(marks):
            if mark.value in to_list(trialStart) and inicio == 0:
                tiempo_inicio = time[index].value
                # This loop creates a list with as many tuples as bin_amount dictates. Each tuple contains the start and
                # end time for each bin. The end of a bin corresponds with the start of the next bin.
                for i in range(bin_amount):
                    tiempo_fin = tiempo_inicio + (bin_size * unit)
                    bin_tuples.append((tiempo_inicio, tiempo_fin))
                    tiempo_inicio = tiempo_fin
                inicio = 1

            elif mark.value in to_list(response) and inicio == 1:
                for idx, bin_tuple in enumerate(bin_tuples):
                    if bin_tuple[0] <= time[index].value < bin_tuple[1]:
                        # If a response is found the program cycles through the list of tuples. If the response time
                        # matches the time contained in a tuple interval, a unit is added to the resp_por_ensayo list in
                        # the corresponding position.
                        resp_por_ensayo[idx] += 1
                if time[index].value >= bin_tuples[-1][-1]:
                    # If the response time is beyond that dictated by the tuples, then a unit is added to the last
                    # position of resp_por_ensayo. This last position has an aggregate of all responses occured after
                    # the last bin but before the next trial.
                    resp_por_ensayo[-1] += 1

            elif mark.value in to_list(trialEnd):
                # After each trial the list with all responses is appended to a higher-order list called resp_totales.
                # The values of resp_por_ensayo and bin_tuples are reset.
                inicio = 0
                resp_totales.append(resp_por_ensayo)
                resp_por_ensayo = [0] * (bin_amount + 1)
                bin_tuples = []

    return resp_totales


def total_count(marks, response):
    """
    Counts all the occurences of a particular mark during the entire sesion regardless of trials.\n
    :param marks: List of marks.
    :param response: Mark for response of interest or list of marks for different responses.
    :return: Int with the total occurences of the response of interest.
    """
    contador = 0
    for n in range(len(marks)):
        if marks[n].value in to_list(response):
            contador += 1
    return contador


def lat_count(marks, time, trialStart, response, unit):
    """
    Computes the latency between the start of a trial and the first response of interest.\n
    :param marks: List of marks.
    :param time: List with session time.
    :param trialStart: Mark for start of trial or list of marks for different starts.
    :param response: Mark for response of interest or list of marks for different responses.
    :param unit: Temporal resolution. The quantity by which seconds are divided in your MedPc setup.
    :return: List with latencies per trial.
    """
    inicio = 0
    lat = []
    tiempoini = 0
    for n in range(len(marks)):
        if marks[n].value in to_list(trialStart):
            inicio = 1
            tiempoini = time[n].value
        elif marks[n].value in to_list(response) and inicio == 1 and time[n].value != tiempoini:
            lat.append((time[n].value - tiempoini) / unit)
            inicio = 0
    if len(lat) == 0:
        lat = [0]
    return lat


def get_col(indivSheet, column):
    """
    Gets an entire column from the individual xlsx files and returns a list with its contents.
    :param indivSheet: The sheet to extract the column from
    :param column: The column to retrieve
    :return: A list with the column contents
    """
    column_list = []
    for cell in indivSheet[column]:
        if cell.value is not None:
            column_list.append(cell.value)
    return column_list


def write_cols(indivSheet, header, column, data):
    """
    Writes entire lists in spreadsheet columns. Useful for writing whole datasets on individual .xlsx files.\n
    :param indivSheet: Worksheet type (Openpyxl) object in which the list will be written.
    :param header: Header which will be written on the first cell of the column.
    :param column: Number corresponding to the column in which the list will be written (1=A, 2=B, etc.).
    :param data: List which will be written on the column.
    """
    indivSheet[get_column_letter(column) + str(1)] = header
    for pos in range(len(data)):
        indivSheet[get_column_letter(column) + str(pos + 2)] = data[pos]


def template():
    """
    Prints a template to facilitate declaration of the analysis list.
    """
    print("""
    analysis_list = [
    {"fetch": {"cell_row": 10,
               "cell_column": 10,
               "sheet": "Sheet_1",
               "summary_distribution": column_dictionary,
               "offset": 0,
               "write_rows": False,  # Optional
               }},

    {"count_resp": {"measures": 2, # Optional argument. Default value: 1
                    "trial_start": 111, "trial_end": 222, "response": 333,
                    "trial_start2": 444, "trial_end2": 555, "response2": 666, # Optional marks. Depends on the value of "measures"
                    "column": 1,
                    "header": "Generic_title",
                    "sheet": "Sheet_2",
                    "summary_distribution": column_dictionary2,
                    "subtract": True, # Optional argument. Default value: False
                    "statistic": "mean",  # Alternative value: "median"
                    "offset": 0,
                    "write_rows": False,  # Optional
                    }},

    {"total_count": {"measures": 2, # Optional argument. Default value: 1
                     "response": 111,
                     "response2": 222, # Optional mark. Depends on the value of "measures"
                     "column": 3,
                     "header": "Generic_title",
                     "sheet": "Sheet_4",
                     "summary_distribution": column_dictionary4,
                     "offset": 0,
                     "write_rows": False,  # Optional
                     }},

    {"lat_count": {"measures": 2, # Optional argument. Default value: 1
                   "trial_start": 111, "response": 222,
                   "trial_start2": 333, "response2": 444, # Optional marks. Depends on the value of "measures"
                   "column": 2,
                   "header": "Generic_title",
                   "sheet": "Sheet_3",
                   "summary_distribution": column_dictionary3,
                   "statistic": "mean",  # Alternative value: "median"
                   "offset": 0,
                   "unit": 20,  # Optional
                   "write_rows": False,  # Optional
                   }},

    {"resp_dist": {"trial_start": 111, "trial_end": 222, "response": 333,
                   "bin_size": 1,
                   "bin_amount": 15,
                   "label": "Generic_label",
                   "statistic": "median",  # Alternative value: "mean"
                   "unit": 20,  # Optional
                   }},
                   
    {"get_cols": {
        "source": "A",
        "column": 1,
        "header": "Generic_header",
    }},
    ]
    """)


def create_sheets(workbook, sheets):
    """
    Creates a dictionary which relates sheet names (given by the sheets argument) to Worksheet objects in the summary
    file. If Worksheets with the given name already exist in the summary file they are opened and added to the
    dictionary.\n
    :param sheets: List of strings corresponding to the names that each Worksheet will have.
    :param workbook: Workbook type object in which Worksheets will be created or opened.
    :return: Dictionary relating names to Worksheets.
    """
    sheet_dict = {}
    for sheet in sheets:
        if sheet not in workbook.sheetnames:
            new_sheet = workbook.create_sheet(sheet)
        else:
            new_sheet = workbook[sheet]
        sheet_dict[sheet] = new_sheet
    return sheet_dict


class Analyzer:
    """
    Object in charge of the main analysis. Converts MedPC files to .xlsx and extracts relevant data according to the
    analysisList parameter.
    """

    def __init__(self, fileName: str, temporaryDirectory: str, permanentDirectory: str, convertedDirectory: str,
                 subjectList: list, suffix: str, sheets: list, analysisList: list, timeColumn: str = None,
                 markColumn: str = None, relocate: bool = True, colDivision: int = 6, spacing: int = 2):
        """
        :param fileName: The name of the summary file.
        :param temporaryDirectory: The directory in which raw MedPC files are stored before the analysis.
        :param permanentDirectory: The directory to which raw MedPC files will be moved after analysis.
        :param convertedDirectory: The directory in which individual .xlsx files and the summary file will be stored after the analysis.
        :param subjectList: A list of strings with the names of all subjects.
        :param suffix: The character or characters which separate the subject name from the session number in the raw MedPC filenames.
        :param sheets: A list of strings which represent the names of each individual sheet which will be created in the summary file.
        :param analysisList: A list of dictionaries which declares the details of every relevant measure to extract.
        :param timeColumn: The column in which the time is written in the individual .xlsx files.
        :param markColumn: The column in which the marks are written in the individual .xlsx files.
        :param relocate: A boolean which indicates if the raw MedPC files should be moved from the temporary directory to the permanent one after the analysis.
        :param colDivision: The number of columns to divide the raw MedPC files into.
        :param spacing: The amount of whitespace left either at the top or the left of the summary file before the data.
        """
        self.file_name = fileName
        self.temp_directory = temporaryDirectory
        self.perm_directory = permanentDirectory
        self.conv_directory = convertedDirectory
        self.col_division = colDivision
        self.subject_list = deepcopy(subjectList)
        self.session_list = []
        self.suffix = suffix
        self.relocate = relocate
        self.sheets = sheets
        self.analysis_list = analysisList
        self.mark_column = markColumn
        self.time_column = timeColumn
        self.spacing = spacing
        self.file_format = None

    def get_sessions(self):
        """
        Generates a list of subjects with at least one associated MedPC file in the temporary directory, and a list of
        lists of sessions associated with each subject currently found in the directory.\n
        """
        sujetosFaltantes = []
        dirTemp = sorted(listdir(self.temp_directory))
        listaTemp = []
        # Subjects with at least one associated file in the temporary directory are added to a temporary list.
        for sbj in self.subject_list:
            for datoTemp in dirTemp:
                if sbj == datoTemp.split(self.suffix)[0] and sbj not in listaTemp:
                    listaTemp.append(sbj)

        # Subjects which are not in the temporary list are added to the sujetosFaltantes list to be printed later.
        for sbj in self.subject_list:
            if sbj not in listaTemp:
                sujetosFaltantes.append(sbj)
        # The name of subjects with no associated files is printed.
        if len(sujetosFaltantes) == 0:
            print("All subjects have at least one session yet to analyze.")
        else:
            print(f"Missing subjects: {str(sujetosFaltantes)}")

        # The name of each subject is compared against the file names of the temporary directory. If a coincidence is
        # found, the session number is added to a sublist which will then be added to the super-list.
        indice = 0
        for sujetoPresente in listaTemp:
            self.session_list.append([])
            sublista = []
            for datoTemp in dirTemp:
                if sujetoPresente == datoTemp.split(self.suffix)[0]:
                    sublista.append(int(datoTemp.split(self.suffix)[-1].split(".")[0]))
                # The file format is updated to that of the files in the temporary directory. If the files have no
                # format then it stays as None
                self.file_format = datoTemp.split(".")[-1] if "." in datoTemp else None
            self.session_list[indice] = sorted(sublista)
            indice += 1

        for i in range(len(listaTemp)):
            print(f"Sessions to analyze for subject {listaTemp[i]}: {self.session_list[i]}")

        # The missing subjects are deleted from the original subject list.
        for sujetoFaltante in sujetosFaltantes:
            del self.subject_list[self.subject_list.index(sujetoFaltante)]
        print("-" * 70)

    def converter(self):
        """
        Converts plain text MedPC files into .xlsx files. Lists are separated into column pairs based on the decimal
        point.\n
        """
        for ind, sjt in enumerate(self.subject_list):
            for ssn in self.session_list[ind]:
                print(f"Converting session {ssn} of subject {sjt}.")
                # Pandas reads the file and separates it into six columns based on white space.
                datos = pandas.read_csv(
                    f"{self.temp_directory}{sjt}{self.suffix}{ssn}{'.' + self.file_format if self.file_format else ''}",
                    header=None,
                    names=range(self.col_division), sep=r'\s+')
                datos.to_excel(f"{self.conv_directory}{sjt}{self.suffix}{ssn}.xlsx", index=False, header=None)

                # The file created by pandas is read by Openpyxl and saved on a variable named hojaCompleta.
                archivoCompleto = load_workbook(f"{self.conv_directory}{sjt}{self.suffix}{ssn}.xlsx")
                hojaCompleta = archivoCompleto.active

                # A list of lists which will contain the data of all arrays is created.
                metalista = [[]]
                contadormetalista = 0

                columna1 = hojaCompleta['B']
                rx = re.compile(r'^\d+\.\d+$')
                # This section will be useful if there is a case in which the arrays begin on a row different from the 11th or
                # 12th. It is a bit slower, so it wont be implemented unless necessary.
                # place = 1
                # while True:
                #     if rx.search(hojaCompleta[f"B{place}"].value):
                #         start = place
                #         break
                #     elif place >= 12:
                #         start = 12
                #         break
                #     else:
                #         place += 1
                # If the 11th row contains data, then it is taken as the starting point. Else, the 12th row is taken.
                start = 12
                if hojaCompleta["B11"].value is not None:
                    if rx.search(hojaCompleta["B11"].value):
                        start = 11
                # The list of lists is populated with the content of all arrays. Each sub-list represents an array.
                # Values are converted to floats so that all of them have a decimal point and can be treated equally.
                for fila in range(start, len(columna1) + 1):
                    for columna in range(2, 6 + 1):
                        if hojaCompleta[f"{get_column_letter(columna)}{fila}"].value is not None:
                            metalista[contadormetalista].append(str(float(hojaCompleta[get_column_letter(columna) +
                                                                                       str(fila)].value)))
                        elif hojaCompleta[get_column_letter(columna) + str(fila)].value is None and columna == 2:
                            metalista.append([])
                            contadormetalista += 1

                # Since Pandas strips numbers of trailing zeros they must be added again manually. This is done with
                # regex. All values are scanned for coincidence with either one or two missing zeros, and if a match is
                # found they are added back.
                regex1 = re.compile(r'^\d+\.\d{2}$')
                regex2 = re.compile(r'^\d+\.\d$')
                for ii in range(len(metalista)):
                    for j in range(len(metalista[ii])):
                        if regex1.search(metalista[ii][j]):
                            metalista[ii][j] += '0'
                        elif regex2.search(metalista[ii][j]):
                            metalista[ii][j] += '00'
                        # Each list is separated in two columns based on the decimal point.
                        hojaCompleta[get_column_letter((ii * 2) + (self.col_division + 3)) + str(j + 1)] = int(
                            metalista[ii][j].split('.')[0])
                        hojaCompleta[get_column_letter((ii * 2) + (self.col_division + 4)) + str(j + 1)] = int(
                            metalista[ii][j].split('.')[1])
                # Finally the individual .xlsx file is saved and the raw file is moved to the permanent directory.
                archivoCompleto.save(f"{self.conv_directory}{sjt}{self.suffix}{ssn}.xlsx")
                if self.relocate:
                    move(
                        f"{self.temp_directory}{sjt}{self.suffix}{ssn}{'.' + self.file_format if self.file_format else ''}",
                        f"{self.perm_directory}{sjt}{self.suffix}{ssn}{'.' + self.file_format if self.file_format else ''}")
            print("")
        print("-" * 70)

    def create_document(self):
        """
        The converted directory is inspected in search of a summary file. If one is found, it is opened; else, it is
        created. This function must be asigned to a variable.\n
        :return: Workbook class object (Openpyxl).
        """
        if self.file_name in listdir(self.conv_directory):
            print('\nSummary file found. Opening...\n')
            wb = load_workbook(self.conv_directory + self.file_name)
        else:
            print('\nSummary file not found. Creating...\n')
            wb = Workbook()
        return wb

    def analyze(self, workbook, sheetDict):
        """
        Main analysis function. Uses all other functions and methods to analyze the variables of interest and
        write them on the individual .xlsx files. It then calculates measures of central tendency and writes
        them in the summary file.\n
        :param workbook: Object created by the create_document function.
        :param sheetDict: Dictionary created by the create_sheets function.
        """
        for index, subject in enumerate(self.subject_list):
            for session in self.session_list[index]:
                print(f"Trying session {session} of subject {subject}...", end="")
                sujetoWb = load_workbook(f"{self.conv_directory}{subject}{self.suffix}{str(session)}.xlsx")
                sujetoWs = sujetoWb.worksheets[0]
                hojaind = sujetoWb.create_sheet('FullLists')
                tiempo = sujetoWs[self.time_column]
                marcadores = sujetoWs[self.mark_column]

                # Each element in the analysis list is inspected and compared with specific keywords which determine
                # the treatment that the data will get.
                for analysis in self.analysis_list:
                    key, value = list(analysis.items())[0]
                    if key == "count_resp":
                        respuestas_totales = []
                        # If the argument "measures" is greater than 1 then several measures are aggregated.
                        for mark_index in range(1, value.get("measures", 1) + 1):
                            if mark_index == 1:
                                respuesta_parcial = count_resp(marcadores, value["trial_start"],
                                                               value["trial_end"],
                                                               value["response"])
                            else:
                                respuesta_parcial = count_resp(marcadores, value[f"trial_start{mark_index}"],
                                                               value[f"trial_end{mark_index}"],
                                                               value[f"response{mark_index}"])
                            respuestas_totales.extend(respuesta_parcial)
                        if value.get("statistic", "mean") == "mean":  # Decide whether mean or median will be written.
                            if value.get("subtract",
                                         False):  # Decide if a response will be subtracted from each trial.
                                respuestas_restadas = [resp - 1 if resp > 0 else resp for resp in respuestas_totales]
                                # Decide whether to write on rows or columns
                                if not value.get("write_rows", False):
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(
                                        value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                        session + self.spacing)] = mean(respuestas_restadas)
                                else:
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                        value["summary_distribution"][subject] + value.get("offset", 0))] = mean(
                                        respuestas_restadas)
                                write_cols(hojaind, value["header"], value["column"], respuestas_restadas)
                            else:
                                # Decide whether to write on rows or columns
                                if not value.get("write_rows", False):
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(
                                        value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                        session + self.spacing)] = mean(respuestas_totales)
                                else:
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                        value["summary_distribution"][subject] + value.get("offset", 0))] = mean(
                                        respuestas_totales)
                                # The entire list is written on the individual .xlsx file.
                                write_cols(hojaind, value["header"], value["column"], respuestas_totales)
                        else:
                            if value.get("subtract", False):
                                respuestas_restadas = [resp - 1 if resp > 0 else resp for resp in respuestas_totales]
                                # Decide whether to write on rows or columns
                                if not value.get("write_rows", False):
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(
                                        value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                        session + self.spacing)] = median(respuestas_restadas)
                                else:
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                        value["summary_distribution"][subject] + value.get("offset", 0))] = median(
                                        respuestas_restadas)
                                write_cols(hojaind, value["header"], value["column"], respuestas_restadas)
                            else:
                                # Decide whether to write on rows or columns
                                if not value.get("write_rows", False):
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(
                                        value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                        session + self.spacing)] = median(respuestas_totales)
                                else:
                                    # The measure of central tendency is written on the summary file.
                                    sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                        value["summary_distribution"][subject] + value.get("offset", 0))] = median(
                                        respuestas_totales)
                                # The entire list is written on the individual .xlsx file.
                                write_cols(hojaind, value["header"], value["column"], respuestas_totales)

                    elif key == "lat_count":
                        # If the argument "measures" is greater than 1 then several measures are aggregated.
                        latencias_totales = []
                        for mark_index in range(1, value.get("measures", 1) + 1):
                            if mark_index == 1:
                                latencia_parcial = lat_count(marcadores, tiempo, value["trial_start"],
                                                             value["response"], unit=value.get("unit", 1))
                            else:
                                latencia_parcial = lat_count(marcadores, tiempo, value[f"trial_start{mark_index}"],
                                                             value[f"response{mark_index}"], unit=value.get("unit", 1))
                            latencias_totales.extend(latencia_parcial)
                        if value.get("statistic", "mean") == "mean":  # Decide whether mean or median will be written.
                            # Decide whether to write on rows or columns
                            if not value.get("write_rows", False):
                                # The measure of central tendency is written on the summary file.
                                sheetDict[value["sheet"]][get_column_letter(
                                    value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                    session + self.spacing)] = mean(latencias_totales)
                            else:
                                # The measure of central tendency is written on the summary file.
                                sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                    value["summary_distribution"][subject] + value.get("offset", 0))] = mean(
                                    latencias_totales)
                        else:
                            # Decide whether to write on rows or columns
                            if not value.get("write_rows", False):
                                # The measure of central tendency is written on the summary file.
                                sheetDict[value["sheet"]][get_column_letter(
                                    value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                    session + self.spacing)] = median(latencias_totales)
                            else:
                                # The measure of central tendency is written on the summary file.
                                sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                    value["summary_distribution"][subject] + value.get("offset", 0))] = median(
                                    latencias_totales)
                        # The entire list is written on the individual .xlsx file.
                        write_cols(hojaind, value["header"], value["column"], latencias_totales)

                    elif key == "total_count":
                        respuestas_totales = 0
                        for mark_index in range(1, value.get("measures", 1) + 1):
                            if mark_index == 1:
                                respuesta_parcial = total_count(marcadores, value["response"])
                            else:
                                respuesta_parcial = total_count(marcadores, value[f"response{mark_index}"])
                            respuestas_totales += respuesta_parcial
                        # Decide whether to write on rows or columns
                        if not value.get("write_rows", False):
                            # The measure is written on the summary file
                            sheetDict[value["sheet"]][
                                get_column_letter(
                                    value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                    session + self.spacing)] = respuestas_totales
                        else:
                            # The measure is written on the summary file
                            sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                value["summary_distribution"][subject] + value.get("offset", 0))] = respuestas_totales

                        # The measure is written on the individual file
                        write_cols(hojaind, value["header"], value["column"], [respuestas_totales])

                    elif key == "fetch":
                        cell_value = fetch(sujetoWs, value["cell_row"], value["cell_column"])
                        # Decide whether to write on rows or columns
                        if not value.get("write_rows", False):
                            # The single value is written on the summary file.
                            sheetDict[value["sheet"]][
                                get_column_letter(
                                    value["summary_distribution"][subject] + value.get("offset", 0)) + str(
                                    session + self.spacing)] = cell_value
                        else:
                            # The measure is written on the summary file
                            sheetDict[value["sheet"]][get_column_letter(session + self.spacing) + str(
                                value["summary_distribution"][subject] + value.get("offset", 0))] = cell_value

                    elif key == "resp_dist":
                        if "label" in value:  # The label argument determines whether the resulting sheet will have a
                            # specific or a generic name.
                            resp_dist_sheet = create_sheets(workbook, [f"{subject}_{value['label']}"])
                            dist_indiv_sheet = sujetoWb.create_sheet(value["label"])
                        else:
                            resp_dist_sheet = create_sheets(workbook, [subject])
                            dist_indiv_sheet = sujetoWb.create_sheet('RespDistrib')
                        # A superlist is populated with lists of response distributions per trial.
                        superlist = resp_dist(marcadores, tiempo, trialStart=value["trial_start"],
                                              trialEnd=value["trial_end"],
                                              response=value["response"], bin_size=value["bin_size"],
                                              bin_amount=value["bin_amount"],
                                              unit=value.get("unit", 1))
                        # Either the mean or the median of each bin (not each trial) is calculated.
                        aggregated = []
                        means = []
                        for i in range(len(superlist[0])):
                            for sublist in superlist:
                                aggregated.append(sublist[i])
                            if value.get("statistic", "mean") == "mean":
                                means.append(mean(aggregated))
                            elif value.get("statistic", "median") == "median":
                                means.append(median(aggregated))
                            else:
                                means.append(sum(aggregated))
                            aggregated = []

                        # The central tendency measures are written on their corresponding sheet in the summary file.
                        if "label" in value:
                            write_cols(resp_dist_sheet[f'{subject}_{value["label"]}'], f"Session {session}",
                                       session + 1, means)
                        else:
                            write_cols(resp_dist_sheet[subject], f"Session {session}", session + 1, means)
                        # The entire array of response distributions is written on a separate sheet in the individual
                        # .xlsx file.
                        for ix, sublist in enumerate(superlist):
                            write_cols(dist_indiv_sheet, f"Trial {ix + 1}", ix + 1, sublist)

                    elif key == "get_cols":
                        # Create a sheet for each subject in the summary file
                        col_sheet = create_sheets(workbook, [f"{subject}_{value['header']}"])
                        # Get an entire column from the individual files
                        entire_col = get_col(sujetoWs, value["source"])
                        # Write column list to wanted column in individual file
                        write_cols(hojaind, value["header"], value["column"], entire_col)
                        # Write column list to wanted column in summary file
                        write_cols(col_sheet[f"{subject}_{value['header']}"], f"Session{session}", session + 1,
                                   entire_col)

                # The individual .xlsx file is saved.
                sujetoWb.save(self.conv_directory + subject + self.suffix + str(session) + '.xlsx')
                print(" Success.\n")
            if len(self.session_list[index]) >= 2:
                print("-" * 70)
        # The summary file is saved.
        workbook.save(self.conv_directory + self.file_name)

    def convert(self):
        self.get_sessions()
        self.converter()

    def complete_analysis(self):
        self.get_sessions()
        self.converter()
        workbook = self.create_document()
        sheets = create_sheets(workbook, self.sheets)
        self.analyze(workbook, sheets)
