#!/usr/bin/python
# -*- coding: utf-8 -*-
# created: '2015-01-29'


"""错误码和错误描述定义生成器"""
import optparse
import os
import sys

Error_Defines = {}
Domain_Base = 0


def generate_error_defines_for_c(filename, prefix="Gcommon"):
    with open(filename, "w") as f:
        f.write("/* This is a file header */\n\n")

        f.write("static const uint32_t k%sErrorCodePageBoundary = 999999;\n\n" % prefix)

        f.write("static const %sErrorEntry errorCodePage[] = {\n" % prefix)

        d = ""
        for domain, code, description, _ in Error_Defines:
            assert code < Domain_Base
            if domain != d:
                d = domain
                f.write("\n")
            f.write(
                '\t{{{:d}, "{domain}", "{desc}"}},\n'.format(
                    domain.value * Domain_Base + code,
                    domain=domain.name,
                    desc=description,
                )
            )

        f.write("};\n\n// end of file\n")


def generate_error_defines_for_thrift(filename, prefix="Gcommon"):
    with open(filename, "w") as f:
        import datetime

        f.write("/* error_define.thrift\n")
        f.write(" * This is auto generated error codes (by gcommon)\n")
        f.write(" * Generated on '{createDate}'\n".format(createDate=datetime.date.today()))
        f.write(" */\n\n")

        f.write("namespace cpp error_define\n")
        f.write("namespace py %sproto.error_define\n\n" % prefix.lower())

        f.write("enum %s_errors\n" % prefix.lower())
        f.write("{\n")

        d = ""
        for domain, code, description, _ in Error_Defines:
            if domain != d:
                d = domain
                f.write("\n")
            f.write("\t{desc} = {err_code},\n".format(desc=description, err_code=domain.value * Domain_Base + code))
        f.write("}\n\n/* End of File */\n")


def generate_error_defines_for_python(filename):
    with open(filename, mode="w", encoding="utf8") as f:
        # Write header
        import datetime

        f.write(
            (
                "#!/usr/bin/python\n"
                "# -*- coding: utf-8 -*-\n"
                "# created: '{createDate}'\n"
                "# author: 'gerror_generator.py'\n\n"
            ).format(createDate=datetime.date.today())
        )

        # write comments
        f.write('"""\n')
        f.write("   Error definition and description.\n")
        f.write("   This file is automatically generated by gerror_generator.py\n")
        f.write("   NOTE: DO NOT change this file manually!!!\n")
        f.write('         Go for gerror_generator_source.py instead."""\n\n')
        f.write("from gcommon.error.gerror import GError\n\n")

        ##############################################
        # Write error dict
        ##############################################
        f.write("error_defines = {")

        # write definition
        d = ""
        for domain, code, name, message in Error_Defines:
            assert code < Domain_Base
            if domain != d:
                d = domain
                f.write("\n")

            err_code = domain.value * Domain_Base + code

            # pat = "    {:0>5d}: ('{name}', u'{msg}'),\n"
            pat = "    {:d}: ('{name}', u'{msg}'),\n"
            f.write(pat.format(err_code, name=name, msg=message))

        # write definition end
        f.write("}\n\n\n")

        ##############################################
        # Description to Error code
        ##############################################
        f.write("class GErrorCodes(object):")

        # write dict
        d = ""
        for domain, code, name, message in Error_Defines:
            if domain != d:
                d = domain
                f.write("\n")

            err_code = domain.value * Domain_Base + code

            pat = "    {name} = {code}\n"
            f.write(pat.format(name=name, code=err_code))

        # write definition end
        f.write("    pass\n\n\n")

        ##############################################
        # Write error class
        ##############################################
        f.write("""class GErrors(object):""")

        # write definition
        d = ""
        for domain, code, name, message in Error_Defines:
            if domain != d:
                d = domain
                f.write("\n")

            err_code = domain.value * Domain_Base + code
            pat = "    %s = GError.create(%d, error_defines)\n"
            f.write(pat % (name, err_code))

        # write definition end
        f.write("\n    pass")

        ##############################################
        # write file end
        ##############################################
        f.write("\n\n")


def generate_error_defines_for_js(filename):
    pass


def main(all_args, prefix="Gcommon"):
    # set usage
    usage_text = """Start to generate error_definition file.
    %(error)s [-f <file_name>]"""
    parser = optparse.OptionParser()

    usage_param = {"error": all_args[0]}
    parser.set_usage(usage_text % usage_param)

    parser.add_option("-f", "--filename", dest="filename", action="store", default="default")
    parser.add_option("-p", "--prefix", dest="prefix", action="store", default="Gcommon")

    all_args = all_args[1:]
    options, args = parser.parse_args(all_args)

    filename, extension = os.path.splitext(options.filename)
    if extension:
        extension = extension[1:]
        if extension == "py":
            generate_error_defines_for_python(options.filename)
        elif extension == "h":
            generate_error_defines_for_c(options.filename)
        elif extension == "js":
            generate_error_defines_for_js(options.filename)
        elif extension == "thrift":
            generate_error_defines_for_thrift(options.filename)
        else:
            parser.error("File type is not suppported yet.")
    elif filename == "default":
        generate_error_defines_for_python("gerror_define.py")
        # generate_error_defines_for_thrift('%s_error_define.thrift' % options.prefix.lower())


def generate_gcommon_error_code():
    global Error_Defines, Domain_Base
    from gcommon.error.gerror_generator_source import domain_base, error_defines

    Error_Defines = error_defines
    Domain_Base = domain_base

    main(sys.argv)


def generate_user_defined_error_code(error_defines, domain_base, prefix="Gcommon"):
    global Error_Defines, Domain_Base
    Error_Defines = error_defines
    Domain_Base = domain_base

    main(sys.argv, prefix=prefix)


# Test Codes
if __name__ == "__main__":
    if len(sys.argv) == 1:
        generate_gcommon_error_code()
    else:
        main(sys.argv)
        print("Done")
