# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ptera']

package_data = \
{'': ['*']}

install_requires = \
['giving>=0.3.6,<0.4.0']

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['contextvars>=2.4,<3.0'],
 ':python_version >= "3.8" and python_version < "4.0"': ['codefind>=0.1.2,<0.2.0']}

setup_kwargs = {
    'name': 'ptera',
    'version': '1.0.1',
    'description': 'Call graph addressing library.',
    'long_description': '\n# Ptera\n\nPtera is a powerful way to instrument your code for logging, debugging and testing purposes. With a simple call to `ptera.probing()`, you can:\n\n* [Obtain a stream of the values taken by any variable.](https://ptera.readthedocs.io/en/latest/guide.html#probe-a-variable)\n* [Probe multiple variables from multiple functions in multiple scopes.](https://ptera.readthedocs.io/en/latest/guide.html#probe-multiple-variables)\n* [Apply maps, filters, reductions, and much more to the streams.](https://ptera.readthedocs.io/en/latest/guide.html#map-filter-reduce)\n* [Override the values of variables based on complex conditions.](https://ptera.readthedocs.io/en/latest/guide.html#overriding-values)\n* Create [external asserts](https://ptera.readthedocs.io/en/latest/guide.html#asserts) or [conditional breakpoints](https://ptera.readthedocs.io/en/latest/guide.html#conditional-breakpoints).\n* Et cetera :)\n\n📖 **[Read the documentation](https://ptera.readthedocs.io/en/latest)**\n\n## Install\n\n```bash\npip install ptera\n```\n\n## Example\n\nYou can use Ptera to observe assignments to any variable in your program:\n\n```python\nfrom ptera import probing\n\ndef f(x):\n    y = 10\n    for i in range(1, x + 1):\n        y = y + i\n    return y\n\nwith probing("f > y").values() as values:\n    f(3)\n\n# These are all the values taken by the y variable in f.\nassert values == [\n    {"y": 10},\n    {"y": 11},\n    {"y": 13},\n    {"y": 16},\n]\n```\n\nIn the above,\n\n1. We *select* the variable `y` of function `f` using the selector `f > y`.\n2. We use the `values()` method to obtain a list in which the values of `y` will be progressively accumulated.\n3. When `f` is called within the `probing` block, assignments to `y` are intercepted and appended to the list.\n4. When the `probing` block finishes, the instrumentation is removed and `f` reverts to its normal behavior.\n\n## Creating probes\n\n* [`ptera.probing`](https://ptera.readthedocs.io/en/latest/ref-probe.html#ptera.probe.probing): Probe variables inside a `with` block.\n* [`ptera.global_probe`](https://ptera.readthedocs.io/en/latest/ref-probe.html#ptera.probe.global_probe): Activate a global probe.\n\n## Using probes\n\nThe interface for Ptera\'s probes is inspired from functional reactive programming and is identical to the interface of [giving](https://github.com/breuleux/giving) (itself based on `rx`). [See here for a complete list of operators.](https://ptera.readthedocs.io/en/latest/ref-operators.html)\n\nYou can always use `with probing(...).values()` as in the example at the top if you want to keep it simple and just obtain a list of values. You can also use `with probing(...).display()` to print the values instead.\n\nBeyond that, you can also define complex data processing pipelines. For example:\n\n```python\nwith probing("f > x") as probe:\n    probe["x"].map(abs).max().print()\n    f(1234)\n```\n\nThe above defines a pipeline that extracts the value of `x`, applies the `abs` function on every element, takes the maximum of these absolute values, and then prints it out. Keep in mind that this statement doesn\'t really *do* anything at the moment it is executed, it only *declares* a pipeline that will be activated whenever a probed variable is set afterwards. That is why `f` is called after and not before.\n\n## More examples\n\nPtera is all about providing new ways to inspect what your programs are doing, so all examples will be based on this simple binary search function:\n\n```python\nfrom ptera import global_probe, probing\n\ndef f(arr, key):\n    lo = -1\n    hi = len(arr)\n    while lo < hi - 1:\n        mid = lo + (hi - lo) // 2\n        if (elem := arr[mid]) > key:\n            hi = mid\n        else:\n            lo = mid\n    return lo + 1\n\n##############################\n# THE PROBING CODE GOES HERE #\n##############################\n\nf(list(range(1, 350, 7)), 136)\n```\n\nTo get the output listed in the right column of the table below, the code in the left column should be inserted before the call to `f`, where the big comment is. Most of the methods on `global_probe` define the pipeline through which the probed values will be routed (the interface is inspired from functional reactive programming), so it is important to define them before the instrumented functions are called.\n\n<table>\n<tr>\n<th>Code</th>\n<th>Output</th>\n</tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nThe `display` method provides a simple way to log values.\n\n```python\nglobal_probe("f > mid").display()\n```\n\n</td>\n<td>\n\n```\nmid: 24\nmid: 11\nmid: 17\nmid: 20\nmid: 18\nmid: 19\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nThe `print` method lets you specify a format string.\n\n```python\nglobal_probe("f(mid) > elem").print("arr[{mid}] == {elem}")\n```\n\n</td>\n<td>\n\n```\narr[24] == 169\narr[11] == 78\narr[17] == 120\narr[20] == 141\narr[18] == 127\narr[19] == 134\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nReductions are easy: extract the key and use `min`, `max`, etc.\n\n```python\nglobal_probe("f > lo")["lo"].max().print("max(lo) = {}")\nglobal_probe("f > hi")["hi"].min().print("min(hi) = {}")\n```\n\n</td>\n<td>\n\n```\nmax(lo) = 19\nmin(hi) = 20\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nDefine assertions with `fail()` (for debugging, also try `.breakpoint()`!)\n\n```python\ndef unordered(xs):\n    return any(x > y for x, y in zip(xs[:-1], xs[1:]))\n\nprobe = global_probe("f > arr")["arr"]\nprobe.filter(unordered).fail("List is unordered: {}")\n\nf([1, 6, 30, 7], 18)\n```\n\n</td>\n<td>\n\n```\nTraceback (most recent call last):\n  ...\n  File "test.py", line 30, in <module>\n    f([1, 6, 30, 7], 18)\n  File "<string>", line 3, in f__ptera_redirect\n  File "test.py", line 3, in f\n    def f(arr, key):\ngiving.gvn.Failure: List is unordered: [1, 6, 30, 7]\n```\n\n</td>\n</tr>\n<tr></tr>\n\n<!--\n####################\n####### ROW ########\n####################\n-->\n\n<tr>\n<td>\n\nAccumulate into a list:\n\n```python\nresults = global_probe("f > mid")["mid"].accum()\nf(list(range(1, 350, 7)), 136)\nprint(results)\n```\n\nOR\n\n```python\nwith probing("f > mid")["mid"].values() as results:\n    f(list(range(1, 350, 7)), 136)\n\nprint(results)\n```\n\n</td>\n<td>\n\n```\n[24, 11, 17, 20, 18, 19]\n```\n\n</td>\n</tr>\n<tr></tr>\n\n</table>\n\n\n## probing\n\nUsage: `with ptera.probing(selector) as probe: ...`\n\nThe **selector** is a specification of which variables in which functions we want to stream through the probe. One of the variables must be the **focus** of the selector, meaning that the probe is triggered when *that* variable is set. The focus may be indicated either as `f(!x)` or `f > x` (the focus is `x` in both cases).\n\nThe **probe** is a wrapper around [rx.Observable](https://github.com/ReactiveX/RxPY) and supports a large number of [operators](https://giving.readthedocs.io/en/latest/ref-operators.html) such as `map`, `filter`, `min`, `average`, `throttle`, etc. (the interface is the same as in [giving](https://github.com/breuleux/giving)).\n\n\n### Example 1: intermediate variables\n\nPtera is capable of capturing any variable in a function, not just inputs and return values:\n\n```python\ndef fact(n):\n    curr = 1\n    for i in range(n):\n        curr = curr * (i + 1)\n    return curr\n\nwith probing("fact(i, !curr)").print():\n    fact(3)\n    # {\'curr\': 1}\n    # {\'curr\': 1, \'i\': 0}\n    # {\'curr\': 2, \'i\': 1}\n    # {\'curr\': 6, \'i\': 2}\n```\n\nThe "!" in the selector above means that the focus is `curr`. This means it is triggered when `curr` is set. This is why the first result does not have a value for `i`. You can use the selector `fact(!i, curr)` to focus on `i` instead:\n\n```python\nwith probing("fact(!i, curr)").print():\n    fact(3)\n    # {\'i\': 0, \'curr\': 1}\n    # {\'i\': 1, \'curr\': 1}\n    # {\'i\': 2, \'curr\': 2}\n```\n\nYou can see that the associations are different (curr is 2 when i is 2, whereas it was 6 with the other selector), but this is simply because they are now triggered when `i` is set.\n\n### Example 2: multiple scopes\n\nA selector may act on several nested scopes in a call graph. For example, the selector `f(x) > g(y) > h > z` would capture variables `x`, `y` and `z` from the scopes of three different functions, but only when `f` calls `g` and `g` calls `h` (either directly or indirectly).\n\n```python\ndef f(x):\n    return g(x + 1) * g(-x - 1)\n\ndef g(x):\n    return x * 2\n\n# Use "as" to rename a variable if there is a name conflict\nwith probing("f(x) > g > x as gx").print():\n    f(5)\n    # {\'gx\': 6, \'x\': 5}\n    # {\'gx\': -6, \'x\': 5}\n    g(10)\n    # Prints nothing\n```\n\n### Example 3: overriding variables\n\nIt is also possible to override the value of a variable with the `override` (or `koverride`) methods:\n\n\n```python\ndef add_ct(x):\n    ct = 1\n    return x + ct\n\nwith probing("add_ct(x) > ct") as probe:\n    # The value of other variables can be used to compute the new value of ct\n    probe.override(lambda data: data["x"])\n\n    # You can also use koverride, which calls func(**data)\n    # probe.koverride(lambda x: x)\n\n    print(add_ct(3))   # sets ct = x = 3; prints 6\n    print(add_ct(10))  # sets ct = x = 20; prints 20\n```\n\n**Important:** override() only overrides the **focus variable**. As explained earlier, the focus variable is the one to the right of `>`, or the one prefixed with `!`. A Ptera selector is only triggered when the focus variable is set, so realistically it is the only one that it makes sense to override.\n\nThis is worth keeping in mind, because otherwise it\'s not always obvious what override is doing. For example:\n\n```python\nwith probing("add_ct(x) > ct") as probe:\n    # The focus is ct, so override will always set ct\n    # Therefore, this sets ct = 10 when x == 3:\n    probe.where(x=3).override(10)\n\n    print(add_ct(3))   # sets ct = 10; prints 13\n    print(add_ct(10))  # does not override anything; prints 11\n```\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mila-iqia/ptera',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
