import abc
import os
import tarfile

from six import with_metaclass


class FileSystemError(Exception):
    pass


class FileSystem(with_metaclass(abc.ABCMeta, object)):
    """Interface for file systems."""

    @abc.abstractmethod
    def isfile(self, path):
        """Is this a file?"""
        pass

    @abc.abstractmethod
    def isdir(self, path):
        """Is this a directory?"""
        pass

    @abc.abstractmethod
    def read(self, path):
        """Read a file."""
        pass

    @abc.abstractmethod
    def refer_to(self, path):
        """Get a fully qualified path for the given path."""
        pass


class StoredFileSystem(FileSystem):
    """File system based on a file list."""

    def __init__(self, files):
        self.files = files
        self.dirs = {os.path.dirname(f) for f in files}

    def isfile(self, path):
        return path in self.files

    def isdir(self, path):
        return path in self.dirs

    def read(self, path):
        return self.files[path]

    def refer_to(self, path):
        return path


class OSFileSystem(FileSystem):
    """File system that uses an OS file system underneath."""

    def __init__(self, root):
        assert root is not None
        self.root = root

    def _join(self, path):
        return os.path.join(self.root, path)

    def isfile(self, path):
        assert path is not None
        return os.path.isfile(self._join(path))

    def isdir(self, path):
        assert path is not None
        return os.path.isdir(self._join(path))

    def read(self, path):
        with open(self._join(path), 'r') as fi:
            return fi.read()

    def refer_to(self, path):
        return self._join(path)


class RemappingFileSystem(with_metaclass(abc.ABCMeta, FileSystem)):
    """File system wrapper that transforms a path before looking it up."""

    def __init__(self, underlying):
        self.underlying = underlying

    @abc.abstractmethod
    def map_path(self, path):
        pass

    def isfile(self, path):
        return self.underlying.isfile(self.map_path(path))

    def isdir(self, path):
        return self.underlying.isdir(self.map_path(path))

    def read(self, path):
        return self.underlying.read(self.map_path(path))

    def refer_to(self, path):
        return self.underlying.refer_to(self.map_path(path))


class ExtensionRemappingFileSystem(RemappingFileSystem):
    """File system that remaps .py file extensions."""

    def __init__(self, underlying, extension):
        super(ExtensionRemappingFileSystem, self).__init__(underlying)
        self.extension = extension

    def map_path(self, path):
        p, ext = os.path.splitext(path)
        if ext == '.py':
            return p + '.' + self.extension
        return path


class PYIFileSystem(ExtensionRemappingFileSystem):
    """File system that remaps .py file extensions to pyi."""

    def __init__(self, underlying):
        super(PYIFileSystem, self).__init__(underlying, 'pyi')


class TarFileSystem(object):
    """Filesystem that serves files out of a .tar."""

    def __init__(self, tar):
        self.tar = tar
        self.files = list(t.name for t in tar.getmembers() if t.isfile())
        self.directories = list(t.name for t in tar.getmembers() if t.isdir())
        self.top_level = {f.split(os.path.sep)[0] for f in self.files}

    def isfile(self, path):
        return any(os.path.join(top, path) in self.files
                   for top in self.top_level)

    def isdir(self, path):
        return any(os.path.join(top, path) in self.files
                   for top in self.top_level)

    def read(self, path):
        return self.tar.extractfile(path).read()

    def refer_to(self, path):
        return 'tar:' + path

    @staticmethod
    def read_tarfile(archive_filename):
        tar = tarfile.open(archive_filename)
        return TarFileSystem(tar)


class Path(object):
    def __init__(self):
        self.paths = []

    def add_path(self, path, kind='os'):
        if kind == 'os':
            path = OSFileSystem(path)
        elif kind == 'pyi':
            path = PYIFileSystem(OSFileSystem(path))
        else:
            raise FileSystemError('Unrecognized filesystem type: ', kind)
        self.paths.append(path)

    def add_fs(self, fs):
        assert isinstance(fs, FileSystem), 'Unrecognised filesystem: %r' % fs
        self.paths.append(fs)
