"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KinesisEventSource = void 0;
const stream_1 = require("./stream");
/**
 * Use an Amazon Kinesis stream as an event source for AWS Lambda.
 */
class KinesisEventSource extends stream_1.StreamEventSource {
    constructor(stream, props) {
        super(props);
        this.stream = stream;
        this._eventSourceMappingId = undefined;
        if (this.props.batchSize !== undefined && (this.props.batchSize < 1 || this.props.batchSize > 10000)) {
            throw new Error(`Maximum batch size must be between 1 and 10000 inclusive (given ${this.props.batchSize})`);
        }
    }
    bind(target) {
        const eventSourceMapping = target.addEventSourceMapping(`KinesisEventSource:${this.stream.node.uniqueId}`, this.enrichMappingOptions({ eventSourceArn: this.stream.streamArn }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        this.stream.grantRead(target);
        // The `grantRead` API provides all the permissions recommended by the Kinesis team for reading a stream.
        // `DescribeStream` permissions are not required to read a stream as it's covered by the `DescribeStreamSummary`
        // and `SubscribeToShard` APIs.
        // The Lambda::EventSourceMapping resource validates against the `DescribeStream` permission. So we add it explicitly.
        // FIXME This permission can be removed when the event source mapping resource drops it from validation.
        this.stream.grant(target, 'kinesis:DescribeStream');
    }
    /**
     * The identifier for this EventSourceMapping
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KinesisEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.KinesisEventSource = KinesisEventSource;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2luZXNpcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImtpbmVzaXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBRUEscUNBQW1FO0FBS25FOztHQUVHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSwwQkFBaUI7SUFHdkQsWUFBcUIsTUFBdUIsRUFBRSxLQUE4QjtRQUMxRSxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUM7UUFETSxXQUFNLEdBQU4sTUFBTSxDQUFpQjtRQUZwQywwQkFBcUIsR0FBWSxTQUFTLENBQUM7UUFLakQsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsS0FBSyxTQUFTLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsR0FBRyxDQUFDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDLEVBQUU7WUFDcEcsTUFBTSxJQUFJLEtBQUssQ0FBQyxtRUFBbUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLEdBQUcsQ0FBQyxDQUFDO1NBQzdHO0lBQ0gsQ0FBQztJQUVNLElBQUksQ0FBQyxNQUF3QjtRQUNsQyxNQUFNLGtCQUFrQixHQUFHLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxzQkFBc0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLEVBQ3ZHLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxFQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsRUFBQyxDQUFDLENBQ25FLENBQUM7UUFDRixJQUFJLENBQUMscUJBQXFCLEdBQUcsa0JBQWtCLENBQUMsb0JBQW9CLENBQUM7UUFFckUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFOUIseUdBQXlHO1FBQ3pHLGdIQUFnSDtRQUNoSCwrQkFBK0I7UUFDL0Isc0hBQXNIO1FBQ3RILHdHQUF3RztRQUN4RyxJQUFJLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxNQUFNLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFXLG9CQUFvQjtRQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDLHFCQUFxQixFQUFFO1lBQy9CLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNuRjtRQUNELE9BQU8sSUFBSSxDQUFDLHFCQUFxQixDQUFDO0lBQ3BDLENBQUM7Q0FDRjtBQXBDRCxnREFvQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBraW5lc2lzIGZyb20gJ0Bhd3MtY2RrL2F3cy1raW5lc2lzJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7U3RyZWFtRXZlbnRTb3VyY2UsIFN0cmVhbUV2ZW50U291cmNlUHJvcHN9IGZyb20gJy4vc3RyZWFtJztcblxuZXhwb3J0IGludGVyZmFjZSBLaW5lc2lzRXZlbnRTb3VyY2VQcm9wcyBleHRlbmRzIFN0cmVhbUV2ZW50U291cmNlUHJvcHMge1xufVxuXG4vKipcbiAqIFVzZSBhbiBBbWF6b24gS2luZXNpcyBzdHJlYW0gYXMgYW4gZXZlbnQgc291cmNlIGZvciBBV1MgTGFtYmRhLlxuICovXG5leHBvcnQgY2xhc3MgS2luZXNpc0V2ZW50U291cmNlIGV4dGVuZHMgU3RyZWFtRXZlbnRTb3VyY2Uge1xuICBwcml2YXRlIF9ldmVudFNvdXJjZU1hcHBpbmdJZD86IHN0cmluZyA9IHVuZGVmaW5lZDtcblxuICBjb25zdHJ1Y3RvcihyZWFkb25seSBzdHJlYW06IGtpbmVzaXMuSVN0cmVhbSwgcHJvcHM6IEtpbmVzaXNFdmVudFNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIocHJvcHMpO1xuXG4gICAgaWYgKHRoaXMucHJvcHMuYmF0Y2hTaXplICE9PSB1bmRlZmluZWQgJiYgKHRoaXMucHJvcHMuYmF0Y2hTaXplIDwgMSB8fCB0aGlzLnByb3BzLmJhdGNoU2l6ZSA+IDEwMDAwKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBNYXhpbXVtIGJhdGNoIHNpemUgbXVzdCBiZSBiZXR3ZWVuIDEgYW5kIDEwMDAwIGluY2x1c2l2ZSAoZ2l2ZW4gJHt0aGlzLnByb3BzLmJhdGNoU2l6ZX0pYCk7XG4gICAgfVxuICB9XG5cbiAgcHVibGljIGJpbmQodGFyZ2V0OiBsYW1iZGEuSUZ1bmN0aW9uKSB7XG4gICAgY29uc3QgZXZlbnRTb3VyY2VNYXBwaW5nID0gdGFyZ2V0LmFkZEV2ZW50U291cmNlTWFwcGluZyhgS2luZXNpc0V2ZW50U291cmNlOiR7dGhpcy5zdHJlYW0ubm9kZS51bmlxdWVJZH1gLFxuICAgICAgdGhpcy5lbnJpY2hNYXBwaW5nT3B0aW9ucyh7ZXZlbnRTb3VyY2VBcm46IHRoaXMuc3RyZWFtLnN0cmVhbUFybn0pLFxuICAgICk7XG4gICAgdGhpcy5fZXZlbnRTb3VyY2VNYXBwaW5nSWQgPSBldmVudFNvdXJjZU1hcHBpbmcuZXZlbnRTb3VyY2VNYXBwaW5nSWQ7XG5cbiAgICB0aGlzLnN0cmVhbS5ncmFudFJlYWQodGFyZ2V0KTtcblxuICAgIC8vIFRoZSBgZ3JhbnRSZWFkYCBBUEkgcHJvdmlkZXMgYWxsIHRoZSBwZXJtaXNzaW9ucyByZWNvbW1lbmRlZCBieSB0aGUgS2luZXNpcyB0ZWFtIGZvciByZWFkaW5nIGEgc3RyZWFtLlxuICAgIC8vIGBEZXNjcmliZVN0cmVhbWAgcGVybWlzc2lvbnMgYXJlIG5vdCByZXF1aXJlZCB0byByZWFkIGEgc3RyZWFtIGFzIGl0J3MgY292ZXJlZCBieSB0aGUgYERlc2NyaWJlU3RyZWFtU3VtbWFyeWBcbiAgICAvLyBhbmQgYFN1YnNjcmliZVRvU2hhcmRgIEFQSXMuXG4gICAgLy8gVGhlIExhbWJkYTo6RXZlbnRTb3VyY2VNYXBwaW5nIHJlc291cmNlIHZhbGlkYXRlcyBhZ2FpbnN0IHRoZSBgRGVzY3JpYmVTdHJlYW1gIHBlcm1pc3Npb24uIFNvIHdlIGFkZCBpdCBleHBsaWNpdGx5LlxuICAgIC8vIEZJWE1FIFRoaXMgcGVybWlzc2lvbiBjYW4gYmUgcmVtb3ZlZCB3aGVuIHRoZSBldmVudCBzb3VyY2UgbWFwcGluZyByZXNvdXJjZSBkcm9wcyBpdCBmcm9tIHZhbGlkYXRpb24uXG4gICAgdGhpcy5zdHJlYW0uZ3JhbnQodGFyZ2V0LCAna2luZXNpczpEZXNjcmliZVN0cmVhbScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBpZGVudGlmaWVyIGZvciB0aGlzIEV2ZW50U291cmNlTWFwcGluZ1xuICAgKi9cbiAgcHVibGljIGdldCBldmVudFNvdXJjZU1hcHBpbmdJZCgpOiBzdHJpbmcge1xuICAgIGlmICghdGhpcy5fZXZlbnRTb3VyY2VNYXBwaW5nSWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignS2luZXNpc0V2ZW50U291cmNlIGlzIG5vdCB5ZXQgYm91bmQgdG8gYW4gZXZlbnQgc291cmNlIG1hcHBpbmcnKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX2V2ZW50U291cmNlTWFwcGluZ0lkO1xuICB9XG59XG4iXX0=