# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['autoembedder']

package_data = \
{'': ['*']}

install_requires = \
['dask==2022.12.0',
 'einops==0.6.0',
 'fastparquet==2022.12.0',
 'numpy==1.23.5',
 'pandas==1.5.2',
 'pytorch-ignite==0.4.10',
 'tensorboard==2.11.0',
 'torch==1.12.1',
 'torchinfo==1.7.1',
 'tqdm==4.64.1']

setup_kwargs = {
    'name': 'autoembedder',
    'version': '0.1.20',
    'description': 'PyTorch autoencoder with additional embeddings layer for categorical data.',
    'long_description': '![The autoembedder](https://raw.githubusercontent.com/chrislemke/autoembedder/master/docs/assets/images/image.png)\n# The Autoembedder\n[![deploy package](https://github.com/chrislemke/autoembedder/actions/workflows/deploy-package.yml/badge.svg)](https://github.com/chrislemke/autoembedder/actions/workflows/deploy-package.yml)\n[![pypi](https://img.shields.io/pypi/v/autoembedder)](https://pypi.org/project/autoembedder/)\n[![python version](https://img.shields.io/pypi/pyversions/autoembedder?logo=python&logoColor=yellow)](https://python.org)\n[![docs](https://img.shields.io/badge/docs-mkdoks%20material-blue)](https://chrislemke.github.io/autoembedder/)\n[![license](https://img.shields.io/github/license/chrislemke/autoembedder)](https://github.com/chrislemke/autoembedder/blob/main/LICENSE)\n[![downloads](https://img.shields.io/pypi/dm/autoembedder)](https://pypistats.org/packages/autoembedder)\n[![mypy](http://www.mypy-lang.org/static/mypy_badge.svg)](http://mypy-lang.org/)\n[![black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![isort](https://img.shields.io/badge/%20imports-isort-%231674b1?style=flat&labelColor=ef8336)](https://pycqa.github.io/isort/)\n[![pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/pre-commit/pre-commit)\n## Introduction\nThe Autoembedder is an autoencoder with additional embedding layers for the categorical columns. Its usage is flexible, and hyperparameters like the number of layers can be easily adjusted and tuned. The data provided for training can be either a path to a [Dask](https://docs.dask.org/en/stable/dataframe.html) or [Pandas](https://pandas.pydata.org/) DataFrame stored in the Parquet format or the DataFrame object directly.\n\n## Installation\nIf you are using [Poetry](https://python-poetry.org/), you can install the package with the following command:\n```bash\npoetry add autoembedder\n```\nIf you are using [pip](https://pypi.org/project/pip/), you can install the package with the following command:\n```bash\npip install autoembedder\n```\n\n\n## Installing dependencies\nWith [Poetry](https://python-poetry.org/):\n```bash\npoetry install\n```\nWith [pip](https://pypi.org/project/pip/):\n```bash\npip install -r requirements.txt\n```\n## Usage\n### 0. Some imports\n```python\nfrom autoembedder import Autoembedder, dataloader, fit\n```\n### 1. Create dataloaders\nFirst, we create two [`dataloaders`](https://chrislemke.github.io/autoembedder/autoembedder.data/#autoembedder.data.Dataset.__init__). One for training, and the other for validation data. As `source` they either accept a path to a Parquet file, to a folder of Parquet files or a [Pandas](https://pandas.pydata.org/docs/reference/api/pandas.DataFrame.html)/[Dask](https://docs.dask.org/en/stable/dataframe.html) DataFrame.\n```python\ntrain_dl = dataloader(train_df)\nvalid_dl = dataloader(vaild_df)\n```\n\n### 2. Set parameters\nNow, we need to set the parameters. They are going to be used for handling the data and training the model. In this example, only parameters for the training are set. [Here](https://github.com/chrislemke/autoembedder#parameters) you find a list of all possible parameters. This should do it:\n```python\nparameters = {\n    "hidden_layers": [[25, 20], [20, 10]],\n    "epochs": 10,\n    "lr": 0.0001,\n    "verbose": 1,\n}\n```\n\n### 3. Initialize the autoembedder\nThen, we need to initialize the [autoembedder](https://chrislemke.github.io/autoembedder/autoembedder.model/#autoembedder.model.Autoembedder). In this example, we are not using any categorical features. So we can skip the `embedding_sizes` argument.\n```python\nmodel = Autoembedder(parameters, num_cont_features=train_df.shape[1])\n```\n\n### 4. Train the model\nEverything is set up. Now we can [fit](https://chrislemke.github.io/autoembedder/autoembedder.learner/#autoembedder.learner.fit) the model.\n```python\nfit(parameters, model, train_dl, valid_dl)\n```\n\n## Example\nCheck out [this Jupyter notebook](https://github.com/chrislemke/autoembedder/blob/main/example.ipynb) for an applied example using the [Credit Card Fraud Detection](https://www.kaggle.com/datasets/mlg-ulb/creditcardfraud) from Kaggle.\n\n## Parameters\nThis is a list of all parameters that can be passed to the Autoembedder for training. The `Required`, `Default value`, and `Comment` columns are only apply if using the training script (`training.py`):\n\n| Argument             | Type  | Required | Default value                 | Comment                                                                                                                                                                                  |\n| -------------------- | ----- | -------- | ----------------------------- | ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |\n| batch_size           | int   | False    | 32                            |                                                                                                                                                                                          |\n| drop_last            | int   | False    | 1                             | True/False                                                                                                                                                                               |\n| pin_memory           | int   | False    | 1                             | True/False                                                                                                                                                                               |\n| num_workers          | int   | False    | 0                             | 0 means that the data will be loaded in the main process                                                                                                                                 |\n| use_mps              | int   | False    | 0                             | Set this to `1` if you want to use the [MPS Backend](https://pytorch.org/docs/master/notes/mps.html) for running on Mac using the M1 GPU. process                                        |\n| model_title          | str   | False    | autoembedder_{`datetime`}.bin |                                                                                                                                                                                          |\n| model_save_path      | str   | False    |                               |                                                                                                                                                                                          |\n| n_save_checkpoints   | int   | False    |                               |                                                                                                                                                                                          |\n| lr                   | float | False    | 0.001                         |                                                                                                                                                                                          |\n| amsgrad              | int   | False    | 0                             | True/False                                                                                                                                                                               |\n| epochs               | int   | True     |                               |\n| dropout_rate         | float | False    | 0                             | Dropout rate for the dropout layers in the encoder and decoder.                                                                                                                          |\n| layer_bias           | int   | False    | 1                             | True/False                                                                                                                                                                               |  |\n| weight_decay         | float | False    | 0                             |                                                                                                                                                                                          |\n| l1_lambda            | float | False    | 0                             |                                                                                                                                                                                          |\n| xavier_init          | int   | False    | 0                             | True/False                                                                                                                                                                               |\n| activation           | str   | False    | tanh                          | Activation function; either `tanh`, `relu`, `leaky_relu` or `elu`                                                                                                                        |\n| tensorboard_log_path | str   | False    |                               |                                                                                                                                                                                          |\n| trim_eval_errors     | int   | False    | 0                             | Removes the max and min loss when calculating the `mean loss diff` and `median loss diff`. This can be useful if some rows create very high losses.                                      |\n| verbose              | int   | False    | 0                             | Set this to `1` if you want to see the model summary and the validation and evaluation results. set this to `2` if you want to see the training progress bar. `0` means no output.       |\n| target               | str   | False    |                               | The target column. If not set no evaluation will be performed.                                                                                                                           |\n| train_input_path     | str   | True     |                               |                                                                                                                                                                                          |\n| test_input_path      | str   | True     |                               |\n| eval_input_path      | str   | False    |                               | Path to the evaluation data. If no path is provided no evaluation will be performed.                                                                                                     |  |\n| hidden_layers        | str   | True     |                               | Contains a string representation of a list of list of integers which represents the hidden layer structure. E.g.: `"[[64, 32], [32, 16], [16, 8]]"` activation                           |\n| cat_columns          | str   | False    | "[]"                          | Contains a string representation of a list of list of categorical columns (strings). The columns which use the same encoder should be together in a list. E.g.: `"[[\'a\', \'b\'], [\'c\']]"`. |\n\n\n## Run the training script\nSomething like this should do it:\n```bash\npython3 training.py --epochs 20 \\\n--train_input_path "path/to/your/train_data" \\\n--test_input_path "path/to/your/test_data" \\\n--hidden_layers "[[12, 6], [6, 3]]"\n```\n\n\n## Why additional embedding layers?\nThe additional embedding layers automatically embed all columns with the Pandas `category` data type. If categorical columns have another data type, they will not be embedded and will be handled like continuous columns. Simply encoding the categorical values (e.g., with the usage of a label encoder) decreases the quality of the outcome.\n',
    'author': 'Christopher Lemke',
    'author_email': 'chris@syhbl.mozmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://chrislemke.github.io/autoembedder/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
