"""
EXPERIMENTAL
Async "fast translation" using stix-shifter and asyncpg
NB: This interface will likely change in the near future.
"""

import logging
import uuid
from collections import OrderedDict, defaultdict
from datetime import datetime

import pandas as pd
import ujson

from firepit.aio.asyncstorage import AsyncStorage
from firepit.exceptions import DuplicateTable
from firepit.props import KNOWN_PROPS
from firepit.raft import json_normalize
from firepit.splitter import shorten_extension_name
from firepit.stix21 import makeid
from firepit.timestamp import timefmt


logger = logging.getLogger(__name__)


#TODO: These SQL schemas should live someplace common.
#TODO: Create a function that will build table creation stmt from this schema
IDENTITY_SCHEMA = {
    "id": "TEXT UNIQUE",
    "identity_class": "TEXT",
    "name": "TEXT",
    "created": "TEXT",
    "modified": "TEXT"
}


COLUMNS_SCHEMA = {
    'otype': 'TEXT',
    'path': 'TEXT',
    'shortname': 'TEXT',
    'dtype': 'TEXT'
}


CONTAINS_SCHEMA = {
    'source_ref': 'TEXT',
    'target_ref': 'TEXT',
    'x_firepit_rank': 'INTEGER',
}


REFLIST_SCHEMA = {
    'ref_name': 'TEXT',
    'source_ref': 'TEXT',
    'target_ref': 'TEXT',
}


def _make_colname(shifter_mapping: dict):
    '''Convert a stix-shifter mapping key to valid STIX object path'''
    shifter_name = shifter_mapping['key']
    parts = shifter_name.split('.')
    outs = []
    for part in parts[1:]:
        if '-' in part:
            part = f"'{part}'"
        outs.append(part)
    if not shifter_mapping.get('cybox', True):
        prop = '.'.join([parts[0]] + outs)
        return f'observed-data:{prop}'
    obj = f'{shifter_mapping["object"]}#' if 'object' in shifter_mapping else ''
    return f"{obj}{parts[0]}:{'.'.join(outs)}" if len(parts) > 1 else f'observed-data:{shifter_name}'


def _get_mapping(mapping: dict, col: str):
    '''Return the stix-shifter mapping for a specific native result col'''
    cmap = mapping.get(col)
    if cmap is None:
        # Look for "nested" mappings, ala elastic_ecs
        tmp = mapping
        for part in col.split('.'):
            tmp = tmp.get(part)
            if not tmp:
                break
        cmap = tmp
    if cmap is not None and not isinstance(cmap, list):
        # there could me more than 1 mapping target
        cmap = [cmap]
    return cmap


def _is_nested(mapping: dict):
    for v in mapping.values():
        if isinstance(v, dict) and "key" not in v:
            return True
    return False


PROTOCOL_LAYERS = {
    'phy': 1,
    'physical': 1,
    'eth': 2,
    'ethernet': 2,
    'ip': 3,
    'ipv4': 3,
    'ipv6': 3,
    'icmp': 4,
    'icmp6': 4,
    'icmpv6': 4,
    'udp': 4,
    'tcp': 4,
    'sctp': 4,
    'ssl': 5,
    'tls': 5,
    'https': 5,
    'ssh': 5,
    'http': 6, # Want this to be > tls
}


def _to_protocols(value):
    '''Special transformer for network-traffic:protocols'''
    if isinstance(value, str):
        value = [value.lower()]
    else:
        # elastic_ecs can have "-" in protocols, but stix-shifter doesn't remove it
        #value = [str(i).lower() for i in value if i not in ('', '-')]
        value = [str(i).lower() for i in value if i != '']
        value = sorted(value, key=lambda x: PROTOCOL_LAYERS.get(x, 8))
    return value


def translate(
        to_stix_map: dict,
        transformers: dict,
        events: list,
        identity: dict
) -> pd.DataFrame:
    """Create a DataFrame from native events as returned by stix-shifter"""
    if _is_nested(to_stix_map):
        # This handles "nested" events like elastic_ecs result format
        events = [json_normalize(e, flat_lists=False) for e in events]
    df = pd.DataFrame.from_records(events)
    df = df.dropna(how='all', axis=1)  # Drop empty columns
    cols = list(df.columns)

    # At this point, the column names are straight from the
    # (flattened) native results.

    # Columns to duplicate (assume same Transformer?)
    #dup_cols = {}
    dup_cols = defaultdict(list)

    # Transformers
    txf_cols = {}

    # Uniform value columns
    val_cols = {}

    # Reference columns
    ref_objs = {}  # Original column -> ref object names
    ref_cols = {}

    # Columns to rename/map
    renames = {}

    # Columns to drop
    drop_cols = set()

    # columns we need to "group"; new name -> set(orig cols)
    group = defaultdict(list)

    # columns we need to "unwrap"
    unwrap = set()

    logger.debug('columns: %s', cols)
    for col in cols:
        logger.debug('column: %s', col)
        cmap = _get_mapping(to_stix_map, col)
        if cmap:
            for i, col_mapping in enumerate(cmap):
                logger.debug('\t%d: %s', i, col_mapping)
                new_col = _make_colname(col_mapping)
                if col_mapping.get('unwrap', False):
                    # split to column into multiple
                    if not new_col.endswith('_ref') and not new_col.endswith('_refs'):
                        unwrap.add(col)
                if col_mapping['key'] in ('ipv4-addr.value', 'ipv6-addr.value'):
                    # Need some special handing for disambiguating IPv4 and IPv6
                    logger.debug('DUP/DROP column "%s" to "%s"', col, new_col)
                    dup_cols[col].append(new_col)
                    drop_cols.add(col)
                elif i == 0:
                    # The first mapping is a simple rename
                    if col_mapping.get('group', False):
                        logger.debug('GROUP column "%s" into "%s"', col, new_col)
                        group[new_col].append(col)
                    else:
                        logger.debug('RENAME column "%s" to "%s"', col, new_col)
                        renames[col] = new_col
                else:
                    # Subsequent mappings are either refs or dups
                    if new_col.endswith('_ref') or new_col.endswith('_refs'):
                        logger.debug('REF column "%s" to "%s"', col, new_col)
                        refs = col_mapping['references']
                        ref_cols[new_col] = refs
                        ref_objs[col] = refs
                    else:
                        # If not a ref, duplicate the column
                        logger.debug('DUP column "%s" to "%s"', col, new_col)
                        dup_cols[col].append(new_col)

                # get transform for this mapping
                if 'transformer' in col_mapping:
                    txf_cols[new_col] = col_mapping['transformer']
                elif 'value' in col_mapping:
                    # It's a constant value for every row
                    val_cols[new_col] = col_mapping['value']
        else:
            # Drop unmapped columns
            logger.debug('DROP unmapped column "%s"', col)
            df = df.drop(col, axis=1)

    logger.debug('Before unwrap: columns = %s', df.columns)

    # Unwrap list columns first to take advantage of v4/v6 filtering and minimize column copies
    obj_renames = defaultdict(list)
    for col in unwrap:
        if col not in df.columns:
            logger.debug('unwrap not found: %s', col)
            continue
        logger.debug('UNWRAP "%s"', col)
        ser = df[col].dropna().apply(pd.Series)
        if ser.empty:
            logger.debug('unwrap empty column: %s', col)
            continue

        prefix = col + '_'

        # Build new df with col names like host.ip_0, host.ip_1, etc.
        # 1 new col for each item in the longest list
        ndf = pd.DataFrame(ser, index=df.index).add_prefix(prefix)
        if ndf.iloc[:, 0].count():  # grab the non-empty columns
            col_renames = list(ndf.columns)
            logger.debug('unwrap col_renames: %s', col_renames)

            # find new name and get obj, rest
            new_col = renames.get(col)
            if new_col:
                # Save object renames since we'll need them when resolving references
                obj, _, rest = new_col.rpartition('#')

                # Update renames, using new "indexed" unwrapped columns
                update = {c: f'{obj}_{i}#{rest}' for i, c in enumerate(ndf.columns)}
                logger.debug('UPDATE renames for %s to %s', col, update)
                renames.update(update)
                drop_cols.add(col)
                del renames[col]

            # Replace original col with new cols in our original df
            df = pd.concat([df, ndf], axis=1)

            # Rewrite dup_cols: need to remove unwrapped cols and add to renames
            dup_list = dup_cols.get(col)
            if dup_list:
                logger.debug('UPDATE dup_cols for %s', col)
                for new_col in col_renames:
                    _, _, idx = new_col.partition('_')
                    new_list = [dup_col.replace('#', f'_{idx}#') for dup_col in dup_list]
                    logger.debug('%s: new_col = %s, new_list = %s', col, new_col, new_list)
                    dup_cols[new_col] = new_list
                    drop_cols.add(new_col)  # Need to drop these after duplicating
                    # TODO: update transform columns too (txf_cols)?
                drop_cols.add(col)
                del dup_cols[col]

            # Update object names for references
            refs = ref_objs.get(col, [])
            if not isinstance(refs, list):
                refs = [refs]
            for ref in refs:
                new_obj = [f'{ref}_{i}' for i in range(len(ndf.columns))]
                obj_renames[ref].extend(new_obj)

    logger.debug('After unwrap: columns = %s', df.columns)

    # Duplicate columns as necessary
    for orig_col, dup_list in dup_cols.items():
        if orig_col not in df.columns:
            continue
        # Things like "src" could be blindly mapped to both ipv4 and v6
        # Split them out appropriately by looking at the actual values
        for dup_col in dup_list:
            if dup_col.endswith('ipv4-addr:value'):  # and dup_col not in unwrap:
                logger.debug('DUP column "%s" to "%s", filtering for IPv4', orig_col, dup_col)
                df[dup_col] = df[orig_col].where(df[orig_col].str.contains(r'\.'), None).copy()
            elif dup_col.endswith('ipv6-addr:value'):  # and dup_col not in unwrap:
                logger.debug('DUP column "%s" to "%s", filtering for IPv6', orig_col, dup_col)
                df[dup_col] = df[orig_col].where(df[orig_col].str.contains(':'), None).copy()
            else:
                logger.debug('DUP column "%s" to "%s"', orig_col, dup_col)
                df[dup_col] = df[orig_col].copy()

    # Rename columns
    logger.debug('RENAME: %s', renames)
    df.rename(columns=renames, inplace=True)

    # Drop columns we don't need anymore
    logger.debug('DROP columns %s', drop_cols)
    df = df.drop(drop_cols, axis=1)

     # Merge group columns
    for new_col, orig_cols in group.items():
        # Combine columns into single list column
        logger.debug('Group %s into "%s"', orig_cols, new_col)
        df[new_col] = df[orig_cols].agg(lambda x: [i for i in list(x) if not pd.isna(i)], axis=1)  #.values.tolist()
        df = df.drop(orig_cols, axis=1)

    # Run transformers
    for txf_col, txf_name in txf_cols.items():
        logger.debug('transform: %s %s', txf_col, txf_name)
        # Accelerate common transforms
        if txf_name == 'ToInteger':
            df[txf_col] = df[txf_col].dropna().astype('int')
        elif txf_name == 'EpochToTimestamp':  # QRadar, QDL
            df[txf_col] = (pd.to_datetime(df[txf_col],
                                          unit="ms",
                                          infer_datetime_format=True)
                           .dt.strftime("%Y-%m-%dT%H:%M:%S.%fZ"))
        elif txf_name in ('FilterIPv4List', 'FilterIPv6List'):
            pass  # We already did this
        #TODO:elif txf_name == 'IntToBool':  # QDL
        #TODO:elif txf_name == 'ToLowercaseArray': QRadar, Splunk, Elastic
        else:
            txf = transformers.get(txf_name)
            if txf:
                try:
                    if (txf_name == 'ToLowercaseArray' and
                        txf_col.endswith('network-traffic:protocols')):
                        # Need to properly sort them
                        df[txf_col] = df[txf_col].apply(_to_protocols)
                    else:
                        df[txf_col] = df[txf_col].dropna().apply(txf.transform)
                except AttributeError as e:
                    logger.error("%s", e, exc_info=e)
                    #TODO: what do we do here?

    # drop empty columns that may have been created by transforms, etc.
    df = df.dropna(how='all', axis=1)

    # Create constant value columns
    for val_col, value in val_cols.items():
        logger.debug('VAL column "%s"', val_col)
        df[val_col] = value

    # create set of obj_names#types
    obj_set = set()
    for col in df.columns:  #TODO: Can we do this above somewhere?
        # col is in the form [<obj_name>#]<obj_type>:<obj_attr>
        c = col.find(':')
        obj_key = col[:c]
        logger.debug('OBJ %s', obj_key)
        obj_set.add(obj_key)

    # Resolved reference columns (map of object name to object ID column name)
    ref_ids = defaultdict(list)

    # Generate STIX 2.1 id using firepit.stix21.makeid()
    # This is expensive!
    for obj_key in obj_set:
        obj, _, sco_type = obj_key.rpartition('#')
        if not obj:
            continue  # i.e. skip observed-data
        logger.debug('ID: obj "%s" type %s', obj, sco_type)
        cols = sorted([c for c in df.columns if c.startswith(f'{obj_key}:')])
        id_col = f'{obj_key}:id'
        ref_ids[obj].append(id_col)
        odf = df[cols]
        ids = []
        # Convert to dicts so we can call makeid
        #TODO: compute id using itertuples instead?
        for rec in odf.to_dict(orient='records'):
            if not any(rec.values()):
                ids.append(None)
                continue
            sco = {k.rpartition(':')[2]: v for k, v in rec.items() if isinstance(v, (list, dict)) or not pd.isna(v)}
            #if sco_type in ('ipv4-addr', 'ipv6-addr', 'mac-addr') and pd.isna(sco.get('value')):
            if len(sco) == 0:
                ids.append(None)
            else:
                sco['type'] = sco_type
                ids.append(makeid(sco))
        df[id_col] = ids

    # Generate STIX ID for observed-data, plus other required columns
    df = (df.assign(temp=[f'observed-data--{uuid.uuid4()}' for i in range(len(df.index))])
          .rename(columns={'temp': 'observed-data:id'}))
    df['observed-data:created_by_ref'] = identity['id']
    df['observed-data:created'] = timefmt(datetime.utcnow())
    df['observed-data:modified'] = df['observed-data:created']

    # Create ref columns
    for ref_col, value in ref_cols.items():
        # value could be an object name or list of object names
        logger.debug('REF "%s" value "%s"', ref_col, value)
        # Ignore ref if we don't have any source objects
        c = ref_col.find(':')
        obj_key = ref_col[:c]
        if obj_key not in obj_set:
            logger.debug('REF "%s" has no source object', ref_col)
            continue

        # Figure out what id_cols this object maps to
        if isinstance(value, str):
            id_cols = ref_ids.get(value)
            if id_cols:
                logger.debug('REF "%s" to "%s"', ref_col, id_cols)
                df[ref_col] = df[id_cols].bfill(axis=1).iloc[:, 0]
        elif isinstance(value, list):  # reflists
            # Remap and flatten the list first
            l = [obj_renames.get(i, [i]) for i in value]
            value = [i for new_list in l for i in new_list]
            new_list = []
            for i in value:
                i = ref_ids.get(i)
                if isinstance(i, list):
                    new_list.extend(i)
                else:
                    new_list.append(i)
            id_cols = [i for i in new_list if i in df.columns]
            df[ref_col] = [[e for e in row if not pd.isna(e)] for row in df[id_cols].values.tolist()]
            df[ref_col] = df[ref_col].mask(df[ref_col].str.len() == 0, None)

    return df


async def _get_schemas(writer):
    props = await writer.properties()
    schemas = defaultdict(OrderedDict)
    for prop in props:
        schema = schemas[prop['table']]
        schema[prop['name']] = prop['type']
    return schemas


def _infer_type(writer, obj_attr, value, dtype):
    if dtype == 'int64':
        return 'BIGINT'
    return writer.infer_type(obj_attr, value)


async def ingest(
        store: AsyncStorage,
        identity: dict,
        df: pd.DataFrame,
        query_id: str
):
    """Ingest translated DataFrame into firepit async storage"""
    logger.debug('df.columns = %s', df.columns)
    writer = store

    # First insert the identity object
    await writer.write_records('identity', [identity], IDENTITY_SCHEMA, False, query_id)

    # Need to split df and rename columns
    schemas = defaultdict(OrderedDict)
    objects = {}
    columns = []
    for col in df.columns:
        # col is in the form [<obj_name>#]<obj_type>:<obj_attr>
        h = col.find('#') # Might be able to do all this in advance?
        c = col.find(':')
        if h > -1:
            obj_name = col[:h]
        else:
            obj_name = ''
        obj_type = col[h + 1:c]
        obj_attr = col[c + 1:]
        objects[obj_name] = obj_type
        if col.endswith('_refs'):
            continue
        pd_col = df[col]
        index = pd_col.first_valid_index()
        if index is not None:
            value = pd_col.loc[index]
        else:
            value = None
        meta = KNOWN_PROPS.get(obj_type, {}).get(obj_attr)
        if meta:
            dtype = meta['dtype']
        else:
            if isinstance(value, list):
                # firepit encodes lists as JSON
                dtype = 'list'
            else:
                dtype = value.__class__.__name__
        if dtype == 'list':
            df[col] = df[col].apply(lambda x: ujson.dumps(x, ensure_ascii=False))
        schemas[obj_type][obj_attr] = _infer_type(writer, obj_attr, value, dtype)
        logger.debug('ingest: col %s val %s dtype %s (%s)', col, value, dtype, schemas[obj_type][obj_attr])

        # shorten key (STIX prop) to make column names more manageable
        if len(obj_attr) > 63 or 'extensions.' in obj_attr:
            shortname = shorten_extension_name(obj_attr)  # Need to detect collisions!
        else:
            shortname = obj_attr

        # Generate __columns entry
        if dtype.endswith('64'):
            dtype = dtype[:-2]  # e.g. turn int64 to int
        columns.append({
            # Not needed: 'type': '__columns',
            'otype': obj_type,
            'path': obj_attr,
            'shortname': shortname,
            'dtype': dtype,
        })

    col_df = pd.DataFrame.from_records(columns)
    col_df = col_df.drop_duplicates(['otype', 'path'])
    await writer.write_df('__columns', col_df, None, COLUMNS_SCHEMA)

    # load existing schemas
    old_schemas = await _get_schemas(writer)

    for obj_name, obj_type in objects.items():
        if obj_name:
            prefix = f'{obj_name}#{obj_type}:'
        else:
            prefix = f'{obj_type}:'
        cols = sorted([c for c in df.columns if c.startswith(prefix)])
        odf = df[cols].dropna().copy()
        odf.columns = [c.rpartition(':')[2] for c in cols]
        logger.debug('Columns for "%s" (%s): %s', obj_name, obj_type, list(odf.columns))

        if 'id' not in odf.columns:
            logger.debug('No id property for "%s" (%s)', obj_name, obj_type)
            continue

        #TODO: need to merge dups, not drop them
        #TODO: use groupby?  Would need to build agg dict first.
        odf = odf.drop_duplicates('id')

        schema = schemas[obj_type]

        # check if we need create this table
        if obj_type not in old_schemas:
            try:
                await writer.new_type(obj_type, schema)
                old_schemas[obj_type] = schema
            except DuplicateTable:
                logger.debug('Duplicate table "%s"', obj_type)
        else:
            # check if we need to alter table using writer.new_property
            for new_col in set(schema.keys()).difference(old_schemas[obj_type].keys()):
                await writer.new_property(obj_type, new_col, schema[new_col])

        # Move reflist cols to new df
        reflist_cols = [c for c in list(odf.columns) if c.endswith('_refs')]
        if reflist_cols:
            ref_df = odf[['id'] + reflist_cols]
            odf = odf.drop(reflist_cols, axis=1)
        await writer.write_df(obj_type, odf, query_id, schema)
        if obj_type == 'observed-data':
            continue

        # __contains table - maps observed-data to SCO
        id_col = f'{prefix}id'
        con_cols = ['observed-data:id', id_col]
        cdf = df[con_cols].copy()
        cdf = cdf.drop_duplicates(con_cols)
        cdf['x_firepit_rank'] = 1  # Initialize rank for everything to 1
        if (obj_name.startswith('dst') or
            obj_name.startswith('destination') or
            obj_name.startswith('target')):  # TODO: need better heuristic
            cdf['x_firepit_rank'] = 0  # firepit normally uses None here

        cdf.rename(columns={'observed-data:id': 'source_ref', id_col: 'target_ref'}, inplace=True)
        await writer.write_df('__contains', cdf, None, CONTAINS_SCHEMA)

        # __reflist table - for SCO reference lists
        # Create new df from odf, then use rdf.explode(reflist_col)
        created_table = False
        for col in reflist_cols:
            if not created_table:
                try:
                    await writer.new_type('__reflist', REFLIST_SCHEMA)  # Shouldn't really have to do this
                except DuplicateTable:
                    pass
                created_table = True
            # Create new df with obj id and reflist column
            rdf = ref_df[['id', col]].copy()
            rdf.rename(columns={'id': 'source_ref', col: 'target_ref'}, inplace=True)
            rdf['ref_name'] = col
            rdf = rdf.explode('target_ref').drop_duplicates(list(rdf.columns))
            await writer.write_df('__reflist', rdf, None, REFLIST_SCHEMA)
