import abc
import dataclasses
import functools
import re
from typing import Optional, Dict, Pattern, Tuple, Any

from manubot.util import import_function
from .citekey import CiteKey

"""
Non-citation prefixes used by the pandoc-xnos suite of Pandoc filters,
including pandoc-fignos, pandoc-tablenos, and pandoc-eqnos.
"""
_pandoc_xnos_prefixes = {"fig", "tbl", "eq"}


_local_handlers = [
    "manubot.cite.arxiv.Handler_arXiv",
    "manubot.cite.doi.Handler_DOI",
    "manubot.cite.isbn.Handler_ISBN",
    "manubot.cite.pubmed.Handler_PMC",
    "manubot.cite.pubmed.Handler_PubMed",
    "manubot.cite.url.Handler_URL",
    "manubot.cite.wikidata.Handler_Wikidata",
]


@functools.lru_cache(maxsize=10_000)
def get_handler(prefix_lower: str) -> "Handler":
    if not isinstance(prefix_lower, str):
        raise TypeError(
            f"prefix_lower should be a str, instead received {prefix_lower.__class__.__name__}"
        )
    assert prefix_lower == prefix_lower.lower()
    handler = prefix_to_handler[prefix_lower]
    handler = import_function(handler)(prefix_lower)
    return handler


def _generate_prefix_to_handler() -> Dict[str, str]:
    """
    Generate complete dictionary for prefix_to_handler.
    """
    import inspect

    from .curie import get_curie_handlers

    curie_handlers = get_curie_handlers()
    pth = {}
    for handler in curie_handlers + _local_handlers:
        if isinstance(handler, str):
            handler = import_function(handler)("dummy_prefix_lower")
        for prefix in handler.prefixes:
            pth[
                prefix
            ] = f"{inspect.getmodule(handler).__name__}.{handler.__class__.__name__}"
    pth = dict(sorted(pth.items()))  # sort for clean diffs of serialized dict
    return pth


@dataclasses.dataclass
class Handler:
    """
    A Handler is a class that provides support for a certain type of citekey.
    For example, a Handler subclass could provide support for DOI citekeys.
    Subclasses enable custom logic for different citekey prefixes,
    including how to standardize the citekey and how to retrieve CSL Item metadata.
    """

    prefix_lower: str
    prefixes = []

    def _get_pattern(self, attribute="accession_pattern") -> Pattern:
        """
        Return a compiled regex pattern stored by `attribute`.
        By default, return `self.accession_pattern`, which Handler subclasses
        can set to provide the expected pattern for `self.accession`.
        """
        # todo: consider caching compilation
        pattern = getattr(self, attribute, None)
        if not pattern:
            return None
        if not isinstance(pattern, Pattern):
            pattern = re.compile(pattern)
        return pattern

    def inspect(self, citekey: CiteKey) -> Optional[str]:
        """
        Check citekeys adhere to expected formats. If an issue is detected a
        string describing the issue is returned. Otherwise returns None.
        """
        pattern = self._get_pattern("accession_pattern")
        if not pattern:
            return None
        if not pattern.fullmatch(citekey.accession):
            return f"{citekey.accession} does not match regex {pattern.pattern}"

    def standardize_prefix_accession(self, accession: str) -> Tuple[str, str]:
        """
        Return (prefix, accession) in standardized form.
        This method defaults to returning `self.standard_prefix`
        (or `self.prefix_lower` if standard_prefix is not defined).
        Subclasses can override this method with more specific standardization logic.
        """
        standard_prefix = getattr(self, "standard_prefix", self.prefix_lower)
        standard_accession = accession
        return standard_prefix, standard_accession

    @abc.abstractmethod
    def get_csl_item(self, citekey) -> Dict[str, Any]:
        """
        Return a CSL_Item with bibliographic details for citekey.
        """
        raise NotImplementedError(
            f"Manubot does not know how to generate a csl_item for {citekey.standard_id!r}"
        )


"""
Mapping from lowercase prefix to Handler class function location as string.
This output is automatically generated using `_generate_prefix_to_handler`.
Hardcoding this mapping reduces startup time and helps keep imports to a minimum,
allowing installations without the full dependencies to function.
"""
prefix_to_handler: Dict[str, str] = {
    "3dmet": "manubot.cite.curie.Handler_CURIE",
    "abs": "manubot.cite.curie.Handler_CURIE",
    "aceview.worm": "manubot.cite.curie.Handler_CURIE",
    "addgene": "manubot.cite.curie.Handler_CURIE",
    "adw": "manubot.cite.curie.Handler_CURIE",
    "affy.probeset": "manubot.cite.curie.Handler_CURIE",
    "aftol.taxonomy": "manubot.cite.curie.Handler_CURIE",
    "agricola": "manubot.cite.curie.Handler_CURIE",
    "allergome": "manubot.cite.curie.Handler_CURIE",
    "amoebadb": "manubot.cite.curie.Handler_CURIE",
    "antibodyregistry": "manubot.cite.curie.Handler_CURIE",
    "antweb": "manubot.cite.curie.Handler_CURIE",
    "aop": "manubot.cite.curie.Handler_CURIE",
    "aop.events": "manubot.cite.curie.Handler_CURIE",
    "aop.relationships": "manubot.cite.curie.Handler_CURIE",
    "aop.stressor": "manubot.cite.curie.Handler_CURIE",
    "apd": "manubot.cite.curie.Handler_CURIE",
    "aphidbase.transcript": "manubot.cite.curie.Handler_CURIE",
    "apid.interactions": "manubot.cite.curie.Handler_CURIE",
    "arachnoserver": "manubot.cite.curie.Handler_CURIE",
    "ardb": "manubot.cite.curie.Handler_CURIE",
    "ark": "manubot.cite.curie.Handler_CURIE",
    "arrayexpress": "manubot.cite.curie.Handler_CURIE",
    "arrayexpress.platform": "manubot.cite.curie.Handler_CURIE",
    "arraymap": "manubot.cite.curie.Handler_CURIE",
    "arxiv": "manubot.cite.arxiv.Handler_arXiv",
    "asap": "manubot.cite.curie.Handler_CURIE",
    "ascl": "manubot.cite.curie.Handler_CURIE",
    "asin": "manubot.cite.curie.Handler_CURIE",
    "aspgd.locus": "manubot.cite.curie.Handler_CURIE",
    "aspgd.protein": "manubot.cite.curie.Handler_CURIE",
    "atc": "manubot.cite.curie.Handler_CURIE",
    "atcc": "manubot.cite.curie.Handler_CURIE",
    "atcvet": "manubot.cite.curie.Handler_CURIE",
    "atfdb.family": "manubot.cite.curie.Handler_CURIE",
    "autdb": "manubot.cite.curie.Handler_CURIE",
    "bacdive": "manubot.cite.curie.Handler_CURIE",
    "bacmap.biog": "manubot.cite.curie.Handler_CURIE",
    "bacmap.map": "manubot.cite.curie.Handler_CURIE",
    "bao": "manubot.cite.curie.Handler_CURIE",
    "bdgp.est": "manubot.cite.curie.Handler_CURIE",
    "bdgp.insertion": "manubot.cite.curie.Handler_CURIE",
    "bdsc": "manubot.cite.curie.Handler_CURIE",
    "beetlebase": "manubot.cite.curie.Handler_CURIE",
    "begdb": "manubot.cite.curie.Handler_CURIE",
    "bgee.family": "manubot.cite.curie.Handler_CURIE",
    "bgee.gene": "manubot.cite.curie.Handler_CURIE",
    "bgee.organ": "manubot.cite.curie.Handler_CURIE",
    "bgee.stage": "manubot.cite.curie.Handler_CURIE",
    "bigg.compartment": "manubot.cite.curie.Handler_CURIE",
    "bigg.metabolite": "manubot.cite.curie.Handler_CURIE",
    "bigg.model": "manubot.cite.curie.Handler_CURIE",
    "bigg.reaction": "manubot.cite.curie.Handler_CURIE",
    "bindingdb": "manubot.cite.curie.Handler_CURIE",
    "biocarta.pathway": "manubot.cite.curie.Handler_CURIE",
    "biocatalogue.service": "manubot.cite.curie.Handler_CURIE",
    "biocyc": "manubot.cite.curie.Handler_CURIE",
    "biogrid": "manubot.cite.curie.Handler_CURIE",
    "biolink": "manubot.cite.curie.Handler_CURIE",
    "biominder": "manubot.cite.curie.Handler_CURIE",
    "biomodels.db": "manubot.cite.curie.Handler_CURIE",
    "biomodels.kisao": "manubot.cite.curie.Handler_CURIE",
    "biomodels.teddy": "manubot.cite.curie.Handler_CURIE",
    "biomodels.vocabulary": "manubot.cite.curie.Handler_CURIE",
    "bionumbers": "manubot.cite.curie.Handler_CURIE",
    "bioportal": "manubot.cite.curie.Handler_CURIE",
    "bioproject": "manubot.cite.curie.Handler_CURIE",
    "biosample": "manubot.cite.curie.Handler_CURIE",
    "biostudies": "manubot.cite.curie.Handler_CURIE",
    "biosystems": "manubot.cite.curie.Handler_CURIE",
    "biotools": "manubot.cite.curie.Handler_CURIE",
    "bitterdb.cpd": "manubot.cite.curie.Handler_CURIE",
    "bitterdb.rec": "manubot.cite.curie.Handler_CURIE",
    "bmrb": "manubot.cite.curie.Handler_CURIE",
    "bold.taxonomy": "manubot.cite.curie.Handler_CURIE",
    "brenda": "manubot.cite.curie.Handler_CURIE",
    "broad": "manubot.cite.curie.Handler_CURIE",
    "bto": "manubot.cite.curie.Handler_CURIE",
    "bugbase.expt": "manubot.cite.curie.Handler_CURIE",
    "bugbase.protocol": "manubot.cite.curie.Handler_CURIE",
    "bykdb": "manubot.cite.curie.Handler_CURIE",
    "cabri": "manubot.cite.curie.Handler_CURIE",
    "cadsr": "manubot.cite.curie.Handler_CURIE",
    "cameo": "manubot.cite.curie.Handler_CURIE",
    "caps": "manubot.cite.curie.Handler_CURIE",
    "cas": "manubot.cite.curie.Handler_CURIE",
    "cath": "manubot.cite.curie.Handler_CURIE",
    "cath.domain": "manubot.cite.curie.Handler_CURIE",
    "cath.superfamily": "manubot.cite.curie.Handler_CURIE",
    "cattleqtldb": "manubot.cite.curie.Handler_CURIE",
    "cazy": "manubot.cite.curie.Handler_CURIE",
    "cbioportal": "manubot.cite.curie.Handler_CURIE",
    "ccds": "manubot.cite.curie.Handler_CURIE",
    "cco": "manubot.cite.curie.Handler_CURIE",
    "cdd": "manubot.cite.curie.Handler_CURIE",
    "cdpd": "manubot.cite.curie.Handler_CURIE",
    "cellimage": "manubot.cite.curie.Handler_CURIE",
    "cellosaurus": "manubot.cite.curie.Handler_CURIE",
    "cgd": "manubot.cite.curie.Handler_CURIE",
    "cgsc": "manubot.cite.curie.Handler_CURIE",
    "charprot": "manubot.cite.curie.Handler_CURIE",
    "chebi": "manubot.cite.curie.Handler_CURIE",
    "chembl.compound": "manubot.cite.curie.Handler_CURIE",
    "chembl.target": "manubot.cite.curie.Handler_CURIE",
    "chemdb": "manubot.cite.curie.Handler_CURIE",
    "chemidplus": "manubot.cite.curie.Handler_CURIE",
    "chemspider": "manubot.cite.curie.Handler_CURIE",
    "chickenqtldb": "manubot.cite.curie.Handler_CURIE",
    "cl": "manubot.cite.curie.Handler_CURIE",
    "classyfire": "manubot.cite.curie.Handler_CURIE",
    "cldb": "manubot.cite.curie.Handler_CURIE",
    "clinicaltrials": "manubot.cite.curie.Handler_CURIE",
    "clinvar": "manubot.cite.curie.Handler_CURIE",
    "clinvar.record": "manubot.cite.curie.Handler_CURIE",
    "clinvar.submission": "manubot.cite.curie.Handler_CURIE",
    "combine.specifications": "manubot.cite.curie.Handler_CURIE",
    "complexportal": "manubot.cite.curie.Handler_CURIE",
    "comptox": "manubot.cite.curie.Handler_CURIE",
    "compulyeast": "manubot.cite.curie.Handler_CURIE",
    "conoserver": "manubot.cite.curie.Handler_CURIE",
    "coriell": "manubot.cite.curie.Handler_CURIE",
    "corum": "manubot.cite.curie.Handler_CURIE",
    "cosmic": "manubot.cite.curie.Handler_CURIE",
    "cpc": "manubot.cite.curie.Handler_CURIE",
    "crisprdb": "manubot.cite.curie.Handler_CURIE",
    "cryptodb": "manubot.cite.curie.Handler_CURIE",
    "csa": "manubot.cite.curie.Handler_CURIE",
    "cst": "manubot.cite.curie.Handler_CURIE",
    "cst.ab": "manubot.cite.curie.Handler_CURIE",
    "ctd.chemical": "manubot.cite.curie.Handler_CURIE",
    "ctd.disease": "manubot.cite.curie.Handler_CURIE",
    "ctd.gene": "manubot.cite.curie.Handler_CURIE",
    "cubedb": "manubot.cite.curie.Handler_CURIE",
    "d1id": "manubot.cite.curie.Handler_CURIE",
    "dailymed": "manubot.cite.curie.Handler_CURIE",
    "dandi": "manubot.cite.curie.Handler_CURIE",
    "darc": "manubot.cite.curie.Handler_CURIE",
    "dashr": "manubot.cite.curie.Handler_CURIE",
    "dashr.expression": "manubot.cite.curie.Handler_CURIE",
    "datf": "manubot.cite.curie.Handler_CURIE",
    "dbd": "manubot.cite.curie.Handler_CURIE",
    "dbest": "manubot.cite.curie.Handler_CURIE",
    "dbg2introns": "manubot.cite.curie.Handler_CURIE",
    "dbgap": "manubot.cite.curie.Handler_CURIE",
    "dbprobe": "manubot.cite.curie.Handler_CURIE",
    "dbsnp": "manubot.cite.curie.Handler_CURIE",
    "degradome": "manubot.cite.curie.Handler_CURIE",
    "depod": "manubot.cite.curie.Handler_CURIE",
    "dev.ga4ghdos": "manubot.cite.curie.Handler_CURIE",
    "dictybase.est": "manubot.cite.curie.Handler_CURIE",
    "dictybase.gene": "manubot.cite.curie.Handler_CURIE",
    "did": "manubot.cite.curie.Handler_CURIE",
    "dip": "manubot.cite.curie.Handler_CURIE",
    "disprot": "manubot.cite.curie.Handler_CURIE",
    "disprot.region": "manubot.cite.curie.Handler_CURIE",
    "dlxb": "manubot.cite.curie.Handler_CURIE",
    "dlxc": "manubot.cite.curie.Handler_CURIE",
    "doi": "manubot.cite.doi.Handler_DOI",
    "doid": "manubot.cite.curie.Handler_CURIE",
    "dommino": "manubot.cite.curie.Handler_CURIE",
    "door": "manubot.cite.curie.Handler_CURIE",
    "doqcs.model": "manubot.cite.curie.Handler_CURIE",
    "doqcs.pathway": "manubot.cite.curie.Handler_CURIE",
    "dpv": "manubot.cite.curie.Handler_CURIE",
    "dragondb.allele": "manubot.cite.curie.Handler_CURIE",
    "dragondb.dna": "manubot.cite.curie.Handler_CURIE",
    "dragondb.locus": "manubot.cite.curie.Handler_CURIE",
    "dragondb.protein": "manubot.cite.curie.Handler_CURIE",
    "drsc": "manubot.cite.curie.Handler_CURIE",
    "drugbank": "manubot.cite.curie.Handler_CURIE",
    "drugbankv4.target": "manubot.cite.curie.Handler_CURIE",
    "ec-code": "manubot.cite.curie.Handler_CURIE",
    "echobase": "manubot.cite.curie.Handler_CURIE",
    "eco": "manubot.cite.curie.Handler_CURIE",
    "ecogene": "manubot.cite.curie.Handler_CURIE",
    "ecoliwiki": "manubot.cite.curie.Handler_CURIE",
    "ecyano.entity": "manubot.cite.curie.Handler_CURIE",
    "ecyano.experiment": "manubot.cite.curie.Handler_CURIE",
    "ecyano.model": "manubot.cite.curie.Handler_CURIE",
    "ecyano.rule": "manubot.cite.curie.Handler_CURIE",
    "edam": "manubot.cite.curie.Handler_CURIE",
    "efo": "manubot.cite.curie.Handler_CURIE",
    "ega.dataset": "manubot.cite.curie.Handler_CURIE",
    "ega.study": "manubot.cite.curie.Handler_CURIE",
    "eggnog": "manubot.cite.curie.Handler_CURIE",
    "elm": "manubot.cite.curie.Handler_CURIE",
    "emdb": "manubot.cite.curie.Handler_CURIE",
    "ena.embl": "manubot.cite.curie.Handler_CURIE",
    "encode": "manubot.cite.curie.Handler_CURIE",
    "ensembl": "manubot.cite.curie.Handler_CURIE",
    "ensembl.bacteria": "manubot.cite.curie.Handler_CURIE",
    "ensembl.fungi": "manubot.cite.curie.Handler_CURIE",
    "ensembl.metazoa": "manubot.cite.curie.Handler_CURIE",
    "ensembl.plant": "manubot.cite.curie.Handler_CURIE",
    "ensembl.protist": "manubot.cite.curie.Handler_CURIE",
    "envipath": "manubot.cite.curie.Handler_CURIE",
    "envo": "manubot.cite.curie.Handler_CURIE",
    "eo": "manubot.cite.curie.Handler_CURIE",
    "epd": "manubot.cite.curie.Handler_CURIE",
    "erm": "manubot.cite.curie.Handler_CURIE",
    "erv": "manubot.cite.curie.Handler_CURIE",
    "eu89h": "manubot.cite.curie.Handler_CURIE",
    "euclinicaltrials": "manubot.cite.curie.Handler_CURIE",
    "exac.gene": "manubot.cite.curie.Handler_CURIE",
    "exac.transcript": "manubot.cite.curie.Handler_CURIE",
    "exac.variant": "manubot.cite.curie.Handler_CURIE",
    "facebase": "manubot.cite.curie.Handler_CURIE",
    "fairsharing": "manubot.cite.curie.Handler_CURIE",
    "fb": "manubot.cite.curie.Handler_CURIE",
    "fbol": "manubot.cite.curie.Handler_CURIE",
    "flowrepository": "manubot.cite.curie.Handler_CURIE",
    "fma": "manubot.cite.curie.Handler_CURIE",
    "foodb.compound": "manubot.cite.curie.Handler_CURIE",
    "foodon": "manubot.cite.curie.Handler_CURIE",
    "fplx": "manubot.cite.curie.Handler_CURIE",
    "fsnp": "manubot.cite.curie.Handler_CURIE",
    "funcbase.fly": "manubot.cite.curie.Handler_CURIE",
    "funcbase.human": "manubot.cite.curie.Handler_CURIE",
    "funcbase.mouse": "manubot.cite.curie.Handler_CURIE",
    "funcbase.yeast": "manubot.cite.curie.Handler_CURIE",
    "fungidb": "manubot.cite.curie.Handler_CURIE",
    "ga4ghdos": "manubot.cite.curie.Handler_CURIE",
    "gabi": "manubot.cite.curie.Handler_CURIE",
    "gcst": "manubot.cite.curie.Handler_CURIE",
    "gdc": "manubot.cite.curie.Handler_CURIE",
    "gdsc": "manubot.cite.curie.Handler_CURIE",
    "genatlas": "manubot.cite.curie.Handler_CURIE",
    "genecards": "manubot.cite.curie.Handler_CURIE",
    "genedb": "manubot.cite.curie.Handler_CURIE",
    "genefarm": "manubot.cite.curie.Handler_CURIE",
    "genetree": "manubot.cite.curie.Handler_CURIE",
    "genewiki": "manubot.cite.curie.Handler_CURIE",
    "genpept": "manubot.cite.curie.Handler_CURIE",
    "genprop": "manubot.cite.curie.Handler_CURIE",
    "geo": "manubot.cite.curie.Handler_CURIE",
    "giardiadb": "manubot.cite.curie.Handler_CURIE",
    "glida.gpcr": "manubot.cite.curie.Handler_CURIE",
    "glida.ligand": "manubot.cite.curie.Handler_CURIE",
    "glycoepitope": "manubot.cite.curie.Handler_CURIE",
    "glycomedb": "manubot.cite.curie.Handler_CURIE",
    "glytoucan": "manubot.cite.curie.Handler_CURIE",
    "gmd": "manubot.cite.curie.Handler_CURIE",
    "gmd.analyte": "manubot.cite.curie.Handler_CURIE",
    "gmd.gcms": "manubot.cite.curie.Handler_CURIE",
    "gmd.profile": "manubot.cite.curie.Handler_CURIE",
    "gmd.ref": "manubot.cite.curie.Handler_CURIE",
    "gnpis": "manubot.cite.curie.Handler_CURIE",
    "go": "manubot.cite.curie.Handler_CURIE",
    "go.ref": "manubot.cite.curie.Handler_CURIE",
    "goa": "manubot.cite.curie.Handler_CURIE",
    "gold": "manubot.cite.curie.Handler_CURIE",
    "gold.genome": "manubot.cite.curie.Handler_CURIE",
    "gold.meta": "manubot.cite.curie.Handler_CURIE",
    "google.patent": "manubot.cite.curie.Handler_CURIE",
    "gpcrdb": "manubot.cite.curie.Handler_CURIE",
    "gpmdb": "manubot.cite.curie.Handler_CURIE",
    "gramene.gene": "manubot.cite.curie.Handler_CURIE",
    "gramene.growthstage": "manubot.cite.curie.Handler_CURIE",
    "gramene.protein": "manubot.cite.curie.Handler_CURIE",
    "gramene.qtl": "manubot.cite.curie.Handler_CURIE",
    "gramene.taxonomy": "manubot.cite.curie.Handler_CURIE",
    "greengenes": "manubot.cite.curie.Handler_CURIE",
    "grid": "manubot.cite.curie.Handler_CURIE",
    "grin.taxonomy": "manubot.cite.curie.Handler_CURIE",
    "gro": "manubot.cite.curie.Handler_CURIE",
    "grsdb": "manubot.cite.curie.Handler_CURIE",
    "gtex": "manubot.cite.curie.Handler_CURIE",
    "gudmap": "manubot.cite.curie.Handler_CURIE",
    "gwascentral.marker": "manubot.cite.curie.Handler_CURIE",
    "gwascentral.phenotype": "manubot.cite.curie.Handler_CURIE",
    "gwascentral.study": "manubot.cite.curie.Handler_CURIE",
    "gxa.expt": "manubot.cite.curie.Handler_CURIE",
    "gxa.gene": "manubot.cite.curie.Handler_CURIE",
    "hamap": "manubot.cite.curie.Handler_CURIE",
    "hcvdb": "manubot.cite.curie.Handler_CURIE",
    "hdr": "manubot.cite.curie.Handler_CURIE",
    "hgmd": "manubot.cite.curie.Handler_CURIE",
    "hgnc": "manubot.cite.curie.Handler_CURIE",
    "hgnc.family": "manubot.cite.curie.Handler_CURIE",
    "hgnc.genefamily": "manubot.cite.curie.Handler_CURIE",
    "hgnc.symbol": "manubot.cite.curie.Handler_CURIE",
    "hinv.locus": "manubot.cite.curie.Handler_CURIE",
    "hinv.protein": "manubot.cite.curie.Handler_CURIE",
    "hinv.transcript": "manubot.cite.curie.Handler_CURIE",
    "hmdb": "manubot.cite.curie.Handler_CURIE",
    "hogenom": "manubot.cite.curie.Handler_CURIE",
    "homd.seq": "manubot.cite.curie.Handler_CURIE",
    "homd.taxon": "manubot.cite.curie.Handler_CURIE",
    "homologene": "manubot.cite.curie.Handler_CURIE",
    "hovergen": "manubot.cite.curie.Handler_CURIE",
    "hp": "manubot.cite.curie.Handler_CURIE",
    "hpa": "manubot.cite.curie.Handler_CURIE",
    "hpm.peptide": "manubot.cite.curie.Handler_CURIE",
    "hpm.protein": "manubot.cite.curie.Handler_CURIE",
    "hprd": "manubot.cite.curie.Handler_CURIE",
    "hpscreg": "manubot.cite.curie.Handler_CURIE",
    "hssp": "manubot.cite.curie.Handler_CURIE",
    "http": "manubot.cite.url.Handler_URL",
    "https": "manubot.cite.url.Handler_URL",
    "huge": "manubot.cite.curie.Handler_CURIE",
    "iao": "manubot.cite.curie.Handler_CURIE",
    "icd": "manubot.cite.curie.Handler_CURIE",
    "iceberg.element": "manubot.cite.curie.Handler_CURIE",
    "iceberg.family": "manubot.cite.curie.Handler_CURIE",
    "ideal": "manubot.cite.curie.Handler_CURIE",
    "ido": "manubot.cite.curie.Handler_CURIE",
    "idoo": "manubot.cite.curie.Handler_CURIE",
    "idot": "manubot.cite.curie.Handler_CURIE",
    "idr": "manubot.cite.curie.Handler_CURIE",
    "imex": "manubot.cite.curie.Handler_CURIE",
    "img.gene": "manubot.cite.curie.Handler_CURIE",
    "img.taxon": "manubot.cite.curie.Handler_CURIE",
    "imgt.hla": "manubot.cite.curie.Handler_CURIE",
    "imgt.ligm": "manubot.cite.curie.Handler_CURIE",
    "inchi": "manubot.cite.curie.Handler_CURIE",
    "inchikey": "manubot.cite.curie.Handler_CURIE",
    "insdc": "manubot.cite.curie.Handler_CURIE",
    "insdc.cds": "manubot.cite.curie.Handler_CURIE",
    "insdc.gca": "manubot.cite.curie.Handler_CURIE",
    "insdc.sra": "manubot.cite.curie.Handler_CURIE",
    "intact": "manubot.cite.curie.Handler_CURIE",
    "intact.molecule": "manubot.cite.curie.Handler_CURIE",
    "interpro": "manubot.cite.curie.Handler_CURIE",
    "ird.segment": "manubot.cite.curie.Handler_CURIE",
    "irefweb": "manubot.cite.curie.Handler_CURIE",
    "isbn": "manubot.cite.isbn.Handler_ISBN",
    "isfinder": "manubot.cite.curie.Handler_CURIE",
    "isni": "manubot.cite.curie.Handler_CURIE",
    "issn": "manubot.cite.curie.Handler_CURIE",
    "iuphar.family": "manubot.cite.curie.Handler_CURIE",
    "iuphar.ligand": "manubot.cite.curie.Handler_CURIE",
    "iuphar.receptor": "manubot.cite.curie.Handler_CURIE",
    "jaxmice": "manubot.cite.curie.Handler_CURIE",
    "jcggdb": "manubot.cite.curie.Handler_CURIE",
    "jcm": "manubot.cite.curie.Handler_CURIE",
    "jcsd": "manubot.cite.curie.Handler_CURIE",
    "jstor": "manubot.cite.curie.Handler_CURIE",
    "jws": "manubot.cite.curie.Handler_CURIE",
    "kaggle": "manubot.cite.curie.Handler_CURIE",
    "kegg": "manubot.cite.curie.Handler_CURIE",
    "kegg.compound": "manubot.cite.curie.Handler_CURIE",
    "kegg.disease": "manubot.cite.curie.Handler_CURIE",
    "kegg.drug": "manubot.cite.curie.Handler_CURIE",
    "kegg.environ": "manubot.cite.curie.Handler_CURIE",
    "kegg.genes": "manubot.cite.curie.Handler_CURIE",
    "kegg.genome": "manubot.cite.curie.Handler_CURIE",
    "kegg.glycan": "manubot.cite.curie.Handler_CURIE",
    "kegg.metagenome": "manubot.cite.curie.Handler_CURIE",
    "kegg.module": "manubot.cite.curie.Handler_CURIE",
    "kegg.orthology": "manubot.cite.curie.Handler_CURIE",
    "kegg.pathway": "manubot.cite.curie.Handler_CURIE",
    "kegg.reaction": "manubot.cite.curie.Handler_CURIE",
    "knapsack": "manubot.cite.curie.Handler_CURIE",
    "lei": "manubot.cite.curie.Handler_CURIE",
    "lgic": "manubot.cite.curie.Handler_CURIE",
    "licebase": "manubot.cite.curie.Handler_CURIE",
    "ligandbook": "manubot.cite.curie.Handler_CURIE",
    "ligandbox": "manubot.cite.curie.Handler_CURIE",
    "ligandexpo": "manubot.cite.curie.Handler_CURIE",
    "lincs.cell": "manubot.cite.curie.Handler_CURIE",
    "lincs.data": "manubot.cite.curie.Handler_CURIE",
    "lincs.protein": "manubot.cite.curie.Handler_CURIE",
    "lincs.smallmolecule": "manubot.cite.curie.Handler_CURIE",
    "lipidbank": "manubot.cite.curie.Handler_CURIE",
    "lipidmaps": "manubot.cite.curie.Handler_CURIE",
    "lrg": "manubot.cite.curie.Handler_CURIE",
    "ma": "manubot.cite.curie.Handler_CURIE",
    "macie": "manubot.cite.curie.Handler_CURIE",
    "maizegdb.locus": "manubot.cite.curie.Handler_CURIE",
    "mamo": "manubot.cite.curie.Handler_CURIE",
    "massbank": "manubot.cite.curie.Handler_CURIE",
    "massive": "manubot.cite.curie.Handler_CURIE",
    "matrixdb.association": "manubot.cite.curie.Handler_CURIE",
    "mdm": "manubot.cite.curie.Handler_CURIE",
    "meddra": "manubot.cite.curie.Handler_CURIE",
    "medgen": "manubot.cite.curie.Handler_CURIE",
    "medlineplus": "manubot.cite.curie.Handler_CURIE",
    "merops": "manubot.cite.curie.Handler_CURIE",
    "merops.family": "manubot.cite.curie.Handler_CURIE",
    "merops.inhibitor": "manubot.cite.curie.Handler_CURIE",
    "mesh": "manubot.cite.curie.Handler_CURIE",
    "mesh.2012": "manubot.cite.curie.Handler_CURIE",
    "mesh.2013": "manubot.cite.curie.Handler_CURIE",
    "metabolights": "manubot.cite.curie.Handler_CURIE",
    "metacyc.compound": "manubot.cite.curie.Handler_CURIE",
    "metacyc.reaction": "manubot.cite.curie.Handler_CURIE",
    "metanetx.chemical": "manubot.cite.curie.Handler_CURIE",
    "metanetx.compartment": "manubot.cite.curie.Handler_CURIE",
    "metanetx.reaction": "manubot.cite.curie.Handler_CURIE",
    "metlin": "manubot.cite.curie.Handler_CURIE",
    "mex": "manubot.cite.curie.Handler_CURIE",
    "mge": "manubot.cite.curie.Handler_CURIE",
    "mgi": "manubot.cite.curie.Handler_CURIE",
    "mgnify.proj": "manubot.cite.curie.Handler_CURIE",
    "mgnify.samp": "manubot.cite.curie.Handler_CURIE",
    "mi": "manubot.cite.curie.Handler_CURIE",
    "microscope": "manubot.cite.curie.Handler_CURIE",
    "microsporidia": "manubot.cite.curie.Handler_CURIE",
    "mim": "manubot.cite.curie.Handler_CURIE",
    "mimodb": "manubot.cite.curie.Handler_CURIE",
    "minid": "manubot.cite.curie.Handler_CURIE",
    "minid.test": "manubot.cite.curie.Handler_CURIE",
    "mint": "manubot.cite.curie.Handler_CURIE",
    "mipmod": "manubot.cite.curie.Handler_CURIE",
    "mir": "manubot.cite.curie.Handler_CURIE",
    "mirbase": "manubot.cite.curie.Handler_CURIE",
    "mirbase.mature": "manubot.cite.curie.Handler_CURIE",
    "mirex": "manubot.cite.curie.Handler_CURIE",
    "miriam.collection": "manubot.cite.curie.Handler_CURIE",
    "miriam.resource": "manubot.cite.curie.Handler_CURIE",
    "mirnest": "manubot.cite.curie.Handler_CURIE",
    "mirtarbase": "manubot.cite.curie.Handler_CURIE",
    "mmdb": "manubot.cite.curie.Handler_CURIE",
    "mmmp:biomaps": "manubot.cite.curie.Handler_CURIE",
    "mmp.cat": "manubot.cite.curie.Handler_CURIE",
    "mmp.db": "manubot.cite.curie.Handler_CURIE",
    "mmp.fun": "manubot.cite.curie.Handler_CURIE",
    "mmp.ref": "manubot.cite.curie.Handler_CURIE",
    "mmrrc": "manubot.cite.curie.Handler_CURIE",
    "mo": "manubot.cite.curie.Handler_CURIE",
    "mobidb": "manubot.cite.curie.Handler_CURIE",
    "mod": "manubot.cite.curie.Handler_CURIE",
    "modeldb": "manubot.cite.curie.Handler_CURIE",
    "molbase": "manubot.cite.curie.Handler_CURIE",
    "mp": "manubot.cite.curie.Handler_CURIE",
    "mpid": "manubot.cite.curie.Handler_CURIE",
    "ms": "manubot.cite.curie.Handler_CURIE",
    "multicellds.cell_line": "manubot.cite.curie.Handler_CURIE",
    "multicellds.collection": "manubot.cite.curie.Handler_CURIE",
    "multicellds.snapshot": "manubot.cite.curie.Handler_CURIE",
    "mw.project": "manubot.cite.curie.Handler_CURIE",
    "mw.study": "manubot.cite.curie.Handler_CURIE",
    "myco.lepra": "manubot.cite.curie.Handler_CURIE",
    "myco.marinum": "manubot.cite.curie.Handler_CURIE",
    "myco.smeg": "manubot.cite.curie.Handler_CURIE",
    "myco.tuber": "manubot.cite.curie.Handler_CURIE",
    "mycobank": "manubot.cite.curie.Handler_CURIE",
    "mzspec": "manubot.cite.curie.Handler_CURIE",
    "napdi": "manubot.cite.curie.Handler_CURIE",
    "napp": "manubot.cite.curie.Handler_CURIE",
    "narcis": "manubot.cite.curie.Handler_CURIE",
    "nasc": "manubot.cite.curie.Handler_CURIE",
    "nbn": "manubot.cite.curie.Handler_CURIE",
    "nbrc": "manubot.cite.curie.Handler_CURIE",
    "ncbigene": "manubot.cite.curie.Handler_CURIE",
    "ncbiprotein": "manubot.cite.curie.Handler_CURIE",
    "ncim": "manubot.cite.curie.Handler_CURIE",
    "ncit": "manubot.cite.curie.Handler_CURIE",
    "ndc": "manubot.cite.curie.Handler_CURIE",
    "nemo": "manubot.cite.curie.Handler_CURIE",
    "neurolex": "manubot.cite.curie.Handler_CURIE",
    "neuromorpho": "manubot.cite.curie.Handler_CURIE",
    "neurondb": "manubot.cite.curie.Handler_CURIE",
    "neurovault.collection": "manubot.cite.curie.Handler_CURIE",
    "neurovault.image": "manubot.cite.curie.Handler_CURIE",
    "nextdb": "manubot.cite.curie.Handler_CURIE",
    "nextprot": "manubot.cite.curie.Handler_CURIE",
    "ngl": "manubot.cite.curie.Handler_CURIE",
    "niaest": "manubot.cite.curie.Handler_CURIE",
    "nmr": "manubot.cite.curie.Handler_CURIE",
    "nmrshiftdb2": "manubot.cite.curie.Handler_CURIE",
    "noncodev3": "manubot.cite.curie.Handler_CURIE",
    "noncodev4.gene": "manubot.cite.curie.Handler_CURIE",
    "noncodev4.rna": "manubot.cite.curie.Handler_CURIE",
    "norine": "manubot.cite.curie.Handler_CURIE",
    "nuclearbd": "manubot.cite.curie.Handler_CURIE",
    "obi": "manubot.cite.curie.Handler_CURIE",
    "occ": "manubot.cite.curie.Handler_CURIE",
    "oci": "manubot.cite.curie.Handler_CURIE",
    "ocid": "manubot.cite.curie.Handler_CURIE",
    "oclc": "manubot.cite.curie.Handler_CURIE",
    "odor": "manubot.cite.curie.Handler_CURIE",
    "oid": "manubot.cite.curie.Handler_CURIE",
    "oma.grp": "manubot.cite.curie.Handler_CURIE",
    "oma.protein": "manubot.cite.curie.Handler_CURIE",
    "omia": "manubot.cite.curie.Handler_CURIE",
    "omit": "manubot.cite.curie.Handler_CURIE",
    "opb": "manubot.cite.curie.Handler_CURIE",
    "opm": "manubot.cite.curie.Handler_CURIE",
    "orcid": "manubot.cite.curie.Handler_CURIE",
    "ordb": "manubot.cite.curie.Handler_CURIE",
    "oridb.sacch": "manubot.cite.curie.Handler_CURIE",
    "oridb.schizo": "manubot.cite.curie.Handler_CURIE",
    "orphanet": "manubot.cite.curie.Handler_CURIE",
    "orphanet.ordo": "manubot.cite.curie.Handler_CURIE",
    "orthodb": "manubot.cite.curie.Handler_CURIE",
    "oryzabase.gene": "manubot.cite.curie.Handler_CURIE",
    "oryzabase.mutant": "manubot.cite.curie.Handler_CURIE",
    "oryzabase.reference": "manubot.cite.curie.Handler_CURIE",
    "oryzabase.stage": "manubot.cite.curie.Handler_CURIE",
    "oryzabase.strain": "manubot.cite.curie.Handler_CURIE",
    "otl": "manubot.cite.curie.Handler_CURIE",
    "p3db.protein": "manubot.cite.curie.Handler_CURIE",
    "p3db.site": "manubot.cite.curie.Handler_CURIE",
    "paleodb": "manubot.cite.curie.Handler_CURIE",
    "panther.family": "manubot.cite.curie.Handler_CURIE",
    "panther.node": "manubot.cite.curie.Handler_CURIE",
    "panther.pathway": "manubot.cite.curie.Handler_CURIE",
    "panther.pthcmp": "manubot.cite.curie.Handler_CURIE",
    "pass2": "manubot.cite.curie.Handler_CURIE",
    "pathwaycommons": "manubot.cite.curie.Handler_CURIE",
    "pato": "manubot.cite.curie.Handler_CURIE",
    "paxdb.organism": "manubot.cite.curie.Handler_CURIE",
    "paxdb.protein": "manubot.cite.curie.Handler_CURIE",
    "pazar": "manubot.cite.curie.Handler_CURIE",
    "pdb": "manubot.cite.curie.Handler_CURIE",
    "pdb-ccd": "manubot.cite.curie.Handler_CURIE",
    "pdb.ligand": "manubot.cite.curie.Handler_CURIE",
    "peptideatlas": "manubot.cite.curie.Handler_CURIE",
    "peptideatlas.dataset": "manubot.cite.curie.Handler_CURIE",
    "peroxibase": "manubot.cite.curie.Handler_CURIE",
    "pfam": "manubot.cite.curie.Handler_CURIE",
    "pgs": "manubot.cite.curie.Handler_CURIE",
    "pgx": "manubot.cite.curie.Handler_CURIE",
    "pharmgkb.disease": "manubot.cite.curie.Handler_CURIE",
    "pharmgkb.drug": "manubot.cite.curie.Handler_CURIE",
    "pharmgkb.gene": "manubot.cite.curie.Handler_CURIE",
    "pharmgkb.pathways": "manubot.cite.curie.Handler_CURIE",
    "phenolexplorer": "manubot.cite.curie.Handler_CURIE",
    "phosphopoint.kinase": "manubot.cite.curie.Handler_CURIE",
    "phosphopoint.protein": "manubot.cite.curie.Handler_CURIE",
    "phosphosite.protein": "manubot.cite.curie.Handler_CURIE",
    "phosphosite.residue": "manubot.cite.curie.Handler_CURIE",
    "phylomedb": "manubot.cite.curie.Handler_CURIE",
    "phytozome.locus": "manubot.cite.curie.Handler_CURIE",
    "pid.pathway": "manubot.cite.curie.Handler_CURIE",
    "pigqtldb": "manubot.cite.curie.Handler_CURIE",
    "pina": "manubot.cite.curie.Handler_CURIE",
    "piroplasma": "manubot.cite.curie.Handler_CURIE",
    "pirsf": "manubot.cite.curie.Handler_CURIE",
    "planttfdb": "manubot.cite.curie.Handler_CURIE",
    "plasmodb": "manubot.cite.curie.Handler_CURIE",
    "pmap.cutdb": "manubot.cite.curie.Handler_CURIE",
    "pmap.substratedb": "manubot.cite.curie.Handler_CURIE",
    "pmc": "manubot.cite.pubmed.Handler_PMC",
    "pmcid": "manubot.cite.pubmed.Handler_PMC",
    "pmdb": "manubot.cite.curie.Handler_CURIE",
    "pmid": "manubot.cite.pubmed.Handler_PubMed",
    "pmp": "manubot.cite.curie.Handler_CURIE",
    "po": "manubot.cite.curie.Handler_CURIE",
    "pocketome": "manubot.cite.curie.Handler_CURIE",
    "polbase": "manubot.cite.curie.Handler_CURIE",
    "pombase": "manubot.cite.curie.Handler_CURIE",
    "pr": "manubot.cite.curie.Handler_CURIE",
    "pride": "manubot.cite.curie.Handler_CURIE",
    "pride.project": "manubot.cite.curie.Handler_CURIE",
    "prints": "manubot.cite.curie.Handler_CURIE",
    "probonto": "manubot.cite.curie.Handler_CURIE",
    "prodom": "manubot.cite.curie.Handler_CURIE",
    "proglyc": "manubot.cite.curie.Handler_CURIE",
    "prosite": "manubot.cite.curie.Handler_CURIE",
    "protclustdb": "manubot.cite.curie.Handler_CURIE",
    "proteomicsdb.peptide": "manubot.cite.curie.Handler_CURIE",
    "proteomicsdb.protein": "manubot.cite.curie.Handler_CURIE",
    "protonet.cluster": "manubot.cite.curie.Handler_CURIE",
    "protonet.proteincard": "manubot.cite.curie.Handler_CURIE",
    "pscdb": "manubot.cite.curie.Handler_CURIE",
    "pseudomonas": "manubot.cite.curie.Handler_CURIE",
    "psimi": "manubot.cite.curie.Handler_CURIE",
    "psipar": "manubot.cite.curie.Handler_CURIE",
    "pubchem.bioassay": "manubot.cite.curie.Handler_CURIE",
    "pubchem.compound": "manubot.cite.curie.Handler_CURIE",
    "pubchem.substance": "manubot.cite.curie.Handler_CURIE",
    "pubmed": "manubot.cite.pubmed.Handler_PubMed",
    "pw": "manubot.cite.curie.Handler_CURIE",
    "px": "manubot.cite.curie.Handler_CURIE",
    "rapdb.locus": "manubot.cite.curie.Handler_CURIE",
    "rapdb.transcript": "manubot.cite.curie.Handler_CURIE",
    "rbk": "manubot.cite.curie.Handler_CURIE",
    "reactome": "manubot.cite.curie.Handler_CURIE",
    "rebase": "manubot.cite.curie.Handler_CURIE",
    "refseq": "manubot.cite.curie.Handler_CURIE",
    "resid": "manubot.cite.curie.Handler_CURIE",
    "rfam": "manubot.cite.curie.Handler_CURIE",
    "rgd": "manubot.cite.curie.Handler_CURIE",
    "rgd.qtl": "manubot.cite.curie.Handler_CURIE",
    "rgd.strain": "manubot.cite.curie.Handler_CURIE",
    "rhea": "manubot.cite.curie.Handler_CURIE",
    "ricegap": "manubot.cite.curie.Handler_CURIE",
    "ricenetdb.compound": "manubot.cite.curie.Handler_CURIE",
    "ricenetdb.gene": "manubot.cite.curie.Handler_CURIE",
    "ricenetdb.mirna": "manubot.cite.curie.Handler_CURIE",
    "ricenetdb.protein": "manubot.cite.curie.Handler_CURIE",
    "ricenetdb.reaction": "manubot.cite.curie.Handler_CURIE",
    "rnacentral": "manubot.cite.curie.Handler_CURIE",
    "rnamods": "manubot.cite.curie.Handler_CURIE",
    "ro": "manubot.cite.curie.Handler_CURIE",
    "rouge": "manubot.cite.curie.Handler_CURIE",
    "rrid": "manubot.cite.curie.Handler_CURIE",
    "sabiork.compound": "manubot.cite.curie.Handler_CURIE",
    "sabiork.ec": "manubot.cite.curie.Handler_CURIE",
    "sabiork.kineticrecord": "manubot.cite.curie.Handler_CURIE",
    "sabiork.reaction": "manubot.cite.curie.Handler_CURIE",
    "sasbdb": "manubot.cite.curie.Handler_CURIE",
    "sbo": "manubot.cite.curie.Handler_CURIE",
    "scop": "manubot.cite.curie.Handler_CURIE",
    "scretf": "manubot.cite.curie.Handler_CURIE",
    "sdbs": "manubot.cite.curie.Handler_CURIE",
    "seed": "manubot.cite.curie.Handler_CURIE",
    "seed.compound": "manubot.cite.curie.Handler_CURIE",
    "seed.reaction": "manubot.cite.curie.Handler_CURIE",
    "sgd": "manubot.cite.curie.Handler_CURIE",
    "sgd.pathways": "manubot.cite.curie.Handler_CURIE",
    "sgn": "manubot.cite.curie.Handler_CURIE",
    "sheepqtldb": "manubot.cite.curie.Handler_CURIE",
    "shortdoi": "manubot.cite.doi.Handler_DOI",
    "sider.drug": "manubot.cite.curie.Handler_CURIE",
    "sider.effect": "manubot.cite.curie.Handler_CURIE",
    "signaling-gateway": "manubot.cite.curie.Handler_CURIE",
    "sisu": "manubot.cite.curie.Handler_CURIE",
    "sitex": "manubot.cite.curie.Handler_CURIE",
    "slm": "manubot.cite.curie.Handler_CURIE",
    "smart": "manubot.cite.curie.Handler_CURIE",
    "smpdb": "manubot.cite.curie.Handler_CURIE",
    "snomedct": "manubot.cite.curie.Handler_CURIE",
    "snp2tfbs": "manubot.cite.curie.Handler_CURIE",
    "so": "manubot.cite.curie.Handler_CURIE",
    "soybase": "manubot.cite.curie.Handler_CURIE",
    "spdx": "manubot.cite.curie.Handler_CURIE",
    "spike.map": "manubot.cite.curie.Handler_CURIE",
    "splash": "manubot.cite.curie.Handler_CURIE",
    "spp": "manubot.cite.curie.Handler_CURIE",
    "stap": "manubot.cite.curie.Handler_CURIE",
    "stitch": "manubot.cite.curie.Handler_CURIE",
    "storedb": "manubot.cite.curie.Handler_CURIE",
    "string": "manubot.cite.curie.Handler_CURIE",
    "subtilist": "manubot.cite.curie.Handler_CURIE",
    "subtiwiki": "manubot.cite.curie.Handler_CURIE",
    "sugarbind": "manubot.cite.curie.Handler_CURIE",
    "supfam": "manubot.cite.curie.Handler_CURIE",
    "swh": "manubot.cite.curie.Handler_CURIE",
    "swiss-model": "manubot.cite.curie.Handler_CURIE",
    "swissregulon": "manubot.cite.curie.Handler_CURIE",
    "t3db": "manubot.cite.curie.Handler_CURIE",
    "tair.gene": "manubot.cite.curie.Handler_CURIE",
    "tair.locus": "manubot.cite.curie.Handler_CURIE",
    "tair.protein": "manubot.cite.curie.Handler_CURIE",
    "tarbase": "manubot.cite.curie.Handler_CURIE",
    "taxonomy": "manubot.cite.curie.Handler_CURIE",
    "tcdb": "manubot.cite.curie.Handler_CURIE",
    "tgd": "manubot.cite.curie.Handler_CURIE",
    "tigrfam": "manubot.cite.curie.Handler_CURIE",
    "tissuelist": "manubot.cite.curie.Handler_CURIE",
    "tol": "manubot.cite.curie.Handler_CURIE",
    "topdb": "manubot.cite.curie.Handler_CURIE",
    "topfind": "manubot.cite.curie.Handler_CURIE",
    "toxoplasma": "manubot.cite.curie.Handler_CURIE",
    "transyt": "manubot.cite.curie.Handler_CURIE",
    "treebase": "manubot.cite.curie.Handler_CURIE",
    "treefam": "manubot.cite.curie.Handler_CURIE",
    "trichdb": "manubot.cite.curie.Handler_CURIE",
    "tritrypdb": "manubot.cite.curie.Handler_CURIE",
    "ttd.drug": "manubot.cite.curie.Handler_CURIE",
    "ttd.target": "manubot.cite.curie.Handler_CURIE",
    "uberon": "manubot.cite.curie.Handler_CURIE",
    "ubio.namebank": "manubot.cite.curie.Handler_CURIE",
    "umbbd.compound": "manubot.cite.curie.Handler_CURIE",
    "umbbd.enzyme": "manubot.cite.curie.Handler_CURIE",
    "umbbd.pathway": "manubot.cite.curie.Handler_CURIE",
    "umbbd.reaction": "manubot.cite.curie.Handler_CURIE",
    "umbbd.rule": "manubot.cite.curie.Handler_CURIE",
    "umls": "manubot.cite.curie.Handler_CURIE",
    "unigene": "manubot.cite.curie.Handler_CURIE",
    "unii": "manubot.cite.curie.Handler_CURIE",
    "unimod": "manubot.cite.curie.Handler_CURIE",
    "uniparc": "manubot.cite.curie.Handler_CURIE",
    "unipathway.compound": "manubot.cite.curie.Handler_CURIE",
    "unipathway.reaction": "manubot.cite.curie.Handler_CURIE",
    "uniprot": "manubot.cite.curie.Handler_CURIE",
    "uniprot.isoform": "manubot.cite.curie.Handler_CURIE",
    "unists": "manubot.cite.curie.Handler_CURIE",
    "unite": "manubot.cite.curie.Handler_CURIE",
    "uo": "manubot.cite.curie.Handler_CURIE",
    "url": "manubot.cite.url.Handler_URL",
    "uspto": "manubot.cite.curie.Handler_CURIE",
    "validatordb": "manubot.cite.curie.Handler_CURIE",
    "vario": "manubot.cite.curie.Handler_CURIE",
    "vbase2": "manubot.cite.curie.Handler_CURIE",
    "vbrc": "manubot.cite.curie.Handler_CURIE",
    "vectorbase": "manubot.cite.curie.Handler_CURIE",
    "vegbank": "manubot.cite.curie.Handler_CURIE",
    "vfb": "manubot.cite.curie.Handler_CURIE",
    "vfdb.gene": "manubot.cite.curie.Handler_CURIE",
    "vfdb.genus": "manubot.cite.curie.Handler_CURIE",
    "vgnc": "manubot.cite.curie.Handler_CURIE",
    "viaf": "manubot.cite.curie.Handler_CURIE",
    "vipr": "manubot.cite.curie.Handler_CURIE",
    "viralzone": "manubot.cite.curie.Handler_CURIE",
    "virsirna": "manubot.cite.curie.Handler_CURIE",
    "vmhmetabolite": "manubot.cite.curie.Handler_CURIE",
    "vmhreaction": "manubot.cite.curie.Handler_CURIE",
    "wb": "manubot.cite.curie.Handler_CURIE",
    "wb.rnai": "manubot.cite.curie.Handler_CURIE",
    "wikidata": "manubot.cite.wikidata.Handler_Wikidata",
    "wikigenes": "manubot.cite.curie.Handler_CURIE",
    "wikipathways": "manubot.cite.curie.Handler_CURIE",
    "wikipedia.en": "manubot.cite.curie.Handler_CURIE",
    "worfdb": "manubot.cite.curie.Handler_CURIE",
    "wormpep": "manubot.cite.curie.Handler_CURIE",
    "worms": "manubot.cite.curie.Handler_CURIE",
    "xenbase": "manubot.cite.curie.Handler_CURIE",
    "ydpm": "manubot.cite.curie.Handler_CURIE",
    "yeastintron": "manubot.cite.curie.Handler_CURIE",
    "yetfasco": "manubot.cite.curie.Handler_CURIE",
    "yid": "manubot.cite.curie.Handler_CURIE",
    "yrcpdr": "manubot.cite.curie.Handler_CURIE",
    "zfin": "manubot.cite.curie.Handler_CURIE",
    "zinc": "manubot.cite.curie.Handler_CURIE",
}
