"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.trimFromStart = exports.calculateFunctionHash = void 0;
const crypto = require("crypto");
const core_1 = require("@aws-cdk/core");
const cx_api_1 = require("@aws-cdk/cx-api");
const function_1 = require("./function");
function calculateFunctionHash(fn) {
    const stack = core_1.Stack.of(fn);
    const functionResource = fn.node.defaultChild;
    // render the cloudformation resource from this function
    const config = stack.resolve(functionResource._toCloudFormation());
    // config is of the shape: { Resources: { LogicalId: { Type: 'Function', Properties: { ... } }}}
    const resources = config.Resources;
    const resourceKeys = Object.keys(resources);
    if (resourceKeys.length !== 1) {
        throw new Error(`Expected one rendered CloudFormation resource but found ${resourceKeys.length}`);
    }
    const logicalId = resourceKeys[0];
    const properties = resources[logicalId].Properties;
    let stringifiedConfig;
    if (core_1.FeatureFlags.of(fn).isEnabled(cx_api_1.LAMBDA_RECOGNIZE_VERSION_PROPS)) {
        const updatedProps = sortProperties(filterUsefulKeys(properties));
        stringifiedConfig = JSON.stringify(updatedProps);
    }
    else {
        const sorted = sortProperties(properties);
        config.Resources[logicalId].Properties = sorted;
        stringifiedConfig = JSON.stringify(config);
    }
    const hash = crypto.createHash('md5');
    hash.update(stringifiedConfig);
    return hash.digest('hex');
}
exports.calculateFunctionHash = calculateFunctionHash;
function trimFromStart(s, maxLength) {
    const desiredLength = Math.min(maxLength, s.length);
    const newStart = s.length - desiredLength;
    return s.substring(newStart);
}
exports.trimFromStart = trimFromStart;
/*
 * The list of properties found in CfnFunction (or AWS::Lambda::Function).
 * They are classified as "locked" to a Function Version or not.
 * When a property is locked, any change to that property will not take effect on previously created Versions.
 * Instead, a new Version must be generated for the change to take effect.
 * Similarly, if a property that's not locked to a Version is modified, a new Version
 * must not be generated.
 *
 * Adding a new property to this list - If the property is part of the UpdateFunctionConfiguration
 * API, then it must be classified as true, otherwise false.
 * See https://docs.aws.amazon.com/lambda/latest/dg/API_UpdateFunctionConfiguration.html
 */
const VERSION_LOCKED = {
    // locked to the version
    Code: true,
    DeadLetterConfig: true,
    Description: true,
    Environment: true,
    FileSystemConfigs: true,
    FunctionName: true,
    Handler: true,
    ImageConfig: true,
    KmsKeyArn: true,
    Layers: true,
    MemorySize: true,
    PackageType: true,
    Role: true,
    Runtime: true,
    Timeout: true,
    TracingConfig: true,
    VpcConfig: true,
    // not locked to the version
    CodeSigningConfigArn: false,
    ReservedConcurrentExecutions: false,
    Tags: false,
};
function filterUsefulKeys(properties) {
    const versionProps = { ...VERSION_LOCKED, ...function_1.Function._VER_PROPS };
    const unclassified = Object.entries(properties)
        .filter(([k, v]) => v != null && !Object.keys(versionProps).includes(k))
        .map(([k, _]) => k);
    if (unclassified.length > 0) {
        throw new Error(`The following properties are not recognized as version properties: [${unclassified}].`
            + ' See the README of the aws-lambda module to learn more about this and to fix it.');
    }
    const notLocked = Object.entries(versionProps).filter(([_, v]) => !v).map(([k, _]) => k);
    notLocked.forEach(p => delete properties[p]);
    const ret = {};
    Object.entries(properties).filter(([k, _]) => versionProps[k]).forEach(([k, v]) => ret[k] = v);
    return ret;
}
function sortProperties(properties) {
    const ret = {};
    // We take all required properties in the order that they were historically,
    // to make sure the hash we calculate is stable.
    // There cannot be more required properties added in the future,
    // as that would be a backwards-incompatible change.
    const requiredProperties = ['Code', 'Handler', 'Role', 'Runtime'];
    for (const requiredProperty of requiredProperties) {
        ret[requiredProperty] = properties[requiredProperty];
    }
    // then, add all of the non-required properties,
    // in the original order
    for (const property of Object.keys(properties)) {
        if (requiredProperties.indexOf(property) === -1) {
            ret[property] = properties[property];
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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