# -*- coding: utf-8 -*-
from __future__ import division
from __future__ import print_function
from __future__ import absolute_import
from __future__ import unicode_literals

import io
import pytest
from mock import Mock
from tabulator import Stream, exceptions
from tabulator.parsers.ods import ODSParser
BASE_URL = 'https://raw.githubusercontent.com/okfn/tabulator-py/master/%s'


# Stream

def test_stream_ods():
    with Stream('data/table.ods', headers=1) as stream:
        assert stream.headers == ['id', 'name']
        assert stream.read(keyed=True) == [
            {'id': 1.0, 'name': 'english'},
            {'id': 2.0, 'name': '中国人'},
        ]


def test_stream_ods_remote():
    source = BASE_URL % 'data/table.ods'
    with Stream(source) as stream:
        assert stream.read() == [['id', 'name'], [1.0, 'english'], [2.0, '中国人']]


def test_stream_ods_sheet_by_index():
    with Stream('data/table.ods', sheet=1) as stream:
        assert stream.read() == [['id', 'name'], [1.0, 'english'], [2.0, '中国人']]


def test_stream_ods_sheet_by_index_not_existent():
    with pytest.raises(exceptions.SourceError) as excinfo:
        Stream('data/table.ods', sheet=3).open()
    assert 'sheet "3"' in str(excinfo.value)


def test_stream_ods_sheet_by_name():
    with Stream('data/table.ods', sheet='Лист1') as stream:
        assert stream.read() == [['id', 'name'], [1.0, 'english'], [2.0, '中国人']]


def test_stream_ods_sheet_by_index_not_existent():
    with pytest.raises(exceptions.SourceError) as excinfo:
        Stream('data/table.ods', sheet='not-existent').open()
    assert 'sheet "not-existent"' in str(excinfo.value)


# Parser

def test_parser_ods():

    source = 'data/table.ods'
    encoding = None
    loader = Mock()
    loader.load = Mock(return_value=io.open(source, 'rb'))
    parser = ODSParser(loader)

    assert parser.closed
    parser.open(source, encoding=encoding)
    assert not parser.closed

    assert list(parser.extended_rows) == [
        (1, None, ['id', 'name']),
        (2, None, [1.0, 'english']),
        (3, None, [2.0, '中国人']),
    ]

    assert len(list(parser.extended_rows)) == 0
    parser.reset()
    assert len(list(parser.extended_rows)) == 3

    parser.close()
    assert parser.closed
