"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IamChanges = void 0;
const cfnspec = require("@aws-cdk/cfnspec");
const chalk = require("chalk");
const diffable_1 = require("../diffable");
const render_intrinsics_1 = require("../render-intrinsics");
const util_1 = require("../util");
const managed_policy_1 = require("./managed-policy");
const statement_1 = require("./statement");
/**
 * Changes to IAM statements
 */
class IamChanges {
    constructor(props) {
        this.statements = new diffable_1.DiffableCollection();
        this.managedPolicies = new diffable_1.DiffableCollection();
        for (const propertyChange of props.propertyChanges) {
            this.readPropertyChange(propertyChange);
        }
        for (const resourceChange of props.resourceChanges) {
            this.readResourceChange(resourceChange);
        }
        this.statements.calculateDiff();
        this.managedPolicies.calculateDiff();
    }
    get hasChanges() {
        return this.statements.hasChanges || this.managedPolicies.hasChanges;
    }
    /**
     * Return whether the changes include broadened permissions
     *
     * Permissions are broadened if positive statements are added or
     * negative statements are removed, or if managed policies are added.
     */
    get permissionsBroadened() {
        return this.statements.additions.some(s => !s.isNegativeStatement)
            || this.statements.removals.some(s => s.isNegativeStatement)
            || this.managedPolicies.hasAdditions;
    }
    /**
     * Return a summary table of changes
     */
    summarizeStatements() {
        const ret = [];
        const header = ['', 'Resource', 'Effect', 'Action', 'Principal', 'Condition'];
        // First generate all lines, then sort on Resource so that similar resources are together
        for (const statement of this.statements.additions) {
            const renderedStatement = statement.render();
            ret.push([
                '+',
                renderedStatement.resource,
                renderedStatement.effect,
                renderedStatement.action,
                renderedStatement.principal,
                renderedStatement.condition,
            ].map(s => chalk.green(s)));
        }
        for (const statement of this.statements.removals) {
            const renderedStatement = statement.render();
            ret.push([
                chalk.red('-'),
                renderedStatement.resource,
                renderedStatement.effect,
                renderedStatement.action,
                renderedStatement.principal,
                renderedStatement.condition,
            ].map(s => chalk.red(s)));
        }
        // Sort by 2nd column
        ret.sort(util_1.makeComparator((row) => [row[1]]));
        ret.splice(0, 0, header);
        return ret;
    }
    summarizeManagedPolicies() {
        const ret = [];
        const header = ['', 'Resource', 'Managed Policy ARN'];
        for (const att of this.managedPolicies.additions) {
            ret.push([
                '+',
                att.identityArn,
                att.managedPolicyArn,
            ].map(s => chalk.green(s)));
        }
        for (const att of this.managedPolicies.removals) {
            ret.push([
                '-',
                att.identityArn,
                att.managedPolicyArn,
            ].map(s => chalk.red(s)));
        }
        // Sort by 2nd column
        ret.sort(util_1.makeComparator((row) => [row[1]]));
        ret.splice(0, 0, header);
        return ret;
    }
    /**
     * Return a machine-readable version of the changes.
     * This is only used in tests.
     *
     * @internal
     */
    _toJson() {
        return util_1.deepRemoveUndefined({
            statementAdditions: util_1.dropIfEmpty(this.statements.additions.map(s => s._toJson())),
            statementRemovals: util_1.dropIfEmpty(this.statements.removals.map(s => s._toJson())),
            managedPolicyAdditions: util_1.dropIfEmpty(this.managedPolicies.additions.map(s => s._toJson())),
            managedPolicyRemovals: util_1.dropIfEmpty(this.managedPolicies.removals.map(s => s._toJson())),
        });
    }
    readPropertyChange(propertyChange) {
        switch (propertyChange.scrutinyType) {
            case cfnspec.schema.PropertyScrutinyType.InlineIdentityPolicies:
                // AWS::IAM::{ Role | User | Group }.Policies
                this.statements.addOld(...this.readIdentityPolicies(propertyChange.oldValue, propertyChange.resourceLogicalId));
                this.statements.addNew(...this.readIdentityPolicies(propertyChange.newValue, propertyChange.resourceLogicalId));
                break;
            case cfnspec.schema.PropertyScrutinyType.InlineResourcePolicy:
                // Any PolicyDocument on a resource (including AssumeRolePolicyDocument)
                this.statements.addOld(...this.readResourceStatements(propertyChange.oldValue, propertyChange.resourceLogicalId));
                this.statements.addNew(...this.readResourceStatements(propertyChange.newValue, propertyChange.resourceLogicalId));
                break;
            case cfnspec.schema.PropertyScrutinyType.ManagedPolicies:
                // Just a list of managed policies
                this.managedPolicies.addOld(...this.readManagedPolicies(propertyChange.oldValue, propertyChange.resourceLogicalId));
                this.managedPolicies.addNew(...this.readManagedPolicies(propertyChange.newValue, propertyChange.resourceLogicalId));
                break;
        }
    }
    readResourceChange(resourceChange) {
        switch (resourceChange.scrutinyType) {
            case cfnspec.schema.ResourceScrutinyType.IdentityPolicyResource:
                // AWS::IAM::Policy
                this.statements.addOld(...this.readIdentityPolicyResource(resourceChange.oldProperties));
                this.statements.addNew(...this.readIdentityPolicyResource(resourceChange.newProperties));
                break;
            case cfnspec.schema.ResourceScrutinyType.ResourcePolicyResource:
                // AWS::*::{Bucket,Queue,Topic}Policy
                this.statements.addOld(...this.readResourcePolicyResource(resourceChange.oldProperties));
                this.statements.addNew(...this.readResourcePolicyResource(resourceChange.newProperties));
                break;
            case cfnspec.schema.ResourceScrutinyType.LambdaPermission:
                this.statements.addOld(...this.readLambdaStatements(resourceChange.oldProperties));
                this.statements.addNew(...this.readLambdaStatements(resourceChange.newProperties));
                break;
        }
    }
    /**
     * Parse a list of policies on an identity
     */
    readIdentityPolicies(policies, logicalId) {
        if (policies === undefined) {
            return [];
        }
        const appliesToPrincipal = 'AWS:${' + logicalId + '}';
        return util_1.flatMap(policies, (policy) => {
            var _a;
            // check if the Policy itself is not an intrinsic, like an Fn::If
            const unparsedStatement = ((_a = policy.PolicyDocument) === null || _a === void 0 ? void 0 : _a.Statement) ? policy.PolicyDocument.Statement
                : policy;
            return defaultPrincipal(appliesToPrincipal, statement_1.parseStatements(render_intrinsics_1.renderIntrinsics(unparsedStatement)));
        });
    }
    /**
     * Parse an IAM::Policy resource
     */
    readIdentityPolicyResource(properties) {
        if (properties === undefined) {
            return [];
        }
        properties = render_intrinsics_1.renderIntrinsics(properties);
        const principals = (properties.Groups || []).concat(properties.Users || []).concat(properties.Roles || []);
        return util_1.flatMap(principals, (principal) => {
            const ref = 'AWS:' + principal;
            return defaultPrincipal(ref, statement_1.parseStatements(properties.PolicyDocument.Statement));
        });
    }
    readResourceStatements(policy, logicalId) {
        if (policy === undefined) {
            return [];
        }
        const appliesToResource = '${' + logicalId + '.Arn}';
        return defaultResource(appliesToResource, statement_1.parseStatements(render_intrinsics_1.renderIntrinsics(policy.Statement)));
    }
    /**
     * Parse an AWS::*::{Bucket,Topic,Queue}policy
     */
    readResourcePolicyResource(properties) {
        if (properties === undefined) {
            return [];
        }
        properties = render_intrinsics_1.renderIntrinsics(properties);
        const policyKeys = Object.keys(properties).filter(key => key.indexOf('Policy') > -1);
        // Find the key that identifies the resource(s) this policy applies to
        const resourceKeys = Object.keys(properties).filter(key => !policyKeys.includes(key) && !key.endsWith('Name'));
        let resources = resourceKeys.length === 1 ? properties[resourceKeys[0]] : ['???'];
        // For some resources, this is a singleton string, for some it's an array
        if (!Array.isArray(resources)) {
            resources = [resources];
        }
        return util_1.flatMap(resources, (resource) => {
            return defaultResource(resource, statement_1.parseStatements(properties[policyKeys[0]].Statement));
        });
    }
    readManagedPolicies(policyArns, logicalId) {
        if (!policyArns) {
            return [];
        }
        const rep = '${' + logicalId + '}';
        return managed_policy_1.ManagedPolicyAttachment.parseManagedPolicies(rep, render_intrinsics_1.renderIntrinsics(policyArns));
    }
    readLambdaStatements(properties) {
        if (!properties) {
            return [];
        }
        return [statement_1.parseLambdaPermission(render_intrinsics_1.renderIntrinsics(properties))];
    }
}
exports.IamChanges = IamChanges;
IamChanges.IamPropertyScrutinies = [
    cfnspec.schema.PropertyScrutinyType.InlineIdentityPolicies,
    cfnspec.schema.PropertyScrutinyType.InlineResourcePolicy,
    cfnspec.schema.PropertyScrutinyType.ManagedPolicies,
];
IamChanges.IamResourceScrutinies = [
    cfnspec.schema.ResourceScrutinyType.ResourcePolicyResource,
    cfnspec.schema.ResourceScrutinyType.IdentityPolicyResource,
    cfnspec.schema.ResourceScrutinyType.LambdaPermission,
];
/**
 * Set an undefined or wildcarded principal on these statements
 */
function defaultPrincipal(principal, statements) {
    statements.forEach(s => s.principals.replaceEmpty(principal));
    statements.forEach(s => s.principals.replaceStar(principal));
    return statements;
}
/**
 * Set an undefined or wildcarded resource on these statements
 */
function defaultResource(resource, statements) {
    statements.forEach(s => s.resources.replaceEmpty(resource));
    statements.forEach(s => s.resources.replaceStar(resource));
    return statements;
}
//# sourceMappingURL=data:application/json;base64,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