"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
/**
 * Split the given CloudFormation resource specification up into a patch set, by service
 *
 * After splitting, only write out those fragments that are valid.(*)
 *
 * The reason for this is that some service updates sometimes contain mistakes,
 * and we want to be able to take partial upgrades. By splitting the spec, we can
 * take updates to some services while leaving updates to other services behind.
 *
 * (*) If `NO_VALIDATE` is set, all pieces are written out. In normal operation this
 *     should not be used.
 */
const path = require("path");
const fs = require("fs-extra");
const patch_set_1 = require("./patch-set");
const validate_cfn_1 = require("./validate-cfn");
async function main(args) {
    if (args.length < 2) {
        throw new Error('Usage: split-spec-by-service <SPECFILE> <DIRECTORY>');
    }
    const [specFile, outDir] = args;
    log(`Loading specification: ${specFile}`);
    const spec = await fs.readJson(specFile);
    // Split
    log('Splitting');
    const byService = {};
    for (const [propTypeName, propType] of Object.entries(spec.PropertyTypes)) {
        const svcName = serviceName(propTypeName);
        serviceSpec(svcName).PropertyTypes[propTypeName] = propType;
    }
    for (const [resTypeName, resType] of Object.entries(spec.ResourceTypes)) {
        const svcName = serviceName(resTypeName);
        serviceSpec(svcName).ResourceTypes[resTypeName] = resType;
    }
    // Write out
    log('Writing');
    for (const [svcName, svcSpec] of Object.entries(byService)) {
        const errors = !process.env.NO_VALIDATE ? validate_cfn_1.CfnSpecValidator.validate(svcSpec) : [];
        if (errors.length === 0) {
            // Change 'ResourceSpecificationVersion' to '$version', otherwise they will all conflict
            const toWrite = {
                PropertyTypes: svcSpec.PropertyTypes,
                ResourceTypes: svcSpec.ResourceTypes,
                $version: svcSpec.ResourceSpecificationVersion,
            };
            await patch_set_1.writeSorted(path.join(outDir, `000_${svcName}.json`), toWrite);
        }
        else {
            console.warn('='.repeat(70));
            console.warn(' '.repeat(Math.floor(35 - svcName.length / 2)) + svcName);
            console.warn('='.repeat(70));
            for (const error of errors) {
                console.warn(validate_cfn_1.formatErrorInContext(error));
            }
        }
    }
    await fs.writeJson(path.join(outDir, '001_Version.json'), {
        ResourceSpecificationVersion: spec.ResourceSpecificationVersion,
    }, { spaces: 2 });
    function serviceSpec(svcName) {
        if (!(svcName in byService)) {
            byService[svcName] = {
                PropertyTypes: {},
                ResourceTypes: {},
                ResourceSpecificationVersion: spec.ResourceSpecificationVersion,
            };
        }
        return byService[svcName];
    }
}
function serviceName(x) {
    return x.split('::').slice(0, 2).join('_');
}
function log(x) {
    // eslint-disable-next-line no-console
    console.log(x);
}
main(process.argv.slice(2)).catch(e => {
    // eslint-disable-next-line no-console
    console.error(e);
    process.exitCode = 1;
});
//# sourceMappingURL=data:application/json;base64,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