#!/usr/bin/env node
"use strict";
/**
 * automatically creates a module for any CloudFormation namespaces that do not
 * have an AWS construct library.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const pkglint = require("@aws-cdk/pkglint");
const fs = require("fs-extra");
const cfnspec = require("../lib");
// don't be a prude:
/* eslint-disable no-console */
/* eslint-disable quote-props */
async function main() {
    const root = path.join(__dirname, '..', '..');
    if (path.basename(root) !== '@aws-cdk') {
        throw new Error(`Something went wrong. We expected ${root} to be the "packages/@aws-cdk" directory. Did you move me?`);
    }
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const cfnSpecPkgJson = require('../package.json');
    const version = cfnSpecPkgJson.version;
    const jestTypesVersion = cfnSpecPkgJson.devDependencies['@types/jest'];
    // iterate over all cloudformation namespaces
    for (const namespace of cfnspec.namespaces()) {
        const module = pkglint.createModuleDefinitionFromCfnNamespace(namespace);
        const lowcaseModuleName = module.moduleBaseName.toLocaleLowerCase();
        const packagePath = path.join(root, module.moduleName);
        // we already have a module for this namesapce, move on.
        if (await fs.pathExists(packagePath)) {
            const packageJsonPath = path.join(packagePath, 'package.json');
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            const packageJson = require(packageJsonPath);
            let scopes = packageJson['cdk-build'].cloudformation;
            if (typeof scopes === 'string') {
                scopes = [scopes];
            }
            if (scopes.indexOf(namespace) !== -1) {
                // V2-style module is already modeled in the root package, nothing to be done!
                continue;
            }
            else if (await fs.pathExists(path.join(root, `${module.moduleFamily}-${module.moduleBaseName}`.toLocaleLowerCase()))) {
                // V2-style package already has it's own package (legacy behavior), nothing to be done!
                continue;
            }
            else {
                // V2-style package needs to be added to it's "V1" package... Get down to business!
                console.error(`Adding ${namespace} to ${module.packageName}`);
                scopes.push(namespace);
                packageJson['cdk-build'].cloudformation = scopes;
                await fs.writeJson(packageJsonPath, packageJson, { encoding: 'utf-8', spaces: 2 });
                const indexTsPath = path.join(packagePath, 'lib', 'index.ts');
                const indexTs = [
                    (await fs.readFile(indexTsPath, { encoding: 'utf8' })).trimRight(),
                    `// ${namespace} CloudFormation Resources:`,
                    `export * from './${lowcaseModuleName}.generated';`,
                ].join('\n');
                await fs.writeFile(indexTsPath, indexTs, { encoding: 'utf8' });
                continue;
            }
        }
        async function write(relativePath, contents) {
            const fullPath = path.join(packagePath, relativePath);
            const dir = path.dirname(fullPath);
            await fs.mkdirp(dir);
            let data;
            if (typeof contents === 'string') {
                data = contents.trimLeft(); // trim first newline
            }
            else if (Array.isArray(contents)) {
                data = contents.join('\n');
            }
            else if (typeof contents === 'object') {
                data = JSON.stringify(contents, undefined, 2);
            }
            else {
                throw new Error('Invalid type of contents: ' + contents);
            }
            await fs.writeFile(fullPath, data + '\n');
        }
        console.log(`generating module for ${module.packageName}...`);
        const description = `${namespace} Construct Library`;
        await write('package.json', {
            name: module.packageName,
            version,
            description,
            main: 'lib/index.js',
            types: 'lib/index.d.ts',
            jsii: {
                outdir: 'dist',
                projectReferences: true,
                targets: {
                    dotnet: {
                        namespace: module.dotnetPackage,
                        packageId: module.dotnetPackage,
                        signAssembly: true,
                        assemblyOriginatorKeyFile: '../../key.snk',
                        iconUrl: 'https://raw.githubusercontent.com/aws/aws-cdk/main/logo/default-256-dark.png',
                    },
                    java: {
                        package: `${module.javaGroupId}.${module.javaPackage}`,
                        maven: {
                            groupId: module.javaGroupId,
                            artifactId: module.javaArtifactId,
                        },
                    },
                    python: {
                        classifiers: [
                            'Framework :: AWS CDK',
                            'Framework :: AWS CDK :: 1',
                        ],
                        distName: module.pythonDistName,
                        module: module.pythonModuleName,
                    },
                },
                metadata: {
                    jsii: {
                        rosetta: {
                            strict: true,
                        },
                    },
                },
            },
            repository: {
                type: 'git',
                url: 'https://github.com/aws/aws-cdk.git',
                directory: `packages/${module.packageName}`,
            },
            homepage: 'https://github.com/aws/aws-cdk',
            scripts: {
                build: 'cdk-build',
                watch: 'cdk-watch',
                lint: 'cdk-lint',
                test: 'cdk-test',
                integ: 'cdk-integ',
                pkglint: 'pkglint -f',
                package: 'cdk-package',
                awslint: 'cdk-awslint',
                cfn2ts: 'cfn2ts',
                'build+test': 'yarn build && yarn test',
                'build+test+package': 'yarn build+test && yarn package',
                compat: 'cdk-compat',
                gen: 'cfn2ts',
                'rosetta:extract': 'yarn --silent jsii-rosetta extract',
                'build+extract': 'yarn build && yarn rosetta:extract',
                'build+test+extract': 'yarn build+test && yarn rosetta:extract',
            },
            'cdk-build': {
                cloudformation: namespace,
                jest: true,
                env: {
                    AWSLINT_BASE_CONSTRUCT: 'true',
                },
            },
            keywords: [
                'aws',
                'cdk',
                'constructs',
                namespace,
                module.moduleName,
            ],
            author: {
                name: 'Amazon Web Services',
                url: 'https://aws.amazon.com',
                organization: true,
            },
            license: 'Apache-2.0',
            devDependencies: {
                '@aws-cdk/assertions': version,
                '@aws-cdk/cdk-build-tools': version,
                '@aws-cdk/cfn2ts': version,
                '@aws-cdk/pkglint': version,
                '@types/jest': jestTypesVersion,
            },
            dependencies: {
                '@aws-cdk/core': version,
            },
            peerDependencies: {
                '@aws-cdk/core': version,
            },
            engines: {
                node: '>= 14.15.0',
            },
            stability: 'experimental',
            maturity: 'cfn-only',
            awscdkio: {
                announce: false,
            },
            publishConfig: {
                tag: 'latest',
            },
        });
        await write('.gitignore', [
            '*.js',
            '*.js.map',
            '*.d.ts',
            'tsconfig.json',
            'node_modules',
            '*.generated.ts',
            'dist',
            '.jsii',
            '',
            '.LAST_BUILD',
            '.nyc_output',
            'coverage',
            '.nycrc',
            '.LAST_PACKAGE',
            '*.snk',
            'nyc.config.js',
            '!.eslintrc.js',
            '!jest.config.js',
            'junit.xml',
        ]);
        await write('.npmignore', [
            '# Don\'t include original .ts files when doing `npm pack`',
            '*.ts',
            '!*.d.ts',
            'coverage',
            '.nyc_output',
            '*.tgz',
            '',
            'dist',
            '.LAST_PACKAGE',
            '.LAST_BUILD',
            '!*.js',
            '',
            '# Include .jsii',
            '!.jsii',
            '',
            '*.snk',
            '',
            '*.tsbuildinfo',
            '',
            'tsconfig.json',
            '',
            '.eslintrc.js',
            'jest.config.js',
            '',
            '# exclude cdk artifacts',
            '**/cdk.out',
            'junit.xml',
            'test/',
            '!*.lit.ts',
        ]);
        await write('lib/index.ts', [
            `// ${namespace} CloudFormation Resources:`,
            `export * from './${lowcaseModuleName}.generated';`,
        ]);
        await write(`test/${lowcaseModuleName}.test.ts`, [
            "import '@aws-cdk/assertions';",
            "import {} from '../lib';",
            '',
            "test('No tests are specified for this package', () => {",
            '  expect(true).toBe(true);',
            '});',
        ]);
        await pkglint.createLibraryReadme(namespace, path.join(packagePath, 'README.md'));
        await write('.eslintrc.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/eslintrc');",
            "baseConfig.parserOptions.project = __dirname + '/tsconfig.json';",
            'module.exports = baseConfig;',
        ]);
        await write('jest.config.js', [
            "const baseConfig = require('@aws-cdk/cdk-build-tools/config/jest.config');",
            'module.exports = baseConfig;',
        ]);
        await write('rosetta/default.ts-fixture', [
            "import { Construct } from 'constructs';",
            "import { Stack } from '@aws-cdk/core';",
            '',
            'class MyStack extends Stack {',
            '  constructor(scope: Construct, id: string) {',
            '    /// here',
            '  }',
            '}',
        ]);
        const templateDir = path.join(__dirname, 'template');
        for (const file of await fs.readdir(templateDir)) {
            await fs.copy(path.join(templateDir, file), path.join(packagePath, file));
        }
        await addDependencyToMegaPackage(path.join('@aws-cdk', 'cloudformation-include'), module.packageName, version, ['dependencies', 'peerDependencies']);
        await addDependencyToMegaPackage('aws-cdk-lib', module.packageName, version, ['devDependencies']);
        await addDependencyToMegaPackage('monocdk', module.packageName, version, ['devDependencies']);
    }
}
/**
 * A few of our packages (e.g., aws-cdk-lib) require a dependency on every service package.
 * This automates adding the dependency (and peer dependency) to the package.json.
 */
async function addDependencyToMegaPackage(megaPackageName, packageName, version, dependencyTypes) {
    const packageJsonPath = path.join(__dirname, '..', '..', '..', megaPackageName, 'package.json');
    const packageJson = JSON.parse(await fs.readFile(packageJsonPath, 'utf8'));
    dependencyTypes.forEach(dependencyType => {
        const unorderedDeps = {
            ...packageJson[dependencyType],
            [packageName]: version,
        };
        packageJson[dependencyType] = {};
        Object.keys(unorderedDeps).sort().forEach(k => {
            packageJson[dependencyType][k] = unorderedDeps[k];
        });
    });
    await fs.writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2) + '\n');
}
main().catch(e => {
    console.error(e);
    process.exit(1);
});
//# sourceMappingURL=data:application/json;base64,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