"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynthUtils = void 0;
const fs = require("fs");
const path = require("path");
const core = require("aws-cdk-lib");
class SynthUtils {
    /**
     * Returns the cloud assembly template artifact for a stack.
     */
    static synthesize(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        return stripNewStyleSynthCfnElements(assembly.getStackArtifact(stack.artifactId));
    }
    /**
     * Synthesizes the stack and returns the resulting CloudFormation template.
     */
    static toCloudFormation(stack, options = {}) {
        const synth = this._synthesizeWithNested(stack, options);
        if (isStackArtifact(synth)) {
            return synth.template;
        }
        else {
            return synth;
        }
    }
    /**
     * @returns Returns a subset of the synthesized CloudFormation template (only specific resource types).
     */
    static subset(stack, options) {
        const template = this.toCloudFormation(stack);
        if (template.Resources) {
            for (const [key, resource] of Object.entries(template.Resources)) {
                if (options.resourceTypes && !options.resourceTypes.includes(resource.Type)) {
                    delete template.Resources[key];
                }
            }
        }
        return template;
    }
    /**
     * Synthesizes the stack and returns a `CloudFormationStackArtifact` which can be inspected.
     * Supports nested stacks as well as normal stacks.
     *
     * @return CloudFormationStackArtifact for normal stacks or the actual template for nested stacks
     * @internal
     */
    static _synthesizeWithNested(stack, options = {}) {
        // always synthesize against the root (be it an App or whatever) so all artifacts will be included
        const assembly = synthesizeApp(stack, options);
        // if this is a nested stack (it has a parent), then just read the template as a string
        if (stack.nestedStackParent) {
            return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
        }
        return stripNewStyleSynthCfnElements(assembly.getStackArtifact(stack.artifactId));
    }
}
exports.SynthUtils = SynthUtils;
/**
 * Synthesizes the app in which a stack resides and returns the cloud assembly object.
 */
function synthesizeApp(stack, options) {
    const root = stack.node.root;
    if (!core.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    // to support incremental assertions (i.e. "expect(stack).toNotContainSomething(); doSomething(); expect(stack).toContainSomthing()")
    const force = true;
    return root.synth({
        force,
        ...options,
    });
}
function stripNewStyleSynthCfnElements(stackArtifact) {
    var _a, _b, _c, _d;
    const synthesizedTemplate = stackArtifact.template;
    // if new-style synthesis is not explicitly set, remove the extra generated Rule and Parameter from the synthesized template,
    // to avoid changing many tests that rely on the template being exactly what it is
    (_a = synthesizedTemplate === null || synthesizedTemplate === void 0 ? void 0 : synthesizedTemplate.Rules) === null || _a === void 0 ? true : delete _a.CheckBootstrapVersion;
    if (Object.keys((_b = synthesizedTemplate === null || synthesizedTemplate === void 0 ? void 0 : synthesizedTemplate.Rules) !== null && _b !== void 0 ? _b : {}).length === 0) {
        synthesizedTemplate === null || synthesizedTemplate === void 0 ? true : delete synthesizedTemplate.Rules;
    }
    (_c = synthesizedTemplate === null || synthesizedTemplate === void 0 ? void 0 : synthesizedTemplate.Parameters) === null || _c === void 0 ? true : delete _c.BootstrapVersion;
    if (Object.keys((_d = synthesizedTemplate === null || synthesizedTemplate === void 0 ? void 0 : synthesizedTemplate.Parameters) !== null && _d !== void 0 ? _d : {}).length === 0) {
        synthesizedTemplate === null || synthesizedTemplate === void 0 ? true : delete synthesizedTemplate.Parameters;
    }
    return stackArtifact;
}
function isStackArtifact(x) {
    return 'template' in x;
}
//# sourceMappingURL=data:application/json;base64,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