import { aws_ec2 as ec2, aws_eks as eks, aws_iam as iam } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import * as runner from './runner';
export interface ProviderProps {
    readonly vpc?: ec2.IVpc;
}
export interface RoleProps {
    readonly accountId: string;
    readonly externalId: string;
}
export interface EksClusterOptions {
    /**
     * create serivce account and rbac ClusterRoleBinding for gitlab
     * @see https://docs.gitlab.com/ee/user/project/clusters/add_remove_clusters.html#add-existing-cluster
     *
     * @default true
     */
    readonly rbac?: boolean;
    /**
     * cluster properties for Amazon EKS cluster
     */
    readonly clusterOptions: eks.ClusterProps;
}
export interface HelmRunnerOptions {
    /**
     * tags for the runner
     *
     * @default - ['eks', 'fargate', 'runner']
     */
    readonly tags?: string[];
    /**
     * GitLab registration token for the runner, you put registrationToken in cdk.context.json like "GITLAB_REGISTRATION_TOKEN": "xxxxxxx".
     */
    readonly registrationToken?: string;
    /**
     * gitlab URL prefix
     *
     * @default - 'https://gitlab.com'
     */
    readonly gitlabURL?: string;
    /**
     * Number of run job in the same time.
     *
     * @default - 10
     */
    readonly concurrent?: number;
    /**
     * Gitlab runners default image when job start not set "image" in gitlab-ci.yaml.
     *
     * @default - alpine:3.11
     */
    readonly jobDefaultImage?: string;
    /**
     * Gitlab helm chart install namespace.
     *
     * if you change this to other namespace, please addFargateProfile() add that you want namespace.
     *
     * @default - default.
     */
    readonly namespace?: string;
}
export interface FargateEksClusterOptions {
    /**
     * create serivce account and rbac ClusterRoleBinding for gitlab
     * @see https://docs.gitlab.com/ee/user/project/clusters/add_remove_clusters.html#add-existing-cluster
     *
     * @default true
     */
    readonly rbac?: boolean;
    /**
     * cluster properties for Amazon EKS cluster
     */
    readonly clusterOptions: eks.FargateClusterProps;
    /**
     * Gitlab helm Chart runner install Options.
     * see https://docs.gitlab.com/runner/install/kubernetes.html
     */
    readonly helmRunnerOptions?: HelmRunnerOptions;
}
/**
 * The Provider to create GitLab Integrations with AWS
 */
export declare class Provider extends Construct {
    gitlabEksRole?: iam.IRole;
    readonly vpc: ec2.IVpc;
    constructor(scope: Construct, id: string, props?: ProviderProps);
    createFargateRunner(executor?: runner.FargateJobExecutor): void;
    createEksCluster(scope: Construct, id: string, props: EksClusterOptions): eks.Cluster;
    createFargateEksCluster(scope: Construct, id: string, props: FargateEksClusterOptions): eks.Cluster;
    createSecurityGroup(): ec2.SecurityGroup;
    createEksServiceRole(): iam.Role;
    createGitlabManagedEksRole(props: RoleProps): void;
    private synthesizeTags;
}
