"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('State Machine Resources', () => {
    test('Tasks can add permissions to the execution role', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const task = new stepfunctions.Task(stack, 'Task', {
            task: {
                bind: () => ({
                    resourceArn: 'resource',
                    policyStatements: [new iam.PolicyStatement({
                            actions: ['resource:Everything'],
                            resources: ['resource'],
                        })],
                }),
            },
        });
        // WHEN
        new stepfunctions.StateMachine(stack, 'SM', {
            definition: task,
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Version: '2012-10-17',
                Statement: [
                    {
                        Action: 'resource:Everything',
                        Effect: 'Allow',
                        Resource: 'resource',
                    },
                ],
            },
        });
    }),
        test('Tasks hidden inside a Parallel state are also included', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        policyStatements: [
                            new iam.PolicyStatement({
                                actions: ['resource:Everything'],
                                resources: ['resource'],
                            }),
                        ],
                    }),
                },
            });
            const para = new stepfunctions.Parallel(stack, 'Para');
            para.branch(task);
            // WHEN
            new stepfunctions.StateMachine(stack, 'SM', {
                definition: para,
            });
            // THEN
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [
                        {
                            Action: 'resource:Everything',
                            Effect: 'Allow',
                            Resource: 'resource',
                        },
                    ],
                },
            });
        }),
        test('Task should render InputPath / Parameters / OutputPath correctly', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                inputPath: '$',
                outputPath: '$.state',
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        parameters: {
                            'input.$': '$',
                            'stringArgument': 'inital-task',
                            'numberArgument': 123,
                            'booleanArgument': true,
                            'arrayArgument': ['a', 'b', 'c'],
                        },
                    }),
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({
                End: true,
                Retry: undefined,
                Catch: undefined,
                InputPath: '$',
                Parameters: {
                    'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'],
                },
                OutputPath: '$.state',
                Type: 'Task',
                Comment: undefined,
                Resource: 'resource',
                ResultPath: undefined,
                TimeoutSeconds: undefined,
                HeartbeatSeconds: undefined,
            });
        }),
        test('Task combines taskobject parameters with direct parameters', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                inputPath: '$',
                outputPath: '$.state',
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        parameters: {
                            a: 'aa',
                        },
                    }),
                },
                parameters: {
                    b: 'bb',
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({
                End: true,
                Retry: undefined,
                Catch: undefined,
                InputPath: '$',
                Parameters: {
                    a: 'aa',
                    b: 'bb',
                },
                OutputPath: '$.state',
                Type: 'Task',
                Comment: undefined,
                Resource: 'resource',
                ResultPath: undefined,
                TimeoutSeconds: undefined,
                HeartbeatSeconds: undefined,
            });
        }),
        test('Created state machine can grant start execution to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantStartExecution(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith(assert_1.objectLike({
                        Action: 'states:StartExecution',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'StateMachine2E01A3A5',
                        },
                    })),
                },
            });
        }),
        test('Created state machine can grant read access to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantRead(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:ListExecutions',
                                'states:ListStateMachines',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'StateMachine2E01A3A5',
                            },
                        },
                        {
                            Action: [
                                'states:DescribeExecution',
                                'states:DescribeStateMachineForExecution',
                                'states:GetExecutionHistory',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':states:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':execution:',
                                        {
                                            'Fn::Select': [
                                                6,
                                                {
                                                    'Fn::Split': [
                                                        ':',
                                                        {
                                                            Ref: 'StateMachine2E01A3A5',
                                                        },
                                                    ],
                                                },
                                            ],
                                        },
                                        ':*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: [
                                'states:ListActivities',
                                'states:DescribeStateMachine',
                                'states:DescribeActivity',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                },
            });
        }),
        test('Created state machine can grant task response actions to the state machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantTaskResponse(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:SendTaskSuccess',
                                'states:SendTaskFailure',
                                'states:SendTaskHeartbeat',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'StateMachine2E01A3A5',
                            },
                        },
                    ],
                },
            });
        }),
        test('Created state machine can grant actions to the executions', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantExecution(role, 'states:GetExecutionHistory');
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:GetExecutionHistory',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':states:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':execution:',
                                        {
                                            'Fn::Select': [
                                                6,
                                                {
                                                    'Fn::Split': [
                                                        ':',
                                                        {
                                                            Ref: 'StateMachine2E01A3A5',
                                                        },
                                                    ],
                                                },
                                            ],
                                        },
                                        ':*',
                                    ],
                                ],
                            },
                        },
                    ],
                },
            });
        }),
        test('Created state machine can grant actions to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({ resourceArn: 'resource' }),
                },
            });
            const stateMachine = new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grant(role, 'states:ListExecution');
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:ListExecution',
                            Effect: 'Allow',
                            Resource: {
                                Ref: 'StateMachine2E01A3A5',
                            },
                        },
                    ],
                },
            });
        }),
        test('Imported state machine can grant start execution to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantStartExecution(role);
            // THEN
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:StartExecution',
                            Effect: 'Allow',
                            Resource: stateMachineArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                PolicyName: 'RoleDefaultPolicy5FFB7DAB',
                Roles: [
                    {
                        Ref: 'Role1ABCC5F0',
                    },
                ],
            });
        }),
        test('Imported state machine can grant read access to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantRead(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:ListExecutions',
                                'states:ListStateMachines',
                            ],
                            Effect: 'Allow',
                            Resource: stateMachineArn,
                        },
                        {
                            Action: [
                                'states:DescribeExecution',
                                'states:DescribeStateMachineForExecution',
                                'states:GetExecutionHistory',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':states:',
                                        {
                                            Ref: 'AWS::Region',
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':execution:*',
                                    ],
                                ],
                            },
                        },
                        {
                            Action: [
                                'states:ListActivities',
                                'states:DescribeStateMachine',
                                'states:DescribeActivity',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        },
                    ],
                },
            });
        }),
        test('Imported state machine can task response permissions to the state machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grantTaskResponse(role);
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'states:SendTaskSuccess',
                                'states:SendTaskFailure',
                                'states:SendTaskHeartbeat',
                            ],
                            Effect: 'Allow',
                            Resource: stateMachineArn,
                        },
                    ],
                },
            });
        }),
        test('Imported state machine can grant access to a role', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const stateMachineArn = 'arn:aws:states:::my-state-machine';
            const stateMachine = stepfunctions.StateMachine.fromStateMachineArn(stack, 'StateMachine', stateMachineArn);
            const role = new iam.Role(stack, 'Role', {
                assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            });
            // WHEN
            stateMachine.grant(role, 'states:ListExecution');
            // THEN
            expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'states:ListExecution',
                            Effect: 'Allow',
                            Resource: stateMachine.stateMachineArn,
                        },
                    ],
                },
            });
        }),
        test('Pass should render InputPath / Parameters / OutputPath correctly', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Pass(stack, 'Pass', {
                inputPath: '$',
                outputPath: '$.state',
                parameters: {
                    'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'],
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toStrictEqual({
                End: true,
                InputPath: '$',
                OutputPath: '$.state',
                Parameters: {
                    'input.$': '$',
                    'stringArgument': 'inital-task',
                    'numberArgument': 123,
                    'booleanArgument': true,
                    'arrayArgument': ['a', 'b', 'c'],
                },
                Type: 'Pass',
                Comment: undefined,
                Result: undefined,
                ResultPath: undefined,
            });
        }),
        test('parameters can be selected from the input with a path', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Pass(stack, 'Pass', {
                parameters: {
                    input: stepfunctions.JsonPath.stringAt('$.myField'),
                },
            });
            // WHEN
            const taskState = task.toStateJson();
            // THEN
            expect(taskState).toEqual({
                End: true,
                Parameters: { 'input.$': '$.myField' },
                Type: 'Pass',
            });
        }),
        test('State machines must depend on their roles', () => {
            // GIVEN
            const stack = new cdk.Stack();
            const task = new stepfunctions.Task(stack, 'Task', {
                task: {
                    bind: () => ({
                        resourceArn: 'resource',
                        policyStatements: [
                            new iam.PolicyStatement({
                                resources: ['resource'],
                                actions: ['lambda:InvokeFunction'],
                            }),
                        ],
                    }),
                },
            });
            new stepfunctions.StateMachine(stack, 'StateMachine', {
                definition: task,
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                DependsOn: [
                    'StateMachineRoleDefaultPolicyDF1E6607',
                    'StateMachineRoleB840431D',
                ],
            }, assert_1.ResourcePart.CompleteDefinition);
        });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGUtbWFjaGluZS1yZXNvdXJjZXMudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInN0YXRlLW1hY2hpbmUtcmVzb3VyY2VzLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSw0Q0FBc0U7QUFDdEUsZ0NBQThCO0FBQzlCLHdDQUF3QztBQUN4QyxxQ0FBcUM7QUFDckMsd0NBQXdDO0FBRXhDLFFBQVEsQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7SUFDdkMsSUFBSSxDQUFDLGlEQUFpRCxFQUFFLEdBQUcsRUFBRTtRQUMzRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDakQsSUFBSSxFQUFFO2dCQUNKLElBQUksRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO29CQUNYLFdBQVcsRUFBRSxVQUFVO29CQUN2QixnQkFBZ0IsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQzs0QkFDekMsT0FBTyxFQUFFLENBQUMscUJBQXFCLENBQUM7NEJBQ2hDLFNBQVMsRUFBRSxDQUFDLFVBQVUsQ0FBQzt5QkFDeEIsQ0FBQyxDQUFDO2lCQUNKLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRTtZQUMxQyxVQUFVLEVBQUUsSUFBSTtTQUNqQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtZQUMvQyxjQUFjLEVBQUU7Z0JBQ2QsT0FBTyxFQUFFLFlBQVk7Z0JBQ3JCLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUUscUJBQXFCO3dCQUM3QixNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUUsVUFBVTtxQkFDckI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyx3REFBd0QsRUFBRSxHQUFHLEVBQUU7WUFDbEUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUM7d0JBQ1gsV0FBVyxFQUFFLFVBQVU7d0JBQ3ZCLGdCQUFnQixFQUFFOzRCQUNoQixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0NBQ3RCLE9BQU8sRUFBRSxDQUFDLHFCQUFxQixDQUFDO2dDQUNoQyxTQUFTLEVBQUUsQ0FBQyxVQUFVLENBQUM7NkJBQ3hCLENBQUM7eUJBQ0g7cUJBQ0YsQ0FBQztpQkFDSDthQUNGLENBQUMsQ0FBQztZQUVILE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDdkQsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUVsQixPQUFPO1lBQ1AsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUU7Z0JBQzFDLFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtCQUFrQixFQUFFO2dCQUMvQyxjQUFjLEVBQUU7b0JBQ2QsT0FBTyxFQUFFLFlBQVk7b0JBQ3JCLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUUscUJBQXFCOzRCQUM3QixNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsVUFBVTt5QkFDckI7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsa0VBQWtFLEVBQUUsR0FBRyxFQUFFO1lBQzVFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQzt3QkFDWCxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsVUFBVSxFQUFFOzRCQUNWLFNBQVMsRUFBRSxHQUFHOzRCQUNkLGdCQUFnQixFQUFFLGFBQWE7NEJBQy9CLGdCQUFnQixFQUFFLEdBQUc7NEJBQ3JCLGlCQUFpQixFQUFFLElBQUk7NEJBQ3ZCLGVBQWUsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDO3lCQUNqQztxQkFDRixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLGFBQWEsQ0FBQztnQkFDOUIsR0FBRyxFQUFFLElBQUk7Z0JBQ1QsS0FBSyxFQUFFLFNBQVM7Z0JBQ2hCLEtBQUssRUFBRSxTQUFTO2dCQUNoQixTQUFTLEVBQUUsR0FBRztnQkFDZCxVQUFVLEVBQ0g7b0JBQ0UsU0FBUyxFQUFFLEdBQUc7b0JBQ2QsZ0JBQWdCLEVBQUUsYUFBYTtvQkFDL0IsZ0JBQWdCLEVBQUUsR0FBRztvQkFDckIsaUJBQWlCLEVBQUUsSUFBSTtvQkFDdkIsZUFBZSxFQUFFLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUM7aUJBQ2pDO2dCQUNSLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixJQUFJLEVBQUUsTUFBTTtnQkFDWixPQUFPLEVBQUUsU0FBUztnQkFDbEIsUUFBUSxFQUFFLFVBQVU7Z0JBQ3BCLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixjQUFjLEVBQUUsU0FBUztnQkFDekIsZ0JBQWdCLEVBQUUsU0FBUzthQUM1QixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO1lBQ3RFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLElBQUksR0FBRyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDakQsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQzt3QkFDWCxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsVUFBVSxFQUFFOzRCQUNWLENBQUMsRUFBRSxJQUFJO3lCQUNSO3FCQUNGLENBQUM7aUJBQ0g7Z0JBQ0QsVUFBVSxFQUFFO29CQUNWLENBQUMsRUFBRSxJQUFJO2lCQUNSO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxXQUFXLEVBQUUsQ0FBQztZQUVyQyxPQUFPO1lBQ1AsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDLGFBQWEsQ0FBQztnQkFDOUIsR0FBRyxFQUFFLElBQUk7Z0JBQ1QsS0FBSyxFQUFFLFNBQVM7Z0JBQ2hCLEtBQUssRUFBRSxTQUFTO2dCQUNoQixTQUFTLEVBQUUsR0FBRztnQkFDZCxVQUFVLEVBQ0g7b0JBQ0UsQ0FBQyxFQUFFLElBQUk7b0JBQ1AsQ0FBQyxFQUFFLElBQUk7aUJBQ1I7Z0JBQ1IsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLElBQUksRUFBRSxNQUFNO2dCQUNaLE9BQU8sRUFBRSxTQUFTO2dCQUNsQixRQUFRLEVBQUUsVUFBVTtnQkFDcEIsVUFBVSxFQUFFLFNBQVM7Z0JBQ3JCLGNBQWMsRUFBRSxTQUFTO2dCQUN6QixnQkFBZ0IsRUFBRSxTQUFTO2FBQzVCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQywyREFBMkQsRUFBRSxHQUFHLEVBQUU7WUFDckUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLENBQUM7aUJBQzFDO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQ3pFLFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUNILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7YUFDNUQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztZQUV2QyxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLGtCQUFrQixFQUFFO2dCQUNuRCxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFLGtCQUFTLENBQUMsbUJBQVUsQ0FBQzt3QkFDOUIsTUFBTSxFQUFFLHVCQUF1Qjt3QkFDL0IsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFOzRCQUNSLEdBQUcsRUFBRSxzQkFBc0I7eUJBQzVCO3FCQUNGLENBQUMsQ0FBQztpQkFDSjthQUNGLENBQUMsQ0FBQztRQUVMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyx1REFBdUQsRUFBRSxHQUFHLEVBQUU7WUFDakUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLENBQUM7aUJBQzFDO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQ3pFLFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUNILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7YUFDNUQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFlBQVksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLENBQUM7WUFFN0IsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbkQsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04sdUJBQXVCO2dDQUN2QiwwQkFBMEI7NkJBQzNCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRTtnQ0FDUixHQUFHLEVBQUUsc0JBQXNCOzZCQUM1Qjt5QkFDRjt3QkFDRDs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04sMEJBQTBCO2dDQUMxQix5Q0FBeUM7Z0NBQ3pDLDRCQUE0Qjs2QkFDN0I7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFO2dDQUNSLFVBQVUsRUFBRTtvQ0FDVixFQUFFO29DQUNGO3dDQUNFLE1BQU07d0NBQ047NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsVUFBVTt3Q0FDVjs0Q0FDRSxHQUFHLEVBQUUsYUFBYTt5Q0FDbkI7d0NBQ0QsR0FBRzt3Q0FDSDs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCxhQUFhO3dDQUNiOzRDQUNFLFlBQVksRUFBRTtnREFDWixDQUFDO2dEQUNEO29EQUNFLFdBQVcsRUFBRTt3REFDWCxHQUFHO3dEQUNIOzREQUNFLEdBQUcsRUFBRSxzQkFBc0I7eURBQzVCO3FEQUNGO2lEQUNGOzZDQUNGO3lDQUNGO3dDQUNELElBQUk7cUNBQ0w7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLHVCQUF1QjtnQ0FDdkIsNkJBQTZCO2dDQUM3Qix5QkFBeUI7NkJBQzFCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxHQUFHO3lCQUNkO3FCQUNGO2lCQUNGO2FBQ0YsQ0FDQSxDQUFDO1FBRUosQ0FBQyxDQUFDO1FBRUYsSUFBSSxDQUFDLDRFQUE0RSxFQUFFLEdBQUcsRUFBRTtZQUN0RixRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ2pELElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxVQUFVLEVBQUUsQ0FBQztpQkFDMUM7YUFDRixDQUFDLENBQUM7WUFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRTtnQkFDekUsVUFBVSxFQUFFLElBQUk7YUFDakIsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQzthQUM1RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsWUFBWSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBRXJDLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25ELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLHdCQUF3QjtnQ0FDeEIsd0JBQXdCO2dDQUN4QiwwQkFBMEI7NkJBQzNCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRTtnQ0FDUixHQUFHLEVBQUUsc0JBQXNCOzZCQUM1Qjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQywyREFBMkQsRUFBRSxHQUFHLEVBQUU7WUFDckUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLENBQUM7aUJBQzFDO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQ3pFLFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUNILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7YUFDNUQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFlBQVksQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLDRCQUE0QixDQUFDLENBQUM7WUFFaEUsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbkQsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUUsNEJBQTRCOzRCQUNwQyxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUU7Z0NBQ1IsVUFBVSxFQUFFO29DQUNWLEVBQUU7b0NBQ0Y7d0NBQ0UsTUFBTTt3Q0FDTjs0Q0FDRSxHQUFHLEVBQUUsZ0JBQWdCO3lDQUN0Qjt3Q0FDRCxVQUFVO3dDQUNWOzRDQUNFLEdBQUcsRUFBRSxhQUFhO3lDQUNuQjt3Q0FDRCxHQUFHO3dDQUNIOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELGFBQWE7d0NBQ2I7NENBQ0UsWUFBWSxFQUFFO2dEQUNaLENBQUM7Z0RBQ0Q7b0RBQ0UsV0FBVyxFQUFFO3dEQUNYLEdBQUc7d0RBQ0g7NERBQ0UsR0FBRyxFQUFFLHNCQUFzQjt5REFDNUI7cURBQ0Y7aURBQ0Y7NkNBQ0Y7eUNBQ0Y7d0NBQ0QsSUFBSTtxQ0FDTDtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyxtREFBbUQsRUFBRSxHQUFHLEVBQUU7WUFDN0QsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxJQUFJLEVBQUU7b0JBQ0osSUFBSSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsVUFBVSxFQUFFLENBQUM7aUJBQzFDO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQ3pFLFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUNILE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLENBQUM7YUFDNUQsQ0FBQyxDQUFDO1lBRUgsT0FBTztZQUNQLFlBQVksQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLHNCQUFzQixDQUFDLENBQUM7WUFFakQsT0FBTztZQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbkQsY0FBYyxFQUFFO29CQUNkLFNBQVMsRUFBRTt3QkFDVDs0QkFDRSxNQUFNLEVBQUUsc0JBQXNCOzRCQUM5QixNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUU7Z0NBQ1IsR0FBRyxFQUFFLHNCQUFzQjs2QkFDNUI7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFFTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO1lBQ3RFLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGVBQWUsR0FBRyxtQ0FBbUMsQ0FBQztZQUM1RCxNQUFNLFlBQVksR0FBRyxhQUFhLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDNUcsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQzthQUM1RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsWUFBWSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBRXZDLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtCQUFrQixFQUFFO2dCQUMvQyxjQUFjLEVBQUU7b0JBQ2QsU0FBUyxFQUFFO3dCQUNUOzRCQUNFLE1BQU0sRUFBRSx1QkFBdUI7NEJBQy9CLE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxlQUFlO3lCQUMxQjtxQkFDRjtvQkFDRCxPQUFPLEVBQUUsWUFBWTtpQkFDdEI7Z0JBQ0QsVUFBVSxFQUFFLDJCQUEyQjtnQkFDdkMsS0FBSyxFQUFFO29CQUNMO3dCQUNFLEdBQUcsRUFBRSxjQUFjO3FCQUNwQjtpQkFDRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyx3REFBd0QsRUFBRSxHQUFHLEVBQUU7WUFDbEUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sZUFBZSxHQUFHLG1DQUFtQyxDQUFDO1lBQzVELE1BQU0sWUFBWSxHQUFHLGFBQWEsQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxlQUFlLENBQUMsQ0FBQztZQUM1RyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxDQUFDO1lBRTdCLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25ELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLHVCQUF1QjtnQ0FDdkIsMEJBQTBCOzZCQUMzQjs0QkFDRCxNQUFNLEVBQUUsT0FBTzs0QkFDZixRQUFRLEVBQUUsZUFBZTt5QkFDMUI7d0JBQ0Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLDBCQUEwQjtnQ0FDMUIseUNBQXlDO2dDQUN6Qyw0QkFBNEI7NkJBQzdCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRTtnQ0FDUixVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDRSxNQUFNO3dDQUNOOzRDQUNFLEdBQUcsRUFBRSxnQkFBZ0I7eUNBQ3RCO3dDQUNELFVBQVU7d0NBQ1Y7NENBQ0UsR0FBRyxFQUFFLGFBQWE7eUNBQ25CO3dDQUNELEdBQUc7d0NBQ0g7NENBQ0UsR0FBRyxFQUFFLGdCQUFnQjt5Q0FDdEI7d0NBQ0QsY0FBYztxQ0FDZjtpQ0FDRjs2QkFDRjt5QkFDRjt3QkFDRDs0QkFDRSxNQUFNLEVBQUU7Z0NBQ04sdUJBQXVCO2dDQUN2Qiw2QkFBNkI7Z0NBQzdCLHlCQUF5Qjs2QkFDMUI7NEJBQ0QsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFLEdBQUc7eUJBQ2Q7cUJBQ0Y7aUJBQ0Y7YUFDRixDQUNBLENBQUM7UUFDSixDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsMkVBQTJFLEVBQUUsR0FBRyxFQUFFO1lBQ3JGLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLGVBQWUsR0FBRyxtQ0FBbUMsQ0FBQztZQUM1RCxNQUFNLFlBQVksR0FBRyxhQUFhLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsZUFBZSxDQUFDLENBQUM7WUFDNUcsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsQ0FBQzthQUM1RCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsWUFBWSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxDQUFDO1lBRXJDLE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25ELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFO2dDQUNOLHdCQUF3QjtnQ0FDeEIsd0JBQXdCO2dDQUN4QiwwQkFBMEI7NkJBQzNCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxlQUFlO3lCQUMxQjtxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyxtREFBbUQsRUFBRSxHQUFHLEVBQUU7WUFDN0QsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sZUFBZSxHQUFHLG1DQUFtQyxDQUFDO1lBQzVELE1BQU0sWUFBWSxHQUFHLGFBQWEsQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLGNBQWMsRUFBRSxlQUFlLENBQUMsQ0FBQztZQUM1RyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtnQkFDdkMsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO2FBQzVELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxZQUFZLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxzQkFBc0IsQ0FBQyxDQUFDO1lBRWpELE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25ELGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUU7d0JBQ1Q7NEJBQ0UsTUFBTSxFQUFFLHNCQUFzQjs0QkFDOUIsTUFBTSxFQUFFLE9BQU87NEJBQ2YsUUFBUSxFQUFFLFlBQVksQ0FBQyxlQUFlO3lCQUN2QztxQkFDRjtpQkFDRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyxrRUFBa0UsRUFBRSxHQUFHLEVBQUU7WUFDNUUsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sSUFBSSxHQUFHLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO2dCQUNqRCxTQUFTLEVBQUUsR0FBRztnQkFDZCxVQUFVLEVBQUUsU0FBUztnQkFDckIsVUFBVSxFQUFFO29CQUNWLFNBQVMsRUFBRSxHQUFHO29CQUNkLGdCQUFnQixFQUFFLGFBQWE7b0JBQy9CLGdCQUFnQixFQUFFLEdBQUc7b0JBQ3JCLGlCQUFpQixFQUFFLElBQUk7b0JBQ3ZCLGVBQWUsRUFBRSxDQUFDLEdBQUcsRUFBRSxHQUFHLEVBQUUsR0FBRyxDQUFDO2lCQUNqQzthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7WUFFckMsT0FBTztZQUNQLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxhQUFhLENBQUM7Z0JBQzlCLEdBQUcsRUFBRSxJQUFJO2dCQUNULFNBQVMsRUFBRSxHQUFHO2dCQUNkLFVBQVUsRUFBRSxTQUFTO2dCQUNyQixVQUFVLEVBQ0g7b0JBQ0UsU0FBUyxFQUFFLEdBQUc7b0JBQ2QsZ0JBQWdCLEVBQUUsYUFBYTtvQkFDL0IsZ0JBQWdCLEVBQUUsR0FBRztvQkFDckIsaUJBQWlCLEVBQUUsSUFBSTtvQkFDdkIsZUFBZSxFQUFFLENBQUMsR0FBRyxFQUFFLEdBQUcsRUFBRSxHQUFHLENBQUM7aUJBQ2pDO2dCQUNSLElBQUksRUFBRSxNQUFNO2dCQUNaLE9BQU8sRUFBRSxTQUFTO2dCQUNsQixNQUFNLEVBQUUsU0FBUztnQkFDakIsVUFBVSxFQUFFLFNBQVM7YUFDdEIsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO1FBRUYsSUFBSSxDQUFDLHVEQUF1RCxFQUFFLEdBQUcsRUFBRTtZQUNqRSxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ2pELFVBQVUsRUFBRTtvQkFDVixLQUFLLEVBQUUsYUFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDO2lCQUNwRDthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUM7WUFFckMsT0FBTztZQUNQLE1BQU0sQ0FBQyxTQUFTLENBQUMsQ0FBQyxPQUFPLENBQUM7Z0JBQ3hCLEdBQUcsRUFBRSxJQUFJO2dCQUNULFVBQVUsRUFDVixFQUFFLFNBQVMsRUFBRSxXQUFXLEVBQUU7Z0JBQzFCLElBQUksRUFBRSxNQUFNO2FBQ2IsQ0FBQyxDQUFDO1FBQ0wsQ0FBQyxDQUFDO1FBRUYsSUFBSSxDQUFDLDJDQUEyQyxFQUFFLEdBQUcsRUFBRTtZQUNyRCxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxJQUFJLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7Z0JBQ2pELElBQUksRUFBRTtvQkFDSixJQUFJLEVBQUUsR0FBRyxFQUFFLENBQUMsQ0FBQzt3QkFDWCxXQUFXLEVBQUUsVUFBVTt3QkFDdkIsZ0JBQWdCLEVBQUU7NEJBQ2hCLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQ0FDdEIsU0FBUyxFQUFFLENBQUMsVUFBVSxDQUFDO2dDQUN2QixPQUFPLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQzs2QkFDbkMsQ0FBQzt5QkFDSDtxQkFDRixDQUFDO2lCQUNIO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxhQUFhLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUU7Z0JBQ3BELFVBQVUsRUFBRSxJQUFJO2FBQ2pCLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtDQUFrQyxFQUFFO2dCQUMvRCxTQUFTLEVBQUU7b0JBQ1QsdUNBQXVDO29CQUN2QywwQkFBMEI7aUJBQzNCO2FBQ0YsRUFBRSxxQkFBWSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDdEMsQ0FBQyxDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGFycmF5V2l0aCwgb2JqZWN0TGlrZSwgUmVzb3VyY2VQYXJ0IH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgc3RlcGZ1bmN0aW9ucyBmcm9tICcuLi9saWInO1xuXG5kZXNjcmliZSgnU3RhdGUgTWFjaGluZSBSZXNvdXJjZXMnLCAoKSA9PiB7XG4gIHRlc3QoJ1Rhc2tzIGNhbiBhZGQgcGVybWlzc2lvbnMgdG8gdGhlIGV4ZWN1dGlvbiByb2xlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgICAgdGFzazoge1xuICAgICAgICBiaW5kOiAoKSA9PiAoe1xuICAgICAgICAgIHJlc291cmNlQXJuOiAncmVzb3VyY2UnLFxuICAgICAgICAgIHBvbGljeVN0YXRlbWVudHM6IFtuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ3Jlc291cmNlOkV2ZXJ5dGhpbmcnXSxcbiAgICAgICAgICAgIHJlc291cmNlczogWydyZXNvdXJjZSddLFxuICAgICAgICAgIH0pXSxcbiAgICAgICAgfSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZShzdGFjaywgJ1NNJywge1xuICAgICAgZGVmaW5pdGlvbjogdGFzayxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246ICdyZXNvdXJjZTpFdmVyeXRoaW5nJyxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiAncmVzb3VyY2UnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdUYXNrcyBoaWRkZW4gaW5zaWRlIGEgUGFyYWxsZWwgc3RhdGUgYXJlIGFsc28gaW5jbHVkZWQnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiB7XG4gICAgICAgIGJpbmQ6ICgpID0+ICh7XG4gICAgICAgICAgcmVzb3VyY2VBcm46ICdyZXNvdXJjZScsXG4gICAgICAgICAgcG9saWN5U3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBhY3Rpb25zOiBbJ3Jlc291cmNlOkV2ZXJ5dGhpbmcnXSxcbiAgICAgICAgICAgICAgcmVzb3VyY2VzOiBbJ3Jlc291cmNlJ10sXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgICBdLFxuICAgICAgICB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBwYXJhID0gbmV3IHN0ZXBmdW5jdGlvbnMuUGFyYWxsZWwoc3RhY2ssICdQYXJhJyk7XG4gICAgcGFyYS5icmFuY2godGFzayk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnU00nLCB7XG4gICAgICBkZWZpbml0aW9uOiBwYXJhLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogJ3Jlc291cmNlOkV2ZXJ5dGhpbmcnLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6ICdyZXNvdXJjZScsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ1Rhc2sgc2hvdWxkIHJlbmRlciBJbnB1dFBhdGggLyBQYXJhbWV0ZXJzIC8gT3V0cHV0UGF0aCBjb3JyZWN0bHknLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICBpbnB1dFBhdGg6ICckJyxcbiAgICAgIG91dHB1dFBhdGg6ICckLnN0YXRlJyxcbiAgICAgIHRhc2s6IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHtcbiAgICAgICAgICByZXNvdXJjZUFybjogJ3Jlc291cmNlJyxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAnaW5wdXQuJCc6ICckJyxcbiAgICAgICAgICAgICdzdHJpbmdBcmd1bWVudCc6ICdpbml0YWwtdGFzaycsXG4gICAgICAgICAgICAnbnVtYmVyQXJndW1lbnQnOiAxMjMsXG4gICAgICAgICAgICAnYm9vbGVhbkFyZ3VtZW50JzogdHJ1ZSxcbiAgICAgICAgICAgICdhcnJheUFyZ3VtZW50JzogWydhJywgJ2InLCAnYyddLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB0YXNrU3RhdGUgPSB0YXNrLnRvU3RhdGVKc29uKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHRhc2tTdGF0ZSkudG9TdHJpY3RFcXVhbCh7XG4gICAgICBFbmQ6IHRydWUsXG4gICAgICBSZXRyeTogdW5kZWZpbmVkLFxuICAgICAgQ2F0Y2g6IHVuZGVmaW5lZCxcbiAgICAgIElucHV0UGF0aDogJyQnLFxuICAgICAgUGFyYW1ldGVyczpcbiAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAnaW5wdXQuJCc6ICckJyxcbiAgICAgICAgICAgICAgICdzdHJpbmdBcmd1bWVudCc6ICdpbml0YWwtdGFzaycsXG4gICAgICAgICAgICAgICAnbnVtYmVyQXJndW1lbnQnOiAxMjMsXG4gICAgICAgICAgICAgICAnYm9vbGVhbkFyZ3VtZW50JzogdHJ1ZSxcbiAgICAgICAgICAgICAgICdhcnJheUFyZ3VtZW50JzogWydhJywgJ2InLCAnYyddLFxuICAgICAgICAgICAgIH0sXG4gICAgICBPdXRwdXRQYXRoOiAnJC5zdGF0ZScsXG4gICAgICBUeXBlOiAnVGFzaycsXG4gICAgICBDb21tZW50OiB1bmRlZmluZWQsXG4gICAgICBSZXNvdXJjZTogJ3Jlc291cmNlJyxcbiAgICAgIFJlc3VsdFBhdGg6IHVuZGVmaW5lZCxcbiAgICAgIFRpbWVvdXRTZWNvbmRzOiB1bmRlZmluZWQsXG4gICAgICBIZWFydGJlYXRTZWNvbmRzOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ1Rhc2sgY29tYmluZXMgdGFza29iamVjdCBwYXJhbWV0ZXJzIHdpdGggZGlyZWN0IHBhcmFtZXRlcnMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICBpbnB1dFBhdGg6ICckJyxcbiAgICAgIG91dHB1dFBhdGg6ICckLnN0YXRlJyxcbiAgICAgIHRhc2s6IHtcbiAgICAgICAgYmluZDogKCkgPT4gKHtcbiAgICAgICAgICByZXNvdXJjZUFybjogJ3Jlc291cmNlJyxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBhOiAnYWEnLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgYjogJ2JiJyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgY29uc3QgdGFza1N0YXRlID0gdGFzay50b1N0YXRlSnNvbigpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdCh0YXNrU3RhdGUpLnRvU3RyaWN0RXF1YWwoe1xuICAgICAgRW5kOiB0cnVlLFxuICAgICAgUmV0cnk6IHVuZGVmaW5lZCxcbiAgICAgIENhdGNoOiB1bmRlZmluZWQsXG4gICAgICBJbnB1dFBhdGg6ICckJyxcbiAgICAgIFBhcmFtZXRlcnM6XG4gICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgYTogJ2FhJyxcbiAgICAgICAgICAgICAgIGI6ICdiYicsXG4gICAgICAgICAgICAgfSxcbiAgICAgIE91dHB1dFBhdGg6ICckLnN0YXRlJyxcbiAgICAgIFR5cGU6ICdUYXNrJyxcbiAgICAgIENvbW1lbnQ6IHVuZGVmaW5lZCxcbiAgICAgIFJlc291cmNlOiAncmVzb3VyY2UnLFxuICAgICAgUmVzdWx0UGF0aDogdW5kZWZpbmVkLFxuICAgICAgVGltZW91dFNlY29uZHM6IHVuZGVmaW5lZCxcbiAgICAgIEhlYXJ0YmVhdFNlY29uZHM6IHVuZGVmaW5lZCxcbiAgICB9KTtcbiAgfSksXG5cbiAgdGVzdCgnQ3JlYXRlZCBzdGF0ZSBtYWNoaW5lIGNhbiBncmFudCBzdGFydCBleGVjdXRpb24gdG8gYSByb2xlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgICAgdGFzazoge1xuICAgICAgICBiaW5kOiAoKSA9PiAoeyByZXNvdXJjZUFybjogJ3Jlc291cmNlJyB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3Qgc3RhdGVNYWNoaW5lID0gbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgZGVmaW5pdGlvbjogdGFzayxcbiAgICB9KTtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YXRlTWFjaGluZS5ncmFudFN0YXJ0RXhlY3V0aW9uKHJvbGUpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBhcnJheVdpdGgob2JqZWN0TGlrZSh7XG4gICAgICAgICAgQWN0aW9uOiAnc3RhdGVzOlN0YXJ0RXhlY3V0aW9uJyxcbiAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgIFJlZjogJ1N0YXRlTWFjaGluZTJFMDFBM0E1JyxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gIH0pLFxuXG4gIHRlc3QoJ0NyZWF0ZWQgc3RhdGUgbWFjaGluZSBjYW4gZ3JhbnQgcmVhZCBhY2Nlc3MgdG8gYSByb2xlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgICAgdGFzazoge1xuICAgICAgICBiaW5kOiAoKSA9PiAoeyByZXNvdXJjZUFybjogJ3Jlc291cmNlJyB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3Qgc3RhdGVNYWNoaW5lID0gbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgZGVmaW5pdGlvbjogdGFzayxcbiAgICB9KTtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YXRlTWFjaGluZS5ncmFudFJlYWQocm9sZSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ3N0YXRlczpMaXN0RXhlY3V0aW9ucycsXG4gICAgICAgICAgICAgICdzdGF0ZXM6TGlzdFN0YXRlTWFjaGluZXMnLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgIFJlZjogJ1N0YXRlTWFjaGluZTJFMDFBM0E1JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFtcbiAgICAgICAgICAgICAgJ3N0YXRlczpEZXNjcmliZUV4ZWN1dGlvbicsXG4gICAgICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVTdGF0ZU1hY2hpbmVGb3JFeGVjdXRpb24nLFxuICAgICAgICAgICAgICAnc3RhdGVzOkdldEV4ZWN1dGlvbkhpc3RvcnknLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAnYXJuOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UGFydGl0aW9uJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOnN0YXRlczonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OlJlZ2lvbicsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6ICdBV1M6OkFjY291bnRJZCcsXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzpleGVjdXRpb246JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgNixcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICAgICAnOicsXG4gICAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBSZWY6ICdTdGF0ZU1hY2hpbmUyRTAxQTNBNScsXG4gICAgICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgJzoqJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOkxpc3RBY3Rpdml0aWVzJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpEZXNjcmliZVN0YXRlTWFjaGluZScsXG4gICAgICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVBY3Rpdml0eScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9LFxuICAgICk7XG5cbiAgfSksXG5cbiAgdGVzdCgnQ3JlYXRlZCBzdGF0ZSBtYWNoaW5lIGNhbiBncmFudCB0YXNrIHJlc3BvbnNlIGFjdGlvbnMgdG8gdGhlIHN0YXRlIG1hY2hpbmUnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiB7XG4gICAgICAgIGJpbmQ6ICgpID0+ICh7IHJlc291cmNlQXJuOiAncmVzb3VyY2UnIH0pLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBkZWZpbml0aW9uOiB0YXNrLFxuICAgIH0pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50VGFza1Jlc3BvbnNlKHJvbGUpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgICdzdGF0ZXM6U2VuZFRhc2tTdWNjZXNzJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpTZW5kVGFza0ZhaWx1cmUnLFxuICAgICAgICAgICAgICAnc3RhdGVzOlNlbmRUYXNrSGVhcnRiZWF0JyxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBSZWY6ICdTdGF0ZU1hY2hpbmUyRTAxQTNBNScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdDcmVhdGVkIHN0YXRlIG1hY2hpbmUgY2FuIGdyYW50IGFjdGlvbnMgdG8gdGhlIGV4ZWN1dGlvbnMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiB7XG4gICAgICAgIGJpbmQ6ICgpID0+ICh7IHJlc291cmNlQXJuOiAncmVzb3VyY2UnIH0pLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBkZWZpbml0aW9uOiB0YXNrLFxuICAgIH0pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50RXhlY3V0aW9uKHJvbGUsICdzdGF0ZXM6R2V0RXhlY3V0aW9uSGlzdG9yeScpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiAnc3RhdGVzOkdldEV4ZWN1dGlvbkhpc3RvcnknLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6c3RhdGVzOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UmVnaW9uJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6QWNjb3VudElkJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOmV4ZWN1dGlvbjonLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICA2LFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgICc6JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogJ1N0YXRlTWFjaGluZTJFMDFBM0E1JyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOionLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdDcmVhdGVkIHN0YXRlIG1hY2hpbmUgY2FuIGdyYW50IGFjdGlvbnMgdG8gYSByb2xlJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zLlRhc2soc3RhY2ssICdUYXNrJywge1xuICAgICAgdGFzazoge1xuICAgICAgICBiaW5kOiAoKSA9PiAoeyByZXNvdXJjZUFybjogJ3Jlc291cmNlJyB9KSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3Qgc3RhdGVNYWNoaW5lID0gbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgZGVmaW5pdGlvbjogdGFzayxcbiAgICB9KTtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YXRlTWFjaGluZS5ncmFudChyb2xlLCAnc3RhdGVzOkxpc3RFeGVjdXRpb24nKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogJ3N0YXRlczpMaXN0RXhlY3V0aW9uJyxcbiAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgIFJlc291cmNlOiB7XG4gICAgICAgICAgICAgIFJlZjogJ1N0YXRlTWFjaGluZTJFMDFBM0E1JyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgfSksXG5cbiAgdGVzdCgnSW1wb3J0ZWQgc3RhdGUgbWFjaGluZSBjYW4gZ3JhbnQgc3RhcnQgZXhlY3V0aW9uIHRvIGEgcm9sZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZUFybiA9ICdhcm46YXdzOnN0YXRlczo6Om15LXN0YXRlLW1hY2hpbmUnO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lLmZyb21TdGF0ZU1hY2hpbmVBcm4oc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCBzdGF0ZU1hY2hpbmVBcm4pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50U3RhcnRFeGVjdXRpb24ocm9sZSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogJ3N0YXRlczpTdGFydEV4ZWN1dGlvbicsXG4gICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICBSZXNvdXJjZTogc3RhdGVNYWNoaW5lQXJuLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyxcbiAgICAgIH0sXG4gICAgICBQb2xpY3lOYW1lOiAnUm9sZURlZmF1bHRQb2xpY3k1RkZCN0RBQicsXG4gICAgICBSb2xlczogW1xuICAgICAgICB7XG4gICAgICAgICAgUmVmOiAnUm9sZTFBQkNDNUYwJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ0ltcG9ydGVkIHN0YXRlIG1hY2hpbmUgY2FuIGdyYW50IHJlYWQgYWNjZXNzIHRvIGEgcm9sZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZUFybiA9ICdhcm46YXdzOnN0YXRlczo6Om15LXN0YXRlLW1hY2hpbmUnO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lLmZyb21TdGF0ZU1hY2hpbmVBcm4oc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCBzdGF0ZU1hY2hpbmVBcm4pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50UmVhZChyb2xlKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOkxpc3RFeGVjdXRpb25zJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpMaXN0U3RhdGVNYWNoaW5lcycsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHN0YXRlTWFjaGluZUFybixcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOkRlc2NyaWJlRXhlY3V0aW9uJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpEZXNjcmliZVN0YXRlTWFjaGluZUZvckV4ZWN1dGlvbicsXG4gICAgICAgICAgICAgICdzdGF0ZXM6R2V0RXhlY3V0aW9uSGlzdG9yeScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHtcbiAgICAgICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICdhcm46JyxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiAnQVdTOjpQYXJ0aXRpb24nLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICc6c3RhdGVzOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6UmVnaW9uJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOicsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogJ0FXUzo6QWNjb3VudElkJyxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAnOmV4ZWN1dGlvbjoqJyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOkxpc3RBY3Rpdml0aWVzJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpEZXNjcmliZVN0YXRlTWFjaGluZScsXG4gICAgICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVBY3Rpdml0eScsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6ICcqJyxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9LFxuICAgICk7XG4gIH0pLFxuXG4gIHRlc3QoJ0ltcG9ydGVkIHN0YXRlIG1hY2hpbmUgY2FuIHRhc2sgcmVzcG9uc2UgcGVybWlzc2lvbnMgdG8gdGhlIHN0YXRlIG1hY2hpbmUnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBzdGF0ZU1hY2hpbmVBcm4gPSAnYXJuOmF3czpzdGF0ZXM6OjpteS1zdGF0ZS1tYWNoaW5lJztcbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZS5mcm9tU3RhdGVNYWNoaW5lQXJuKHN0YWNrLCAnU3RhdGVNYWNoaW5lJywgc3RhdGVNYWNoaW5lQXJuKTtcbiAgICBjb25zdCByb2xlID0gbmV3IGlhbS5Sb2xlKHN0YWNrLCAnUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdsYW1iZGEuYW1hem9uYXdzLmNvbScpLFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIHN0YXRlTWFjaGluZS5ncmFudFRhc2tSZXNwb25zZShyb2xlKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZSgnQVdTOjpJQU06OlBvbGljeScsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICAnc3RhdGVzOlNlbmRUYXNrU3VjY2VzcycsXG4gICAgICAgICAgICAgICdzdGF0ZXM6U2VuZFRhc2tGYWlsdXJlJyxcbiAgICAgICAgICAgICAgJ3N0YXRlczpTZW5kVGFza0hlYXJ0YmVhdCcsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHN0YXRlTWFjaGluZUFybixcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfSksXG5cbiAgdGVzdCgnSW1wb3J0ZWQgc3RhdGUgbWFjaGluZSBjYW4gZ3JhbnQgYWNjZXNzIHRvIGEgcm9sZScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZUFybiA9ICdhcm46YXdzOnN0YXRlczo6Om15LXN0YXRlLW1hY2hpbmUnO1xuICAgIGNvbnN0IHN0YXRlTWFjaGluZSA9IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lLmZyb21TdGF0ZU1hY2hpbmVBcm4oc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCBzdGF0ZU1hY2hpbmVBcm4pO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2xhbWJkYS5hbWF6b25hd3MuY29tJyksXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgc3RhdGVNYWNoaW5lLmdyYW50KHJvbGUsICdzdGF0ZXM6TGlzdEV4ZWN1dGlvbicpO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiAnc3RhdGVzOkxpc3RFeGVjdXRpb24nLFxuICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgUmVzb3VyY2U6IHN0YXRlTWFjaGluZS5zdGF0ZU1hY2hpbmVBcm4sXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ1Bhc3Mgc2hvdWxkIHJlbmRlciBJbnB1dFBhdGggLyBQYXJhbWV0ZXJzIC8gT3V0cHV0UGF0aCBjb3JyZWN0bHknLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuUGFzcyhzdGFjaywgJ1Bhc3MnLCB7XG4gICAgICBpbnB1dFBhdGg6ICckJyxcbiAgICAgIG91dHB1dFBhdGg6ICckLnN0YXRlJyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgJ2lucHV0LiQnOiAnJCcsXG4gICAgICAgICdzdHJpbmdBcmd1bWVudCc6ICdpbml0YWwtdGFzaycsXG4gICAgICAgICdudW1iZXJBcmd1bWVudCc6IDEyMyxcbiAgICAgICAgJ2Jvb2xlYW5Bcmd1bWVudCc6IHRydWUsXG4gICAgICAgICdhcnJheUFyZ3VtZW50JzogWydhJywgJ2InLCAnYyddLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBjb25zdCB0YXNrU3RhdGUgPSB0YXNrLnRvU3RhdGVKc29uKCk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHRhc2tTdGF0ZSkudG9TdHJpY3RFcXVhbCh7XG4gICAgICBFbmQ6IHRydWUsXG4gICAgICBJbnB1dFBhdGg6ICckJyxcbiAgICAgIE91dHB1dFBhdGg6ICckLnN0YXRlJyxcbiAgICAgIFBhcmFtZXRlcnM6XG4gICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgJ2lucHV0LiQnOiAnJCcsXG4gICAgICAgICAgICAgICAnc3RyaW5nQXJndW1lbnQnOiAnaW5pdGFsLXRhc2snLFxuICAgICAgICAgICAgICAgJ251bWJlckFyZ3VtZW50JzogMTIzLFxuICAgICAgICAgICAgICAgJ2Jvb2xlYW5Bcmd1bWVudCc6IHRydWUsXG4gICAgICAgICAgICAgICAnYXJyYXlBcmd1bWVudCc6IFsnYScsICdiJywgJ2MnXSxcbiAgICAgICAgICAgICB9LFxuICAgICAgVHlwZTogJ1Bhc3MnLFxuICAgICAgQ29tbWVudDogdW5kZWZpbmVkLFxuICAgICAgUmVzdWx0OiB1bmRlZmluZWQsXG4gICAgICBSZXN1bHRQYXRoOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gIH0pLFxuXG4gIHRlc3QoJ3BhcmFtZXRlcnMgY2FuIGJlIHNlbGVjdGVkIGZyb20gdGhlIGlucHV0IHdpdGggYSBwYXRoJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgdGFzayA9IG5ldyBzdGVwZnVuY3Rpb25zLlBhc3Moc3RhY2ssICdQYXNzJywge1xuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBpbnB1dDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC5teUZpZWxkJyksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IHRhc2tTdGF0ZSA9IHRhc2sudG9TdGF0ZUpzb24oKTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3QodGFza1N0YXRlKS50b0VxdWFsKHtcbiAgICAgIEVuZDogdHJ1ZSxcbiAgICAgIFBhcmFtZXRlcnM6XG4gICAgICB7ICdpbnB1dC4kJzogJyQubXlGaWVsZCcgfSxcbiAgICAgIFR5cGU6ICdQYXNzJyxcbiAgICB9KTtcbiAgfSksXG5cbiAgdGVzdCgnU3RhdGUgbWFjaGluZXMgbXVzdCBkZXBlbmQgb24gdGhlaXIgcm9sZXMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCB0YXNrID0gbmV3IHN0ZXBmdW5jdGlvbnMuVGFzayhzdGFjaywgJ1Rhc2snLCB7XG4gICAgICB0YXNrOiB7XG4gICAgICAgIGJpbmQ6ICgpID0+ICh7XG4gICAgICAgICAgcmVzb3VyY2VBcm46ICdyZXNvdXJjZScsXG4gICAgICAgICAgcG9saWN5U3RhdGVtZW50czogW1xuICAgICAgICAgICAgbmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICByZXNvdXJjZXM6IFsncmVzb3VyY2UnXSxcbiAgICAgICAgICAgICAgYWN0aW9uczogWydsYW1iZGE6SW52b2tlRnVuY3Rpb24nXSxcbiAgICAgICAgICAgIH0pLFxuICAgICAgICAgIF0sXG4gICAgICAgIH0pLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoc3RhY2ssICdTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBkZWZpbml0aW9uOiB0YXNrLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6U3RlcEZ1bmN0aW9uczo6U3RhdGVNYWNoaW5lJywge1xuICAgICAgRGVwZW5kc09uOiBbXG4gICAgICAgICdTdGF0ZU1hY2hpbmVSb2xlRGVmYXVsdFBvbGljeURGMUU2NjA3JyxcbiAgICAgICAgJ1N0YXRlTWFjaGluZVJvbGVCODQwNDMxRCcsXG4gICAgICBdLFxuICAgIH0sIFJlc291cmNlUGFydC5Db21wbGV0ZURlZmluaXRpb24pO1xuICB9KTtcblxufSk7XG4iXX0=