"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('Activity', () => {
    test('instantiate Activity', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new stepfunctions.Activity(stack, 'Activity');
        // THEN
        expect(stack).toHaveResource('AWS::StepFunctions::Activity', {
            Name: 'Activity',
        });
    });
    test('Activity exposes metrics', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const activity = new stepfunctions.Activity(stack, 'Activity');
        // THEN
        const sharedMetric = {
            period: cdk.Duration.minutes(5),
            namespace: 'AWS/States',
            dimensions: { ActivityArn: { Ref: 'Activity04690B0A' } },
        };
        expect((stack.resolve(activity.metricRunTime()))).toEqual({
            ...sharedMetric,
            metricName: 'ActivityRunTime',
            statistic: 'Average',
        });
        expect(stack.resolve(activity.metricFailed())).toEqual({
            ...sharedMetric,
            metricName: 'ActivitiesFailed',
            statistic: 'Sum',
        });
    });
    test('Activity can grant permissions to a role', () => {
        // GIVEN
        const stack = new cdk.Stack();
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        const activity = new stepfunctions.Activity(stack, 'Activity');
        // WHEN
        activity.grant(role, 'states:SendTaskSuccess');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith(assert_1.objectLike({
                    Action: 'states:SendTaskSuccess',
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'Activity04690B0A',
                    },
                })),
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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