"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachine = exports.LogLevel = exports.StateMachineType = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const state_graph_1 = require("./state-graph");
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Two types of state machines are available in AWS Step Functions: EXPRESS AND STANDARD.
 *
 * @default STANDARD
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-standard-vs-express.html
 */
var StateMachineType;
(function (StateMachineType) {
    StateMachineType["EXPRESS"] = "EXPRESS";
    StateMachineType["STANDARD"] = "STANDARD";
})(StateMachineType = exports.StateMachineType || (exports.StateMachineType = {}));
/**
 * Defines which category of execution history events are logged.
 *
 * @default ERROR
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html
 */
var LogLevel;
(function (LogLevel) {
    LogLevel["OFF"] = "OFF";
    LogLevel["ALL"] = "ALL";
    LogLevel["ERROR"] = "ERROR";
    LogLevel["FATAL"] = "FATAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * A new or imported state machine.
 */
class StateMachineBase extends core_1.Resource {
    /**
     * Import a state machine.
     */
    static fromStateMachineArn(scope, id, stateMachineArn) {
        class Import extends StateMachineBase {
            constructor() {
                super(...arguments);
                this.stateMachineArn = stateMachineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant the given identity permissions to start an execution of this state machine.
     */
    grantStartExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions to read results from state machine.
     */
    grantRead(identity) {
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListExecutions',
                'states:ListStateMachines',
            ],
            resourceArns: [this.stateMachineArn],
        });
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:DescribeExecution',
                'states:DescribeStateMachineForExecution',
                'states:GetExecutionHistory',
            ],
            resourceArns: [`${this.executionArn()}:*`],
        });
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListActivities',
                'states:DescribeStateMachine',
                'states:DescribeActivity',
            ],
            resourceArns: ['*'],
        });
    }
    /**
     * Grant the given identity task response permissions on a state machine.
     */
    grantTaskResponse(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:SendTaskSuccess',
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
            ],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions on all executions of the state machine.
     */
    grantExecution(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [`${this.executionArn()}:*`],
        });
    }
    /**
     * Grant the given identity custom permissions.
     */
    grant(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Returns the pattern for the execution ARN's of the state machine
     */
    executionArn() {
        return core_1.Stack.of(this).formatArn({
            resource: 'execution',
            service: 'states',
            resourceName: core_1.Arn.parse(this.stateMachineArn, ':').resourceName,
            sep: ':',
        });
    }
}
/**
 * Define a StepFunctions State Machine.
 */
class StateMachine extends StateMachineBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stateMachineName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        const graph = new state_graph_1.StateGraph(props.definition.startState, `State Machine ${id} definition`);
        graph.timeout = props.timeout;
        this.stateMachineType = props.stateMachineType ? props.stateMachineType : StateMachineType.STANDARD;
        const resource = new stepfunctions_generated_1.CfnStateMachine(this, 'Resource', {
            stateMachineName: this.physicalName,
            stateMachineType: props.stateMachineType ? props.stateMachineType : undefined,
            roleArn: this.role.roleArn,
            definitionString: core_1.Stack.of(this).toJsonString(graph.toGraphJson()),
            loggingConfiguration: props.logs ? this.buildLoggingConfiguration(props.logs) : undefined,
            tracingConfiguration: props.tracingEnabled ? this.buildTracingConfiguration() : undefined,
        });
        resource.node.addDependency(this.role);
        for (const statement of graph.policyStatements) {
            this.addToRolePolicy(statement);
        }
        this.stateMachineName = this.getResourceNameAttribute(resource.attrName);
        this.stateMachineArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'stateMachine',
            resourceName: this.physicalName,
            sep: ':',
        });
    }
    /**
     * Add the given statement to the role's policy.
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Return the given named metric for this State Machine's executions.
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { StateMachineArn: this.stateMachineArn },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of executions that failed.
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.metric('ExecutionsFailed', props);
    }
    /**
     * Metric for the number of executions that were throttled.
     *
     * @default sum over 5 minutes
     */
    metricThrottled(props) {
        return this.metric('ExecutionThrottled', props);
    }
    /**
     * Metric for the number of executions that were aborted.
     *
     * @default sum over 5 minutes
     */
    metricAborted(props) {
        return this.metric('ExecutionsAborted', props);
    }
    /**
     * Metric for the number of executions that succeeded.
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.metric('ExecutionsSucceeded', props);
    }
    /**
     * Metric for the number of executions that succeeded.
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.metric('ExecutionsTimedOut', props);
    }
    /**
     * Metric for the number of executions that were started.
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.metric('ExecutionsStarted', props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes.
     *
     * @default sum over 5 minutes
     */
    metricTime(props) {
        return this.metric('ExecutionTime', props);
    }
    buildLoggingConfiguration(logOptions) {
        // https://docs.aws.amazon.com/step-functions/latest/dg/cw-logs.html#cloudwatch-iam-policy
        this.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogDelivery',
                'logs:GetLogDelivery',
                'logs:UpdateLogDelivery',
                'logs:DeleteLogDelivery',
                'logs:ListLogDeliveries',
                'logs:PutResourcePolicy',
                'logs:DescribeResourcePolicies',
                'logs:DescribeLogGroups',
            ],
            resources: ['*'],
        }));
        return {
            destinations: [{
                    cloudWatchLogsLogGroup: { logGroupArn: logOptions.destination.logGroupArn },
                }],
            includeExecutionData: logOptions.includeExecutionData,
            level: logOptions.level || 'ERROR',
        };
    }
    buildTracingConfiguration() {
        this.addToRolePolicy(new iam.PolicyStatement({
            // https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#xray-permissions-resources
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            enabled: true,
        };
    }
}
exports.StateMachine = StateMachine;
//# sourceMappingURL=data:application/json;base64,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