"""
Definition of Avro topics and subjects given a schema name.

It uses the fully qualified name of a schema according to the strategy
"io.confluent.kafka.serializers.subject.TopicRecordNameStrategy".
It is used to employ multiple schemas per topic.
See [https://www.confluent.io/blog/put-several-event-types-kafka-topic/]
"""

__author__ = "Nicola Bova"
__copyright__ = "Copyright 2018, Jaumo GmbH"
__email__ = "nicola.bova@jaumo.com"


class AvroNaming:
    """
    Returns topic and subject names given a schema fully qualified name
    """

    autogenerated_key_doc = "The autogenerated Kafka key for topic"

    @classmethod
    def get_subject_name_for_value(
            cls, base_namespace: str, schema_fully_qualified_name: str) -> str:
        """
        Returns the Avro subject name given the schema fully qualified name.

        The fully qualified name of a schema is generated according to
        "io.confluent.kafka.serializers.subject.TopicRecordNameStrategy"

        :param base_namespace: The schemas base namespace
        :param schema_fully_qualified_name: Fully qualified name of the schema
        :return: The subject name
        """
        topic = cls.get_topic(base_namespace, schema_fully_qualified_name)
        return topic + '-' + schema_fully_qualified_name

    @classmethod
    def get_subject_name_for_key(
            cls, base_namespace: str, schema_fully_qualified_name: str) -> str:
        """
        Returns the Avro subject name given the schema fully qualified name.

        The fully qualified name of a schema is generated according to
        "io.confluent.kafka.serializers.subject.TopicRecordNameStrategy"

        # It uses the "io.confluent.kafka.serializers.subject.TopicNameStrategy",
        # i.e. "<topic>-key"

        :param base_namespace: The schemas base namespace
        :param schema_fully_qualified_name: Fully qualified name of the schema
        :return: The subject name
        """
        topic = cls.get_topic(base_namespace, schema_fully_qualified_name)
        return topic + '-' + schema_fully_qualified_name
        # return topic + '-key'

    @classmethod
    def get_topic(cls, base_namespace: str, schema_fully_qualified_name: str) -> str:
        """
        Get the topic name given the schema.

        Topic name is the last two parents before schema name, e.g.

        'com.jaumo.event.domain.user.users.RoleAssigned' -> 'user.users'

        :param base_namespace: The schemas base namespace
        :param schema_fully_qualified_name: Fully qualified name of the schema
        :return: The topic name
        """
        return '.'.join(
            schema_fully_qualified_name.replace(base_namespace + '.', '').split('.')[:-1]
        )

    @classmethod
    def get_key_fully_qualified_name(cls, schema_fully_qualified_name: str) -> str:
        """
        Get the key name given the schema name.

        :param schema_fully_qualified_name: Fully qualified name of the schema
        :return: The key fully qualified name
        """
        return '.'.join(
            schema_fully_qualified_name.split('.')[:-1] + ['Key']
        )
