"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        const lambda = new transliterator_1.Transliterator(this, 'Resource', {
            description: `Creates ${props.language} documentation from jsii-enabled npm packages`,
            environment: { ...environment, TARGET_LANGUAGE: props.language.toString() },
            logRetention: (_b = props.logRetention) !== null && _b !== void 0 ? _b : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.ISOLATED },
        });
        this.function = lambda;
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.function);
        // The handler reads & writes to this bucket.
        bucket.grantRead(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}${constants.NOT_SUPPORTED_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
        props.monitoring.watchful.watchLambdaFunction('Transliterator Function', lambda);
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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