"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 })
            .addCatch(new aws_stepfunctions_1.Pass(this, 'Failed to add to catalog.json', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue));
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue.next(new aws_stepfunctions_1.Fail(this, 'Fail')))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => new tasks.LambdaInvoke(this, `Generate ${language} docs`, {
            lambdaFunction: new transliterator_1.Transliterator(this, `DocGen-${language}`, { ...props, language }).function,
            outputPath: '$.result',
            resultSelector: {
                result: {
                    'language': language,
                    'success.$': '$.Payload',
                },
            },
        }).addRetry({ interval: core_1.Duration.seconds(30) })
            .addCatch(new aws_stepfunctions_1.Pass(this, `Failed ${language}`, {
            parameters: {
                'error.$': 'States.StringToJson($.Cause)',
                language,
            },
        }))))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        this.reprocessAllFunction = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.bucket.grantRead(this.reprocessAllFunction);
        this.stateMachine.grantStartExecution(this.reprocessAllFunction);
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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