# -*- coding: utf-8 -*-
"""
Created on Wed Sep 22 10:51:37 2021

@author: jpeacock
"""

import unittest
import xarray as xr
import numpy as np

from mt_metadata.transfer_functions.core import TF, TFError


class TestTFPeriodInput(unittest.TestCase):
    def setUp(self):
        self.n_period = 20
        self.period = np.logspace(-3, 3, self.n_period)

        self.tf = TF()
        self.tf.period = self.period

    def test_period_shape(self):

        self.assertEqual(self.tf.period.size, self.n_period)

    def test_tf_shape(self):
        self.assertEqual(self.tf.dataset.transfer_function.shape[0], self.n_period)

    def test_isp_shape(self):
        self.assertEqual(self.tf.dataset.inverse_signal_power.shape[0], self.n_period)

    def test_res_shape(self):
        self.assertEqual(self.tf.dataset.residual_covariance.shape[0], self.n_period)


class TestTFImpedanceInput(unittest.TestCase):
    def setUp(self):
        self.n_period = 20
        self.n_fail = 22
        self.period = np.logspace(-3, 3, self.n_period)
        self.fail_period = np.logspace(-3, 3, self.n_fail)
        self.test_z = xr.DataArray(
            data=np.random.rand(self.n_period, 2, 2),
            dims=["period", "output", "input"],
            coords={
                "period": self.period,
                "output": ["ex", "ey"],
                "input": ["hx", "hy"],
            },
            name="impedance",
        )
        self.new_z = np.random.rand(self.n_period, 2, 2)

        self.fail_z = xr.DataArray(
            data=np.random.rand(self.n_fail, 2, 2),
            dims=["period", "output", "input"],
            coords={
                "period": self.fail_period,
                "output": ["ex", "ey"],
                "input": ["hx", "hy"],
            },
            name="impedance",
        )

        self.tf = TF()
        self.tf.impedance = self.test_z

    def test_has_impedance(self):

        self.assertTrue(self.tf.has_impedance())

    def test_shape(self):
        self.assertEqual(self.tf.impedance.data.shape, (self.n_period, 2, 2))

    def test_xarray(self):
        self.assertTrue(np.all(self.tf.impedance == self.test_z))

    def test_fail(self):
        def set_value(value):
            self.tf.impedance = value

        self.assertRaises(TFError, set_value, self.fail_z)

    def test_set_from_array(self):
        self.tf.impedance = self.new_z
        self.assertTrue(np.all(self.tf.impedance.data == self.new_z))


class TestTFTipperInput(unittest.TestCase):
    def setUp(self):
        self.n_period = 20
        self.n_fail = 22
        self.period = np.logspace(-3, 3, self.n_period)
        self.fail_period = np.logspace(-3, 3, self.n_fail)
        self.test_t = xr.DataArray(
            data=np.random.rand(self.n_period, 1, 2),
            dims=["period", "output", "input"],
            coords={"period": self.period, "output": ["hz"], "input": ["hx", "hy"]},
            name="tipper",
        )
        self.new_t = np.random.rand(self.n_period, 1, 2)
        self.fail_t = xr.DataArray(
            data=np.random.rand(self.n_fail, 1, 2),
            dims=["period", "output", "input"],
            coords={
                "period": self.fail_period,
                "output": ["hz"],
                "input": ["hx", "hy"],
            },
            name="tipper",
        )

        self.tf = TF()
        self.tf.tipper = self.test_t

    def test_has_tipper(self):

        self.assertTrue(self.tf.has_tipper())

    def test_shape(self):
        self.assertEqual(self.tf.tipper.data.shape, (self.n_period, 1, 2))

    def test_xarray(self):
        self.assertTrue(np.all(self.tf.tipper == self.test_t))

    def test_fail(self):
        def set_value(value):
            self.tf.tipper = value

        self.assertRaises(TFError, set_value, self.fail_t)

    def test_set_from_array(self):
        self.tf.tipper = self.new_t
        self.assertTrue(np.all(self.tf.tipper.data == self.new_t))


class TestTFISPInput(unittest.TestCase):
    def setUp(self):
        self.n_period = 20
        self.n_fail = 22
        self.period = np.logspace(-3, 3, self.n_period)
        self.fail_period = np.logspace(-3, 3, self.n_fail)
        self.test_t = xr.DataArray(
            data=np.random.rand(self.n_period, 2, 2),
            dims=["period", "output", "input"],
            coords={
                "period": self.period,
                "output": ["hx", "hy"],
                "input": ["hx", "hy"],
            },
            name="inverse_signal_power",
        )
        self.new_t = np.random.rand(self.n_period, 2, 2)
        self.fail_t = xr.DataArray(
            data=np.random.rand(self.n_fail, 2, 2),
            dims=["period", "output", "input"],
            coords={
                "period": self.fail_period,
                "output": ["hx", "hy"],
                "input": ["hx", "hy"],
            },
            name="inverse_signal_power",
        )

        self.tf = TF()
        self.tf.inverse_signal_power = self.test_t

    def test_has_isp(self):

        self.assertTrue(self.tf.has_inverse_signal_power())

    def test_shape(self):
        self.assertEqual(self.tf.inverse_signal_power.data.shape, (self.n_period, 2, 2))

    def test_xarray(self):
        self.assertTrue(
            np.all(self.tf.inverse_signal_power.data == self.test_t) == True
        )

    def test_fail(self):
        def set_value(value):
            self.tf.inverse_signal_power = value

        self.assertRaises(TFError, set_value, self.fail_t)

    def test_set_from_array(self):
        self.tf.inverse_signal_power = self.new_t
        self.assertTrue(np.all(self.tf.inverse_signal_power.data == self.new_t) == True)


class TestTFResidualInput(unittest.TestCase):
    def setUp(self):
        self.n_period = 20
        self.n_fail = 22
        self.period = np.logspace(-3, 3, self.n_period)
        self.fail_period = np.logspace(-3, 3, self.n_fail)
        self.test_t = xr.DataArray(
            data=np.random.rand(self.n_period, 3, 3),
            dims=["period", "output", "input"],
            coords={
                "period": self.period,
                "output": ["ex", "ey", "hz"],
                "input": ["ex", "ey", "hz"],
            },
            name="inverse_signal_power",
        )
        self.new_t = np.random.rand(self.n_period, 3, 3)
        self.fail_t = xr.DataArray(
            data=np.random.rand(self.n_fail, 3, 3),
            dims=["period", "output", "input"],
            coords={
                "period": self.fail_period,
                "output": ["ex", "ey", "hz"],
                "input": ["ex", "ey", "hz"],
            },
            name="inverse_signal_power",
        )

        self.tf = TF()
        self.tf.residual_covariance = self.test_t

    def test_has_residual(self):

        self.assertTrue(self.tf.has_residual_covariance())

    def test_shape(self):
        self.assertEqual(self.tf.residual_covariance.data.shape, (self.n_period, 3, 3))

    def test_xarray(self):
        self.assertTrue(np.all(self.tf.residual_covariance.data == self.test_t) == True)

    def test_fail(self):
        def set_value(value):
            self.tf.residual_covariance = value

        self.assertRaises(TFError, set_value, self.fail_t)

    def test_set_from_array(self):
        self.tf.residual_covariance = self.new_t
        self.assertTrue(np.all(self.tf.residual_covariance.data == self.new_t) == True)


# =============================================================================
# run
# =============================================================================
if __name__ == "__main__":
    unittest.main()
