import * as React from 'react';

import classNames from 'classnames';

import { IDisposable } from '@lumino/disposable';

import { Signal, ISignal } from '@lumino/signaling';

import { VDomModel, VDomRenderer } from '@jupyterlab/apputils';

export interface IDatabaseSummaryIModel extends IDisposable {
  navigateToTable: ISignal<this, string>;
  navigateToCustomQuery: ISignal<this, void>;
}

export class DatabaseSummaryModel
  extends VDomModel
  implements IDatabaseSummaryIModel
{
  constructor(tables: Array<string>) {
    super();
    this.tables = tables;
    this.onNavigateToTable = this.onNavigateToTable.bind(this);
    this.onNavigateToCustomQuery = this.onNavigateToCustomQuery.bind(this);
  }

  get navigateToTable(): ISignal<this, string> {
    return this._navigateToTable;
  }

  get navigateToCustomQuery(): ISignal<this, void> {
    return this._navigateToCustomQuery;
  }

  onNavigateToTable(tableName: string): void {
    this._navigateToTable.emit(tableName);
  }

  onNavigateToCustomQuery(): void {
    this._navigateToCustomQuery.emit(void 0);
  }

  readonly tables: Array<string>;
  private readonly _navigateToTable = new Signal<this, string>(this);
  private readonly _navigateToCustomQuery = new Signal<this, void>(this);
}

export class DatabaseSummaryWidget extends VDomRenderer<DatabaseSummaryModel> {
  constructor(model: DatabaseSummaryModel) {
    super(model);
    this.addClass('p-Sql-DatabaseSummary-Container');
  }

  static withModel(model: DatabaseSummaryModel): DatabaseSummaryWidget {
    const tableList = new DatabaseSummaryWidget(model);
    return tableList;
  }

  render(): JSX.Element | null {
    if (!this.model) {
      return null;
    } else {
      const { tables, onNavigateToTable, onNavigateToCustomQuery } = this.model;
      return (
        <TableList
          tableNames={tables}
          onNavigateToTable={onNavigateToTable}
          onNavigateToCustomQuery={onNavigateToCustomQuery}
        />
      );
    }
  }
}

namespace TableList {
  export type Props = {
    tableNames: Array<string>;
    onNavigateToTable: (tableName: string) => void;
    onNavigateToCustomQuery: () => void;
  };

  export type State = {
    selectedItem: number | null;
  };
}

class TableList extends React.Component<TableList.Props, TableList.State> {
  constructor(props: TableList.Props) {
    super(props);
    this.state = {
      selectedItem: null
    };
    this.onTableItemClick = this.onTableItemClick.bind(this);
  }

  onTableItemClick(itemNumber: number) {
    this.setState({ selectedItem: itemNumber });
  }

  render() {
    const { tableNames, onNavigateToTable, onNavigateToCustomQuery } =
      this.props;
    const { selectedItem } = this.state;
    const tableItems = tableNames.map((tableName, i) => (
      <TableListItem
        tableName={tableName}
        key={i}
        onClick={() => this.onTableItemClick(i)}
        onDoubleClick={() => onNavigateToTable(tableName)}
        selected={i === selectedItem}
      />
    ));
    return (
      <div className="p-Sql-TableList-container">
        <ul className="p-Sql-TableList-content">
          <ListHeader headerText="Actions" />
          <CustomQueryItem onClick={onNavigateToCustomQuery} />
          <ListHeader headerText="Tables" />
          {tableItems}
        </ul>
      </div>
    );
  }
}

namespace TableListItem {
  export type Props = {
    tableName: string;
    selected: boolean;
    onClick: () => void;
    onDoubleClick: () => void;
  };
}

class TableListItem extends React.Component<TableListItem.Props> {
  render() {
    const { tableName, onClick, onDoubleClick, selected } = this.props;
    const classes = classNames('jp-DirListing-item', {
      'jp-mod-selected': selected
    });
    return (
      <li
        onClick={onClick}
        onDoubleClick={onDoubleClick}
        className={classes}
        title={tableName}
      >
        <span className="jp-DirListing-itemIcon jp-MaterialIcon jp-SpreadsheetIcon" />
        <span className="jp-DirListing-itemText">{tableName}</span>
      </li>
    );
  }
}

namespace CustomQueryItem {
  export type Props = {
    onClick: () => void;
  };
}

class CustomQueryItem extends React.Component<CustomQueryItem.Props> {
  render() {
    const { onClick } = this.props;
    return (
      <li
        onClick={onClick}
        className="jp-DirListing-item"
        title="Custom SQL query"
      >
        <span className="jp-DirListing-itemIcon jp-MaterialIcon jp-CodeConsoleIcon" />
        <span className="jp-DirListing-itemText">Custom SQL query</span>
      </li>
    );
  }
}

namespace ListHeader {
  export type Props = {
    headerText: string;
  };
}

class ListHeader extends React.Component<ListHeader.Props> {
  render() {
    const { headerText } = this.props;
    return <li className="p-Sql-TableList-header">{headerText}</li>;
  }
}
