"""Transform a python class into a corresponding pydantic model.

The :class:`BackendClass` model in this module generates subclasses based upon
a python class (similarly as the
:class:`~demessaging.backend.function.BackendFunction` does it for functions).
"""
from __future__ import annotations

import inspect
import json
from functools import partial
from textwrap import dedent
from typing import (
    TYPE_CHECKING,
    Any,
    Callable,
    ClassVar,
    Dict,
    List,
    Optional,
    Type,
    Union,
    cast,
)

import docstring_parser
from pydantic import Field  # pylint: disable=no-name-in-module
from pydantic import BaseModel, create_model
from pydantic.json import (  # pylint: disable=no-name-in-module
    custom_pydantic_encoder,
)

from demessaging.backend import utils
from demessaging.backend.function import BackendFunction, BackendFunctionConfig
from demessaging.config import ClassConfig


class BackendClassConfig(ClassConfig):
    """Configuration class for a backend module class."""

    models: Dict[str, Type[BackendFunction]] = Field(
        default_factory=dict,
        description=(
            "Mapping of method name to the function model for the "
            "methods of this class"
        ),
    )

    Class: Type[object] = Field(
        description="The class that corresponds to this config."
    )

    class_name: str = Field(description="Name of the model class")

    @property
    def method_configs(self) -> List[BackendFunctionConfig]:
        """Get a list of the method configs."""
        return [model.backend_config for model in self.models.values()]

    def update_from_cls(self) -> None:
        """Update the config from the corresponding function."""
        Class = self.Class
        if not self.name:
            self.name = Class.__name__ or ""
        if not self.doc:
            self.doc = dedent(inspect.getdoc(Class) or "")
        if not self.init_doc:
            self.init_doc = dedent(inspect.getdoc(Class.__init__) or "")
        if not self.signature:
            self.signature = inspect.signature(Class.__init__)


class BackendClass(BaseModel):
    """A basis for class models

    Do not directly instantiate from this class, rather use the
    :meth:`create_model` method.
    """

    class Config:
        json_encoders = {int: json.dumps}

    backend_config: ClassVar[BackendClassConfig]

    @property
    def return_model(self) -> Type[BaseModel]:
        """The return model of the member function."""
        return self.function.return_model

    if TYPE_CHECKING:
        # added properties for subclasses generated by create_model
        function: BackendFunction
        class_name: str

    def __call__(self) -> BaseModel:

        kws = utils.get_kws(self.backend_config.signature, self)

        func_kws = utils.get_kws(
            self.function.backend_config.signature, self.function
        )

        ini: Any = self.backend_config.Class(**kws)  # type: ignore
        func_name = self.function.func_name

        ret = getattr(ini, func_name)(**func_kws)

        # now update the function model and return it
        function: BackendFunction = self.function  # type: ignore
        return function.return_model.parse_obj(ret)

    @classmethod
    def create_model(
        cls,
        Class,
        config: Optional[ClassConfig] = None,
        methods: Optional[
            List[Union[Type[BackendFunction], Callable, str]]
        ] = None,
        class_name: Optional[str] = None,
        **kwargs: Any,
    ) -> Type[BackendClass]:
        """Generate a pydantic model from a class.    Parameters
        ----------
        func: type
            A class
        config: ClassConfig, optional
            The configuration to use. If given, this overrides the
            ``__pulsar_config__`` of the given `Class`
        methods: list of methods, optional
            A list of methods or model classes generated with
            :func:`FunctionModel`. This overrides the methods in `config` or
            the ``__pulsar_config__`` attribute of `Class`
        class_name: str, optional
            The name for the generated subclass of :class:`pydantic.BaseModel`.
            If not given, the name of `Class` is used
        ``**kwargs``
            Any other parameter for the :func:`pydantic.create_model` function

        Returns
        -------
        Subclass of BackendClass
            The newly generated model that represents this class.
        """
        sig = inspect.signature(Class.__init__)
        docstring = docstring_parser.parse(Class.__doc__)
        init_docstring = docstring_parser.parse(Class.__init__.__doc__)
        docstring.params.extend(init_docstring.params)
        docstring.meta.extend(init_docstring.meta)

        if config is None:
            config = getattr(Class, "__pulsar_config__", ClassConfig())
        config = cast(ClassConfig, config)

        name = Class.__name__
        if not class_name:
            class_name = utils.snake_to_camel("Class", name)

        config = BackendClassConfig(
            Class=Class, class_name=class_name, **config.copy(deep=True).dict()
        )

        fields = utils.get_fields(name, sig, docstring, config)
        fields["class_name"] = fields.pop("func_name")

        if "function" in fields:
            raise ValueError(
                f"`function` must not be an init parameter for {name}!"
            )

        if methods:
            pass
        elif config.methods:
            methods = list(config.methods)
        if not methods:
            names_members = inspect.getmembers(
                Class, predicate=inspect.isfunction
            )
            methods = [t[0] for t in names_members if not t[0].startswith("_")]

        if not methods:
            raise ValueError("No methods of the class have been specified!")

        for method in methods:
            if inspect.isclass(method) and issubclass(method, BackendFunction):  # type: ignore  # noqa: E501
                method_name: str = method.backend_config.name  # type: ignore
                FuncModel: Type[BackendFunction] = cast(
                    Type[BackendFunction], method
                )
            elif callable(method):
                method_name = cast(str, method.__name__)
                FuncModel = BackendFunction.create_model(
                    cast(Callable, method),
                    class_name=utils.snake_to_camel(
                        "Meth", class_name, method_name
                    ),
                )
            else:
                method_name = method
                FuncModel = BackendFunction.create_model(
                    getattr(Class, method_name),
                    class_name=utils.snake_to_camel(
                        "Meth", class_name, method_name
                    ),
                )
            if method_name not in config.models:
                config.models[method_name] = FuncModel

        config.methods = list(config.models)

        models = list(config.models.values())
        function_types = models[0]
        for model in models[1:]:
            function_types = Union[function_types, model]  # type: ignore

        fields["function"] = (
            function_types,
            Field(description="The method to call."),
        )
        kwargs.update(fields)

        Model: Type[BackendClass] = create_model(  # type: ignore
            class_name,
            __validators__=config.validators,
            __module__=Class.__module__,
            __base__=cls,
            **kwargs,  # type: ignore
        )

        Model.backend_config = config

        if config.registry.json_encoders:
            # it would be better, to set this via __config__ in create_model,
            # but this is not possible if we use `__base__`
            Model.__config__.json_encoders = config.registry.json_encoders
            Model.__json_encoder__ = partial(
                custom_pydantic_encoder, config.registry.json_encoders
            )

        config.Class = Class
        config.update_from_cls()

        Model.__doc__ = config.doc

        return Model


ClassConfig.update_forward_refs()
