import pytest
import dask
import dask.array as da
import numpy.testing as npt
from daskms import xds_from_storage_ms, xds_from_storage_table
from daskms.experimental.fragments import xds_to_table_fragment, xds_from_table_fragment

# Prevent warning pollution generated by all calls to xds_from_zarr with
# unsupported kwargs.
pytestmark = pytest.mark.filterwarnings(
    "ignore:The following unsupported kwargs were ignored in xds_from_zarr"
)


@pytest.fixture(
    scope="module",
    params=[
        ("DATA_DESC_ID", "FIELD_ID", "SCAN_NUMBER"),
        ("DATA_DESC_ID", "FIELD_ID"),
        ("DATA_DESC_ID",),
    ],
)
def group_cols(request):
    return request.param


# -----------------------------MAIN_TABLE_TESTS--------------------------------


def test_fragment_with_noop(ms, tmp_path_factory, group_cols):
    """Unchanged data_vars must remain the same when read from a fragment."""

    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"

    writes = xds_to_table_fragment(reads, fragment_path, ms, columns=("DATA",))

    dask.compute(writes)

    fragment_reads = xds_from_table_fragment(
        fragment_path,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    for rxds, frxds in zip(reads, fragment_reads):
        assert rxds.equals(frxds), "Datasets not identical."


def test_fragment_with_update(ms, tmp_path_factory, group_cols):
    """Updated data_vars must change when read from a fragment."""

    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"

    updates = [
        xds.assign(**{"DATA": (xds.DATA.dims, da.ones_like(xds.DATA.data))})
        for xds in reads
    ]

    writes = xds_to_table_fragment(updates, fragment_path, ms, columns=("DATA",))

    dask.compute(writes)

    fragment_reads = xds_from_table_fragment(
        fragment_path,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    for frxds in fragment_reads:
        npt.assert_array_equal(1, frxds.DATA.data)


def test_nonoverlapping_parents(ms, tmp_path_factory, group_cols):
    """All updated data_vars must change when read from a fragment."""

    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment0_path = tmp_dir / "fragment0.ms"
    fragment1_path = tmp_dir / "fragment1.ms"

    updates = [
        xds.assign(**{"DATA": (xds.DATA.dims, da.zeros_like(xds.DATA.data))})
        for xds in reads
    ]

    writes = xds_to_table_fragment(updates, fragment0_path, ms, columns=("DATA",))

    dask.compute(writes)

    fragment0_reads = xds_from_table_fragment(
        fragment0_path,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    updates = [
        xds.assign(**{"UVW": (xds.UVW.dims, da.zeros_like(xds.UVW.data))})
        for xds in fragment0_reads
    ]

    writes = xds_to_table_fragment(
        updates, fragment1_path, fragment0_path, columns=("UVW",)
    )

    dask.compute(writes)

    fragment1_reads = xds_from_table_fragment(
        fragment1_path,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    for frxds in fragment1_reads:
        npt.assert_array_equal(0, frxds.DATA.data)
        npt.assert_array_equal(0, frxds.UVW.data)


def test_overlapping_parents(ms, tmp_path_factory, group_cols):
    """Youngest child takes priority if updated data_vars overlap."""
    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment0_path = tmp_dir / "fragment0.ms"
    fragment1_path = tmp_dir / "fragment1.ms"

    updates = [
        xds.assign(**{"DATA": (xds.DATA.dims, da.ones_like(xds.DATA.data))})
        for xds in reads
    ]

    writes = xds_to_table_fragment(updates, fragment0_path, ms, columns=("DATA",))

    dask.compute(writes)

    fragment0_reads = xds_from_table_fragment(
        fragment0_path,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    updates = [
        xds.assign(**{"DATA": (xds.DATA.dims, da.zeros_like(xds.DATA.data))})
        for xds in fragment0_reads
    ]

    writes = xds_to_table_fragment(
        updates, fragment1_path, fragment0_path, columns=("DATA",)
    )

    dask.compute(writes)

    fragment1_reads = xds_from_table_fragment(
        fragment1_path,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    for frxds in fragment1_reads:
        npt.assert_array_equal(0, frxds.DATA.data)


def test_inconsistent_partitioning(ms, tmp_path_factory, group_cols):
    """Raises a ValueError when parititoning would be inconsistent."""
    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"

    writes = xds_to_table_fragment(reads, fragment_path, ms, columns=("DATA",))

    dask.compute(writes)

    with pytest.raises(ValueError, match="consolidate failed"):
        xds_from_table_fragment(
            fragment_path,
            index_cols=("TIME",),
            group_cols=(),
        )


def test_mutate_parent(ms, tmp_path_factory):
    """Raises a ValueError when a fragment would be its own parent."""
    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=("DATA_DESC_ID", "FIELD_ID", "SCAN_NUMBER"),
    )

    with pytest.raises(ValueError, match="store and parent arguments"):
        xds_to_table_fragment(reads, ms, ms, columns=("DATA",))


def test_missing_parent(ms, tmp_path_factory):
    """Raises a ValueError when a fragment is missing a parent."""

    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=("DATA_DESC_ID", "FIELD_ID", "SCAN_NUMBER"),
    )

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"
    missing_parent = tmp_dir / "missing.ms"

    writes = xds_to_table_fragment(
        reads, fragment_path, missing_parent, columns=("DATA",)
    )

    dask.compute(writes)

    with pytest.raises(FileNotFoundError, match="No root/fragment found at"):
        xds_from_table_fragment(
            fragment_path,
            index_cols=("TIME",),
            group_cols=("DATA_DESC_ID", "FIELD_ID", "SCAN_NUMBER"),
        )


def test_datavar_in_parent(ms, tmp_path_factory, group_cols):
    """Datavars not present in the fragment must be read from the parent."""

    reads = xds_from_storage_ms(
        ms,
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"

    writes = xds_to_table_fragment(reads, fragment_path, ms, columns=("DATA",))

    dask.compute(writes)

    fragment_reads = xds_from_table_fragment(
        fragment_path,
        columns=("UVW",),  # Not in fragment.
        index_cols=("TIME",),
        group_cols=group_cols,
    )

    for rxds, frxds in zip(reads, fragment_reads):
        npt.assert_array_equal(rxds.UVW.data, frxds.UVW.data)


# ------------------------------SUBTABLE_TESTS---------------------------------


def test_subtable_fragment_with_noop(spw_table, tmp_path_factory):
    """Unchanged data_vars must remain the same when read from a fragment."""

    reads = xds_from_storage_table(spw_table, group_cols=("__row__",))

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"

    writes = xds_to_table_fragment(
        reads, fragment_path, spw_table, columns=("CHAN_FREQ",)
    )

    dask.compute(writes)

    fragment_reads = xds_from_table_fragment(fragment_path, group_cols=("__row__",))

    for rxds, frxds in zip(reads, fragment_reads):
        assert rxds.equals(frxds), "Datasets not identical."


def test_subtable_fragment_with_update(spw_table, tmp_path_factory):
    """Updated data_vars must change when read from a fragment."""

    reads = xds_from_storage_table(spw_table, group_cols=("__row__",))

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"

    updates = [
        xds.assign(
            **{"CHAN_FREQ": (xds.CHAN_FREQ.dims, da.ones_like(xds.CHAN_FREQ.data))}
        )
        for xds in reads
    ]

    writes = xds_to_table_fragment(
        updates, fragment_path, spw_table, columns=("CHAN_FREQ",)
    )

    dask.compute(writes)

    fragment_reads = xds_from_table_fragment(fragment_path, group_cols=("__row__",))

    for frxds in fragment_reads:
        npt.assert_array_equal(1, frxds.CHAN_FREQ.data)


def test_subtable_datavar_in_parent(spw_table, tmp_path_factory):
    """Datavars not present in the fragment must be read from the parent."""

    reads = xds_from_storage_table(spw_table, group_cols=("__row__",))

    tmp_dir = tmp_path_factory.mktemp("fragments")
    fragment_path = tmp_dir / "fragment.ms"

    writes = xds_to_table_fragment(
        reads, fragment_path, spw_table, columns=("CHAN_FREQ",)
    )

    dask.compute(writes)

    fragment_reads = xds_from_table_fragment(
        fragment_path, columns=("NUM_CHAN",), group_cols=("__row__",)
    )

    for rxds, frxds in zip(reads, fragment_reads):
        npt.assert_array_equal(rxds.NUM_CHAN.data, frxds.NUM_CHAN.data)


# -----------------------------------------------------------------------------
