from typing import Optional

import typer

from .client import user_client
from .client.base import BayesGQLClient
from .commands import gear, ssh
from .commands import org
from .error import credential_notfound_error, Error
from bayes.model.file.settings import BayesSettings, BayesEnvConfig
from bayes.usercases.switch_usercase import clean, is_url, is_exist
from .usercases import auth_usecase
from .utils import Utils

app = typer.Typer(
    help="OpenBayes CLI",
    context_settings={"help_option_names": ["-h", "--help"]}
)

app.add_typer(ssh.app, name="ssh")
app.add_typer(gear.app, name="gear")
app.add_typer(org.app, name="org")


@app.callback()
def main(ctx: typer.Context):
    ctx.obj = BayesSettings()


@app.command()
def login(ctx: typer.Context,
          username: str = typer.Argument(
              None, envvar="OPENBAYES_TOKEN", help="用户名 | 令牌"
          )):
    """
    登录

    用法：
      bayes login [用户名 | 令牌] [选项]

    可用选项：
      -h, --help   查看 login 的帮助
    """
    # 使用账户密码登录时：当用户名为 null 的时候要去 credentials 文件中看看 username 是否为null，因为登出的时候 username 是不覆盖为None
    # 如果 username is None，让用户输入，否则直接显示 请输入 xxx 的密码
    bayes_settings = ctx.obj
    default_env: Optional[BayesEnvConfig] = bayes_settings.default_env
    if default_env.token:
        print(f"{default_env.username} 已成功登入 {default_env.name}")
        raise typer.Exit()

    if username is None:
        env_username = default_env.username
        if env_username:
            username = env_username
        else:
            username = input("请输入用户名：")

    gql_client = BayesGQLClient(default_env.graphQL, None)

    if len(username) > 40:
        try:
            result = user_client.login_with_token(gql_client, username)
            bayes_settings.login(result.username, result.token)

            print(f"{result.username} 已成功登入 {default_env.name}")
            raise typer.Exit()
        except Error as e:
            print(e.message)
            raise typer.Exit(code=1)
    else:
        password = typer.prompt(f"请输入 {username} 的密码", hide_input=True)
        try:
            result = user_client.login(gql_client, username, password)
            bayes_settings.login(result.username, result.token)

            print(f"{result.username} 已成功登入 {default_env.name}")
            raise typer.Exit()
        except Error as e:
            print(e.message)
            raise typer.Exit(code=1)


@app.command()
def logout(ctx: typer.Context):
    """
    注销

    用法:
      bayes logout [选项]

    可用选项:
      -h, --help   查看 logout 的帮助
    """
    bayes_settings = ctx.obj
    logout_result = bayes_settings.logout()
    if logout_result:
        print("已成功登出")
    else:
        print("登出失败")


@app.command()
def switch(ctx: typer.Context, name: str,
           endpoint: str = typer.Option(None, "--endpoint", "-e", help="填入 endpoint")):
    """
    切换服务端环境

    用法：
       bayes switch [配置名] [选项]

    可用选项：
      -e, --endpoint string   [可选] 填入 endpoint
      -h, --help              查看 switch 的帮助
    """

    if name == "default" and endpoint:
        print("配置名 default 已存在，且不能被修改，请选择其他配置名")
        raise typer.Exit(code=1)

    bayes_settings = ctx.obj

    if endpoint:
        if not is_url(endpoint):
            print(f"{endpoint} 不是合法的链接")
            raise typer.Exit(code=1)

        # 临时创建一个新的环境，然后判断是否可以访问
        new_env = BayesEnvConfig(name=name, endpoint=clean(endpoint))
        if not is_exist(new_env.graphQL):
            print(f"{endpoint} 无法访问，请再次确认你所输入的链接")
            raise typer.Exit(code=1)

        bayes_settings.add_new_env(name, new_env.endpoint)

    switch_result = bayes_settings.switch_default_env(name)
    if switch_result:
        print(f"已成功切换到 {name}")
    else:
        error = credential_notfound_error(name)
        print(error)


@app.command()
def status(ctx: typer.Context):
    """
    登录信息

    用法：
        bayes status [选项]

    可用选项：
        -h, --help   查看 status 的帮助
    """
    settings: BayesSettings = ctx.obj
    default_env: Optional[BayesEnvConfig] = settings.default_env
    userinfo = auth_usecase.get_default_credential_userinfo()
    if userinfo is not None:
        print(f"当前环境: {default_env.endpoint}")
        print(f"当前组织: {default_env.orgName}")
        print(f"用户名: {default_env.username}")
        print(f"邮箱: {userinfo.email}")
    else:
        print(f"当前环境: {default_env.endpoint}")


@app.command()
def version(ctx: typer.Context):
    version_number = Utils.get_version()
    print(f"openbayes 的版本号为: {version_number}")
