"""The module that defines the ``RunProgramLog`` model.

SPDX-License-Identifier: AGPL-3.0-only OR BSD-3-Clause-Clear
"""
from dataclasses import dataclass, field
from typing import Any, ClassVar, Dict, Optional, Type

import cg_request_args as rqa

from ..utils import to_dict
from .auto_test_step_log_base import AutoTestStepLogBase


@dataclass
class RunProgramLog(AutoTestStepLogBase):
    """The result of a RunProgram step."""

    #: The stdout produced by the step.
    stdout: "str"
    #: The stderr produced by the step.
    stderr: "str"
    #: The exit code of the step.
    exit_code: "int"
    #: The time spend running the step.
    time_spend: "float"

    raw_data: Optional[Dict[str, Any]] = field(init=False, repr=False)

    data_parser: ClassVar = rqa.Lazy(
        lambda: AutoTestStepLogBase.data_parser.parser.combine(
            rqa.FixedMapping(
                rqa.RequiredArgument(
                    "stdout",
                    rqa.SimpleValue.str,
                    doc="The stdout produced by the step.",
                ),
                rqa.RequiredArgument(
                    "stderr",
                    rqa.SimpleValue.str,
                    doc="The stderr produced by the step.",
                ),
                rqa.RequiredArgument(
                    "exit_code",
                    rqa.SimpleValue.int,
                    doc="The exit code of the step.",
                ),
                rqa.RequiredArgument(
                    "time_spend",
                    rqa.SimpleValue.float,
                    doc="The time spend running the step.",
                ),
            )
        ).use_readable_describe(True)
    )

    def to_dict(self) -> Dict[str, Any]:
        res: Dict[str, Any] = {}
        res["stdout"] = to_dict(self.stdout)
        res["stderr"] = to_dict(self.stderr)
        res["exit_code"] = to_dict(self.exit_code)
        res["time_spend"] = to_dict(self.time_spend)
        return res

    @classmethod
    def from_dict(
        cls: Type["RunProgramLog"], d: Dict[str, Any]
    ) -> "RunProgramLog":
        parsed = cls.data_parser.try_parse(d)

        res = cls(
            stdout=parsed.stdout,
            stderr=parsed.stderr,
            exit_code=parsed.exit_code,
            time_spend=parsed.time_spend,
        )
        res.raw_data = d
        return res
