#
#  Copyright © 2021-2024 Mergify SAS
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.
import json

import pytest
import respx

from mergify_cli.stack import push
from mergify_cli.tests import utils as test_utils


@pytest.mark.parametrize(
    "valid_branch_name",
    [
        ("my-branch"),
        ("prefix/my-branch"),
        ("my-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50"),
    ],
)
def test_check_local_branch_valid(valid_branch_name: str) -> None:
    # Should not raise an error
    push.check_local_branch(
        branch_name=valid_branch_name,
        branch_prefix="prefix",
    )


def test_check_local_branch_invalid() -> None:
    with pytest.raises(
        push.LocalBranchInvalidError,
        match="Local branch is a branch generated by Mergify CLI",
    ):
        push.check_local_branch(
            branch_name="prefix/my-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
            branch_prefix="prefix",
        )


@pytest.mark.respx(base_url="https://api.github.com/")
async def test_stack_create(
    git_mock: test_utils.GitMock,
    respx_mock: respx.MockRouter,
) -> None:
    # Mock 2 commits on branch `current-branch`
    git_mock.commit(
        test_utils.Commit(
            sha="commit1_sha",
            title="Title commit 1",
            message="Message commit 1",
            change_id="I29617d37762fd69809c255d7e7073cb11f8fbf50",
        ),
    )
    git_mock.commit(
        test_utils.Commit(
            sha="commit2_sha",
            title="Title commit 2",
            message="Message commit 2",
            change_id="I29617d37762fd69809c255d7e7073cb11f8fbf51",
        ),
    )

    # Mock HTTP calls
    respx_mock.get("/user").respond(200, json={"login": "author"})
    respx_mock.get("/search/issues").respond(200, json={"items": []})
    post_pull1_mock = respx_mock.post(
        "/repos/user/repo/pulls",
        json__title="Title commit 1",
    ).respond(
        200,
        json={
            "html_url": "https://github.com/repo/user/pull/1",
            "number": "1",
            "title": "Title commit 1",
            "head": {"sha": "commit1_sha"},
            "state": "open",
            "merged_at": None,
            "draft": False,
            "node_id": "",
        },
    )
    post_pull2_mock = respx_mock.post(
        "/repos/user/repo/pulls",
        json__title="Title commit 2",
    ).respond(
        200,
        json={
            "html_url": "https://github.com/repo/user/pull/2",
            "number": "2",
            "title": "Title commit 2",
            "head": {"sha": "commit2_sha"},
            "state": "open",
            "merged_at": None,
            "draft": False,
            "node_id": "",
        },
    )
    respx_mock.get("/repos/user/repo/issues/1/comments").respond(200, json=[])
    post_comment1_mock = respx_mock.post("/repos/user/repo/issues/1/comments").respond(
        200,
    )
    respx_mock.get("/repos/user/repo/issues/2/comments").respond(200, json=[])
    post_comment2_mock = respx_mock.post("/repos/user/repo/issues/2/comments").respond(
        200,
    )

    await push.stack_push(
        github_server="https://api.github.com/",
        token="",
        skip_rebase=False,
        next_only=False,
        branch_prefix="",
        dry_run=False,
        trunk=("origin", "main"),
    )

    # First pull request is created
    assert len(post_pull1_mock.calls) == 1
    assert json.loads(post_pull1_mock.calls.last.request.content) == {
        "head": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
        "base": "main",
        "title": "Title commit 1",
        "body": "Message commit 1",
        "draft": False,
    }

    # Second pull request is created
    assert len(post_pull2_mock.calls) == 1
    assert json.loads(post_pull2_mock.calls.last.request.content) == {
        "head": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf51",
        "base": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
        "title": "Title commit 2",
        "body": "Message commit 2\n\nDepends-On: #1",
        "draft": False,
    }

    # First stack comment is created
    assert len(post_comment1_mock.calls) == 1
    expected_body = """This pull request is part of a stack:
1. Title commit 1 ([#1](https://github.com/repo/user/pull/1)) 👈
1. Title commit 2 ([#2](https://github.com/repo/user/pull/2))
"""
    assert json.loads(post_comment1_mock.calls.last.request.content) == {
        "body": expected_body,
    }

    # Second stack comment is created
    assert len(post_comment2_mock.calls) == 1
    expected_body = """This pull request is part of a stack:
1. Title commit 1 ([#1](https://github.com/repo/user/pull/1))
1. Title commit 2 ([#2](https://github.com/repo/user/pull/2)) 👈
"""
    assert json.loads(post_comment2_mock.calls.last.request.content) == {
        "body": expected_body,
    }


@pytest.mark.respx(base_url="https://api.github.com/")
async def test_stack_create_single_pull(
    git_mock: test_utils.GitMock,
    respx_mock: respx.MockRouter,
) -> None:
    # Mock 1 commits on branch `current-branch`
    git_mock.commit(
        test_utils.Commit(
            sha="commit1_sha",
            title="Title commit 1",
            message="Message commit 1",
            change_id="I29617d37762fd69809c255d7e7073cb11f8fbf50",
        ),
    )

    # Mock HTTP calls
    respx_mock.get("/user").respond(200, json={"login": "author"})
    respx_mock.get("/search/issues").respond(200, json={"items": []})

    post_pull_mock = respx_mock.post(
        "/repos/user/repo/pulls",
        json__title="Title commit 1",
    ).respond(
        200,
        json={
            "html_url": "https://github.com/repo/user/pull/1",
            "number": "1",
            "title": "Title commit 1",
            "head": {"sha": "commit1_sha"},
            "state": "open",
            "merged_at": None,
            "draft": False,
            "node_id": "",
        },
    )
    respx_mock.get("/repos/user/repo/issues/1/comments").respond(200, json=[])

    await push.stack_push(
        github_server="https://api.github.com/",
        token="",
        skip_rebase=False,
        next_only=False,
        branch_prefix="",
        dry_run=False,
        trunk=("origin", "main"),
    )

    # Pull request is created without stack comment
    assert len(post_pull_mock.calls) == 1
    assert json.loads(post_pull_mock.calls.last.request.content) == {
        "head": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
        "base": "main",
        "title": "Title commit 1",
        "body": "Message commit 1",
        "draft": False,
    }


@pytest.mark.respx(base_url="https://api.github.com/")
async def test_stack_update_no_rebase(
    git_mock: test_utils.GitMock,
    respx_mock: respx.MockRouter,
) -> None:
    # Mock 1 commits on branch `current-branch`
    git_mock.commit(
        test_utils.Commit(
            sha="commit_sha",
            title="Title",
            message="Message",
            change_id="I29617d37762fd69809c255d7e7073cb11f8fbf50",
        ),
    )

    # Mock HTTP calls: the stack already exists but it's out of date, it should
    # be updated
    respx_mock.get("/user").respond(200, json={"login": "author"})
    respx_mock.get("/search/issues").respond(
        200,
        json={
            "items": [
                {
                    "pull_request": {
                        "url": "https://api.github.com/repos/user/repo/pulls/123",
                    },
                },
            ],
        },
    )

    respx_mock.get(
        "/repos/user/repo/pulls/123",
    ).respond(
        200,
        json={
            "html_url": "",
            "number": "123",
            "title": "Title",
            "head": {
                "sha": "previous_commit_sha",
                "ref": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
            },
            "body": "body",
            "state": "open",
            "merged_at": None,
            "draft": False,
            "node_id": "",
        },
    )
    patch_pull_mock = respx_mock.patch("/repos/user/repo/pulls/123").respond(
        200,
        json={},
    )
    respx_mock.get("/repos/user/repo/issues/123/comments").respond(
        200,
        json=[
            {
                "body": "This pull request is part of a stack:\n...",
                "url": "https://api.github.com/repos/user/repo/issues/comments/456",
            },
        ],
    )
    respx_mock.patch("/repos/user/repo/issues/comments/456").respond(200)

    await push.stack_push(
        github_server="https://api.github.com/",
        token="",
        skip_rebase=True,
        next_only=False,
        branch_prefix="",
        dry_run=False,
        trunk=("origin", "main"),
    )
    assert not git_mock.has_been_called_with("pull", "--rebase", "origin", "main")

    # The pull request is updated
    assert len(patch_pull_mock.calls) == 1
    assert json.loads(patch_pull_mock.calls.last.request.content) == {
        "head": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
        "base": "main",
        "title": "Title",
        "body": "Message",
    }


@pytest.mark.respx(base_url="https://api.github.com/")
async def test_stack_update(
    git_mock: test_utils.GitMock,
    respx_mock: respx.MockRouter,
) -> None:
    # Mock 1 commits on branch `current-branch`
    git_mock.commit(
        test_utils.Commit(
            sha="commit_sha",
            title="Title",
            message="Message",
            change_id="I29617d37762fd69809c255d7e7073cb11f8fbf50",
        ),
    )

    # Mock HTTP calls: the stack already exists but it's out of date, it should
    # be updated
    respx_mock.get("/user").respond(200, json={"login": "author"})
    respx_mock.get("/search/issues").respond(
        200,
        json={
            "items": [
                {
                    "pull_request": {
                        "url": "https://api.github.com/repos/user/repo/pulls/123",
                    },
                },
            ],
        },
    )

    respx_mock.get(
        "/repos/user/repo/pulls/123",
    ).respond(
        200,
        json={
            "html_url": "",
            "number": "123",
            "title": "Title",
            "head": {
                "sha": "previous_commit_sha",
                "ref": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
            },
            "body": "body",
            "state": "open",
            "merged_at": None,
            "draft": False,
            "node_id": "",
        },
    )
    patch_pull_mock = respx_mock.patch("/repos/user/repo/pulls/123").respond(
        200,
        json={},
    )
    respx_mock.get("/repos/user/repo/issues/123/comments").respond(
        200,
        json=[
            {
                "body": "This pull request is part of a stack:\n...",
                "url": "https://api.github.com/repos/user/repo/issues/comments/456",
            },
        ],
    )
    respx_mock.patch("/repos/user/repo/issues/comments/456").respond(200)

    await push.stack_push(
        github_server="https://api.github.com/",
        token="",
        skip_rebase=False,
        next_only=False,
        branch_prefix="",
        dry_run=False,
        trunk=("origin", "main"),
    )
    assert git_mock.has_been_called_with("pull", "--rebase", "origin", "main")

    # The pull request is updated
    assert len(patch_pull_mock.calls) == 1
    assert json.loads(patch_pull_mock.calls.last.request.content) == {
        "head": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
        "base": "main",
        "title": "Title",
        "body": "Message",
    }


@pytest.mark.respx(base_url="https://api.github.com/")
async def test_stack_update_keep_title_and_body(
    git_mock: test_utils.GitMock,
    respx_mock: respx.MockRouter,
) -> None:
    # Mock 1 commits on branch `current-branch`
    git_mock.commit(
        test_utils.Commit(
            sha="commit_sha",
            title="New Title that should be ignored",
            message="New Message that should be ignored",
            change_id="I29617d37762fd69809c255d7e7073cb11f8fbf50",
        ),
    )

    # Mock HTTP calls: the stack already exists but it's out of date, it should
    # be updated
    respx_mock.get("/user").respond(200, json={"login": "author"})
    respx_mock.get("/search/issues").respond(
        200,
        json={
            "items": [
                {
                    "pull_request": {
                        "url": "https://api.github.com/repos/user/repo/pulls/123",
                    },
                },
            ],
        },
    )
    respx_mock.get(
        "/repos/user/repo/pulls/123",
    ).respond(
        200,
        json={
            "html_url": "",
            "number": "123",
            "title": "Title",
            "head": {
                "sha": "previous_commit_sha",
                "ref": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
            },
            "state": "open",
            "merged_at": None,
            "draft": False,
            "node_id": "",
            "body": "DONT TOUCH ME\n\nDepends-On: #12345\n",
        },
    )
    patch_pull_mock = respx_mock.patch("/repos/user/repo/pulls/123").respond(
        200,
        json={},
    )
    respx_mock.get("/repos/user/repo/issues/123/comments").respond(
        200,
        json=[
            {
                "body": "This pull request is part of a stack:\n...",
                "url": "https://api.github.com/repos/user/repo/issues/comments/456",
            },
        ],
    )
    respx_mock.patch("/repos/user/repo/issues/comments/456").respond(200)

    await push.stack_push(
        github_server="https://api.github.com/",
        token="",
        skip_rebase=False,
        next_only=False,
        branch_prefix="",
        dry_run=False,
        trunk=("origin", "main"),
        keep_pull_request_title_and_body=True,
    )

    # The pull request is updated
    assert len(patch_pull_mock.calls) == 1
    assert json.loads(patch_pull_mock.calls.last.request.content) == {
        "head": "current-branch/I29617d37762fd69809c255d7e7073cb11f8fbf50",
        "base": "main",
        "body": "DONT TOUCH ME",
    }


@pytest.mark.respx(base_url="https://api.github.com/")
async def test_stack_on_destination_branch_raises_an_error(
    git_mock: test_utils.GitMock,
    respx_mock: respx.MockRouter,
) -> None:
    respx_mock.get("/user").respond(200, json={"login": "author"})
    git_mock.mock("rev-parse", "--abbrev-ref", "HEAD", output="main")
    git_mock.mock(
        "remote",
        "get-url",
        "origin",
        output="https://github.com/foo/bar.git",
    )

    with pytest.raises(SystemExit, match="1"):
        await push.stack_push(
            github_server="https://api.github.com/",
            token="",
            skip_rebase=False,
            next_only=False,
            branch_prefix="",
            dry_run=False,
            trunk=("origin", "main"),
        )


@pytest.mark.respx(base_url="https://api.github.com/")
async def test_stack_without_common_commit_raises_an_error(
    git_mock: test_utils.GitMock,
    respx_mock: respx.MockRouter,
) -> None:
    respx_mock.get("/user").respond(200, json={"login": "author"})
    git_mock.mock("merge-base", "--fork-point", "origin/main", output="")

    with pytest.raises(SystemExit, match="1"):
        await push.stack_push(
            github_server="https://api.github.com/",
            token="",
            skip_rebase=False,
            next_only=False,
            branch_prefix="",
            dry_run=False,
            trunk=("origin", "main"),
        )
