from uuid6 import uuid7
from datetime import datetime

from cattle_grid.model.processing import StoreActivityMessage
from cattle_grid.dependencies.processing import MessageActor
from cattle_grid.dependencies import CommittingSession, InternalExchangePublisher
from cattle_grid.database.activity_pub_actor import StoredActivity


async def store_activity_subscriber(
    message: StoreActivityMessage,
    actor: MessageActor,
    session: CommittingSession,
    publisher: InternalExchangePublisher,
):
    """This method used internally to store activities
    generated by `cattle_grid`, e.g. an automatically generated
    accept requests. We note that it assigns an id to
    the activity.

    After storing the activity, it is published to the
    appropriate `outgoing.*` topic, which triggers sending
    the activity.
    """
    base_url = "/".join(actor.actor_id.split("/")[:-2])
    uuid = str(uuid7())
    activity_id = f"{base_url}/object/{uuid}"

    activity = message.data
    activity["id"] = activity_id
    activity_type = activity.get("type")

    session.add(
        StoredActivity(id=uuid, actor=actor, data=activity, published=datetime.now())
    )

    await publisher(
        {"actor": actor.actor_id, "data": activity},
        routing_key=f"outgoing.{activity_type}",
    )
