"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.applyLayers = void 0;
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
const layers = new Map();
function applyLayers(scope, region, lambdas, pythonLayerVersion, nodeLayerVersion, javaLayerVersion, extensionLayerVersion) {
    // TODO: check region availability
    const errors = [];
    loglevel_1.default.debug("Applying layers to Lambda functions...");
    lambdas.forEach((lam) => {
        const runtime = lam.runtime.name;
        const lambdaRuntimeType = index_1.runtimeLookup[runtime];
        const isARM = lam.architecture.dockerPlatform === aws_lambda_1.Architecture.ARM_64.dockerPlatform;
        const isNode = lambdaRuntimeType === index_1.RuntimeType.NODE;
        if (lambdaRuntimeType === index_1.RuntimeType.UNSUPPORTED) {
            loglevel_1.default.debug(`Unsupported runtime: ${runtime}`);
            return;
        }
        let lambdaLayerArn;
        let extensionLayerArn;
        if (lambdaRuntimeType === index_1.RuntimeType.PYTHON) {
            if (pythonLayerVersion === undefined) {
                const errorMessage = index_1.getMissingLayerVersionErrorMsg(lam.node.id, "Python", "python");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = index_1.getLambdaLayerArn(region, pythonLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using Python Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.NODE) {
            if (nodeLayerVersion === undefined) {
                const errorMessage = index_1.getMissingLayerVersionErrorMsg(lam.node.id, "Node.js", "node");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = index_1.getLambdaLayerArn(region, nodeLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using Node Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.JAVA) {
            if (javaLayerVersion === undefined) {
                const errorMessage = index_1.getMissingLayerVersionErrorMsg(lam.node.id, "Java", "java");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = index_1.getLambdaLayerArn(region, javaLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using dd-trace-java layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (extensionLayerVersion !== undefined) {
            extensionLayerArn = index_1.getExtensionLayerArn(region, extensionLayerVersion, isARM);
            loglevel_1.default.debug(`Using extension layer: ${extensionLayerArn}`);
            addLayer(extensionLayerArn, true, scope, lam, runtime);
        }
    });
    return errors;
}
exports.applyLayers = applyLayers;
function addLayer(layerArn, isExtensionLayer, scope, lam, runtime) {
    const layerId = index_1.generateLayerId(isExtensionLayer, lam.functionArn, runtime);
    if (layerArn !== undefined) {
        let lambdaLayer = layers.get(layerArn);
        if (lambdaLayer === undefined) {
            lambdaLayer = lambda.LayerVersion.fromLayerVersionArn(scope, layerId, layerArn);
            layers.set(layerArn, lambdaLayer);
        }
        loglevel_1.default.debug(`Adding layer ${lambdaLayer} to Lambda: ${lam.functionName}`);
        lam.addLayers(lambdaLayer);
    }
}
//# sourceMappingURL=data:application/json;base64,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