"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.redirectHandlers = void 0;
const loglevel_1 = require("loglevel");
const constants_1 = require("./constants");
/**
 * To avoid modifying code in the user's lambda handler, redirect the handler to a Datadog
 * handler that initializes the Lambda Layers and then calls the original handler.
 * 'DD_LAMBDA_HANDLER' is set to the original handler in the lambda's environment for the
 * replacement handler to find.
 *
 * Unchanged aside from parameter type
 */
function redirectHandlers(lambdas, addLayers) {
    loglevel_1.default.debug(`Wrapping Lambda function handlers with Datadog handler...`);
    lambdas.forEach((lam) => {
        const runtime = lam.runtime.name;
        const lambdaRuntime = constants_1.runtimeLookup[runtime];
        if (lambdaRuntime === constants_1.RuntimeType.JAVA) {
            lam.addEnvironment(constants_1.AWS_JAVA_WRAPPER_ENV_VAR, constants_1.AWS_JAVA_WRAPPER_ENV_VAR_VALUE);
        }
        else {
            const cfnFunction = lam.node.defaultChild;
            const originalHandler = cfnFunction.handler;
            lam.addEnvironment(constants_1.DD_HANDLER_ENV_VAR, originalHandler);
            const handler = getDDHandler(lambdaRuntime, addLayers);
            if (handler === undefined) {
                loglevel_1.default.debug("Unable to get Datadog handler");
                return;
            }
            cfnFunction.handler = handler;
        }
    });
}
exports.redirectHandlers = redirectHandlers;
function getDDHandler(lambdaRuntime, addLayers) {
    if (lambdaRuntime === undefined || lambdaRuntime === constants_1.RuntimeType.UNSUPPORTED) {
        loglevel_1.default.debug("Unsupported/undefined Lambda runtime");
        return;
    }
    switch (lambdaRuntime) {
        case constants_1.RuntimeType.NODE:
            return addLayers ? constants_1.JS_HANDLER_WITH_LAYERS : constants_1.JS_HANDLER;
        case constants_1.RuntimeType.PYTHON:
            return constants_1.PYTHON_HANDLER;
        case constants_1.RuntimeType.JAVA:
            return null;
    }
}
//# sourceMappingURL=data:application/json;base64,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