"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setDDEnvVariables = exports.applyEnvVariables = exports.setGitEnvironmentVariables = exports.DD_APM_FLUSH_DEADLINE_MILLISECONDS = exports.DD_DECODE_AUTHORIZER_CONTEXT = exports.DD_ENCODE_AUTHORIZER_CONTEXT = exports.DD_PROFILING_ENABLED = exports.DD_COLD_START_TRACE_SKIP_LIB = exports.DD_MIN_COLD_START_DURATION = exports.DD_COLD_START_TRACING = exports.DD_TAGS = exports.DD_VERSION_ENV_VAR = exports.DD_SERVICE_ENV_VAR = exports.DD_ENV_ENV_VAR = exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR = exports.ENABLE_DD_LOGS_ENV_VAR = exports.LOG_LEVEL_ENV_VAR = exports.INJECT_LOG_CONTEXT_ENV_VAR = exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR = exports.ENABLE_DD_TRACING_ENV_VAR = void 0;
const loglevel_1 = require("loglevel");
exports.ENABLE_DD_TRACING_ENV_VAR = "DD_TRACE_ENABLED";
exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR = "DD_MERGE_XRAY_TRACES";
exports.INJECT_LOG_CONTEXT_ENV_VAR = "DD_LOGS_INJECTION";
exports.LOG_LEVEL_ENV_VAR = "DD_LOG_LEVEL";
exports.ENABLE_DD_LOGS_ENV_VAR = "DD_SERVERLESS_LOGS_ENABLED";
exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR = "DD_CAPTURE_LAMBDA_PAYLOAD";
exports.DD_ENV_ENV_VAR = "DD_ENV";
exports.DD_SERVICE_ENV_VAR = "DD_SERVICE";
exports.DD_VERSION_ENV_VAR = "DD_VERSION";
exports.DD_TAGS = "DD_TAGS";
exports.DD_COLD_START_TRACING = "DD_COLD_START_TRACING";
exports.DD_MIN_COLD_START_DURATION = "DD_MIN_COLD_START_DURATION";
exports.DD_COLD_START_TRACE_SKIP_LIB = "DD_COLD_START_TRACE_SKIP_LIB";
exports.DD_PROFILING_ENABLED = "DD_PROFILING_ENABLED";
exports.DD_ENCODE_AUTHORIZER_CONTEXT = "DD_ENCODE_AUTHORIZER_CONTEXT";
exports.DD_DECODE_AUTHORIZER_CONTEXT = "DD_DECODE_AUTHORIZER_CONTEXT";
exports.DD_APM_FLUSH_DEADLINE_MILLISECONDS = "DD_APM_FLUSH_DEADLINE_MILLISECONDS";
const execSync = require("child_process").execSync;
const URL = require("url").URL;
function setGitEnvironmentVariables(lambdas) {
    loglevel_1.default.debug("Adding source code integration...");
    const { hash, gitRepoUrl } = getGitData();
    if (hash == "" || gitRepoUrl == "")
        return;
    // We're using an any type here because AWS does not expose the `environment` field in their type
    lambdas.forEach((lambda) => {
        if (lambda.environment[exports.DD_TAGS] !== undefined) {
            lambda.environment[exports.DD_TAGS].value += `,git.commit.sha:${hash}`;
        }
        else {
            lambda.addEnvironment(exports.DD_TAGS, `git.commit.sha:${hash}`);
        }
        lambda.environment[exports.DD_TAGS].value += `,git.repository_url:${gitRepoUrl}`;
    });
}
exports.setGitEnvironmentVariables = setGitEnvironmentVariables;
function getGitData() {
    let hash;
    let gitRepoUrl;
    try {
        hash = execSync("git rev-parse HEAD").toString().trim();
        gitRepoUrl = execSync("git config --get remote.origin.url").toString().trim();
    }
    catch (e) {
        loglevel_1.default.debug(`Failed to add source code integration. Error: ${e}`);
        return { hash: "", gitRepoUrl: "" };
    }
    return { hash, gitRepoUrl: filterAndFormatGithubRemote(gitRepoUrl) };
}
// Removes sensitive info from the given git remote url and normalizes the url prefix.
// "git@github.com:" and "https://github.com/" prefixes will be normalized into "github.com/"
function filterAndFormatGithubRemote(rawRemote) {
    rawRemote = filterSensitiveInfoFromRepository(rawRemote);
    if (!rawRemote) {
        return rawRemote;
    }
    rawRemote = rawRemote.replace(/git@github\.com:|https:\/\/github\.com\//, "github.com/");
    return rawRemote;
}
function filterSensitiveInfoFromRepository(repositoryUrl) {
    try {
        if (!repositoryUrl) {
            return repositoryUrl;
        }
        if (repositoryUrl.startsWith("git@")) {
            return repositoryUrl;
        }
        const { protocol, hostname, pathname } = new URL(repositoryUrl);
        if (!protocol || !hostname) {
            return repositoryUrl;
        }
        return `${protocol}//${hostname}${pathname}`;
    }
    catch (e) {
        return repositoryUrl;
    }
}
function applyEnvVariables(lambdas, baseProps) {
    loglevel_1.default.debug(`Setting environment variables...`);
    lambdas.forEach((lam) => {
        lam.addEnvironment(exports.ENABLE_DD_TRACING_ENV_VAR, baseProps.enableDatadogTracing.toString().toLowerCase());
        lam.addEnvironment(exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR, baseProps.enableMergeXrayTraces.toString().toLowerCase());
        lam.addEnvironment(exports.INJECT_LOG_CONTEXT_ENV_VAR, baseProps.injectLogContext.toString().toLowerCase());
        lam.addEnvironment(exports.ENABLE_DD_LOGS_ENV_VAR, baseProps.enableDatadogLogs.toString().toLowerCase());
        lam.addEnvironment(exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR, baseProps.captureLambdaPayload.toString().toLowerCase());
        if (baseProps.logLevel) {
            lam.addEnvironment(exports.LOG_LEVEL_ENV_VAR, baseProps.logLevel);
        }
    });
}
exports.applyEnvVariables = applyEnvVariables;
function setDDEnvVariables(lambdas, props) {
    lambdas.forEach((lam) => {
        if (props.extensionLayerVersion) {
            if (props.env) {
                lam.addEnvironment(exports.DD_ENV_ENV_VAR, props.env);
            }
            if (props.service) {
                lam.addEnvironment(exports.DD_SERVICE_ENV_VAR, props.service);
            }
            if (props.version) {
                lam.addEnvironment(exports.DD_VERSION_ENV_VAR, props.version);
            }
            if (props.tags) {
                lam.addEnvironment(exports.DD_TAGS, props.tags);
            }
        }
        if (props.enableColdStartTracing !== undefined) {
            lam.addEnvironment(exports.DD_COLD_START_TRACING, props.enableColdStartTracing.toString().toLowerCase());
        }
        if (props.minColdStartTraceDuration !== undefined) {
            lam.addEnvironment(exports.DD_MIN_COLD_START_DURATION, props.minColdStartTraceDuration.toString().toLowerCase());
        }
        if (props.coldStartTraceSkipLibs !== undefined) {
            lam.addEnvironment(exports.DD_COLD_START_TRACE_SKIP_LIB, props.coldStartTraceSkipLibs);
        }
        if (props.enableProfiling !== undefined) {
            lam.addEnvironment(exports.DD_PROFILING_ENABLED, props.enableProfiling.toString().toLowerCase());
        }
        if (props.encodeAuthorizerContext !== undefined) {
            lam.addEnvironment(exports.DD_ENCODE_AUTHORIZER_CONTEXT, props.encodeAuthorizerContext.toString().toLowerCase());
        }
        if (props.decodeAuthorizerContext !== undefined) {
            lam.addEnvironment(exports.DD_DECODE_AUTHORIZER_CONTEXT, props.decodeAuthorizerContext.toString().toLowerCase());
        }
        if (props.apmFlushDeadline !== undefined) {
            lam.addEnvironment(exports.DD_APM_FLUSH_DEADLINE_MILLISECONDS, props.apmFlushDeadline.toString().toLowerCase());
        }
    });
}
exports.setDDEnvVariables = setDDEnvVariables;
//# sourceMappingURL=data:application/json;base64,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