"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.applyLayers = void 0;
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const loglevel_1 = require("loglevel");
const index_1 = require("./index");
const layers = new Map();
function applyLayers(scope, region, lambdas, pythonLayerVersion, nodeLayerVersion, javaLayerVersion, extensionLayerVersion) {
    // TODO: check region availability
    const errors = [];
    loglevel_1.default.debug("Applying layers to Lambda functions...");
    lambdas.forEach((lam) => {
        const runtime = lam.runtime.name;
        const lambdaRuntimeType = index_1.runtimeLookup[runtime];
        const isARM = lam.architecture.dockerPlatform === aws_lambda_1.Architecture.ARM_64.dockerPlatform;
        const isNode = lambdaRuntimeType === index_1.RuntimeType.NODE;
        if (lambdaRuntimeType === index_1.RuntimeType.UNSUPPORTED) {
            loglevel_1.default.debug(`Unsupported runtime: ${runtime}`);
            return;
        }
        let lambdaLayerArn;
        let extensionLayerArn;
        if (lambdaRuntimeType === index_1.RuntimeType.PYTHON) {
            if (pythonLayerVersion === undefined) {
                const errorMessage = (0, index_1.getMissingLayerVersionErrorMsg)(lam.node.id, "Python", "python");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = (0, index_1.getLambdaLayerArn)(region, pythonLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using Python Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.NODE) {
            if (nodeLayerVersion === undefined) {
                const errorMessage = (0, index_1.getMissingLayerVersionErrorMsg)(lam.node.id, "Node.js", "node");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = (0, index_1.getLambdaLayerArn)(region, nodeLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using Node Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === index_1.RuntimeType.JAVA) {
            if (javaLayerVersion === undefined) {
                const errorMessage = (0, index_1.getMissingLayerVersionErrorMsg)(lam.node.id, "Java", "java");
                loglevel_1.default.error(errorMessage);
                errors.push(errorMessage);
                return;
            }
            lambdaLayerArn = (0, index_1.getLambdaLayerArn)(region, javaLayerVersion, runtime, isARM, isNode);
            loglevel_1.default.debug(`Using dd-trace-java layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (extensionLayerVersion !== undefined) {
            extensionLayerArn = (0, index_1.getExtensionLayerArn)(region, extensionLayerVersion, isARM);
            loglevel_1.default.debug(`Using extension layer: ${extensionLayerArn}`);
            addLayer(extensionLayerArn, true, scope, lam, runtime);
        }
    });
    return errors;
}
exports.applyLayers = applyLayers;
function addLayer(layerArn, isExtensionLayer, scope, lam, runtime) {
    const layerId = (0, index_1.generateLayerId)(isExtensionLayer, lam.functionArn, runtime);
    if (layerArn !== undefined) {
        let lambdaLayer = layers.get(layerArn);
        if (lambdaLayer === undefined) {
            lambdaLayer = lambda.LayerVersion.fromLayerVersionArn(scope, layerId, layerArn);
            layers.set(layerArn, lambdaLayer);
        }
        loglevel_1.default.debug(`Adding layer ${lambdaLayer} to Lambda: ${lam.functionName}`);
        lam.addLayers(lambdaLayer);
    }
}
//# sourceMappingURL=data:application/json;base64,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