"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transport = exports.transportDefaults = exports.FLUSH_METRICS_TO_LOGS_ENV_VAR = exports.SITE_URL_ENV_VAR = exports.KMS_API_KEY_ENV_VAR = exports.API_KEY_SECRET_ARN_ENV_VAR = exports.API_KEY_ENV_VAR = void 0;
const loglevel_1 = require("loglevel");
const constants_1 = require("./constants");
exports.API_KEY_ENV_VAR = "DD_API_KEY";
exports.API_KEY_SECRET_ARN_ENV_VAR = "DD_API_KEY_SECRET_ARN";
exports.KMS_API_KEY_ENV_VAR = "DD_KMS_API_KEY";
exports.SITE_URL_ENV_VAR = "DD_SITE";
exports.FLUSH_METRICS_TO_LOGS_ENV_VAR = "DD_FLUSH_TO_LOG";
exports.transportDefaults = {
    site: "datadoghq.com",
    flushMetricsToLogs: true,
    enableDatadogTracing: true,
};
class Transport {
    constructor(flushMetricsToLogs, site, apiKey, apiKeySecretArn, apiKmsKey, extensionLayerVersion) {
        if (flushMetricsToLogs === undefined) {
            loglevel_1.default.debug(`No value provided for flushMetricsToLogs, defaulting to ${exports.transportDefaults.flushMetricsToLogs}`);
            this.flushMetricsToLogs = exports.transportDefaults.flushMetricsToLogs;
        }
        else {
            this.flushMetricsToLogs = flushMetricsToLogs;
        }
        this.extensionLayerVersion = extensionLayerVersion;
        // If the extension is used, metrics will be submitted via the extension.
        if (this.extensionLayerVersion !== undefined) {
            loglevel_1.default.debug(`Using extension version ${this.extensionLayerVersion}, metrics will be submitted via the extension`);
            this.flushMetricsToLogs = false;
        }
        if (site === undefined) {
            loglevel_1.default.debug(`No value provided for site, defaulting to ${exports.transportDefaults.site}`);
            this.site = exports.transportDefaults.site;
        }
        else {
            this.site = site;
        }
        this.apiKey = apiKey;
        this.apiKeySecretArn = apiKeySecretArn;
        this.apiKmsKey = apiKmsKey;
    }
    applyEnvVars(lambdas) {
        loglevel_1.default.debug(`Setting Datadog transport environment variables...`);
        lambdas.forEach((lam) => {
            lam.addEnvironment(exports.FLUSH_METRICS_TO_LOGS_ENV_VAR, this.flushMetricsToLogs.toString());
            if (this.site !== undefined && this.flushMetricsToLogs === false) {
                lam.addEnvironment(exports.SITE_URL_ENV_VAR, this.site);
            }
            if (this.apiKey !== undefined) {
                lam.addEnvironment(exports.API_KEY_ENV_VAR, this.apiKey);
            }
            if (this.apiKeySecretArn !== undefined) {
                const isNode = constants_1.runtimeLookup[lam.runtime.name] === constants_1.RuntimeType.NODE;
                const isSendingSynchronousMetrics = this.extensionLayerVersion === undefined && !this.flushMetricsToLogs;
                if (isSendingSynchronousMetrics && isNode) {
                    throw new Error(`\`apiKeySecretArn\` is not supported for Node runtimes when using Synchronous Metrics. Use either \`apiKey\` or \`apiKmsKey\`.`);
                }
                lam.addEnvironment(exports.API_KEY_SECRET_ARN_ENV_VAR, this.apiKeySecretArn);
            }
            if (this.apiKmsKey !== undefined) {
                lam.addEnvironment(exports.KMS_API_KEY_ENV_VAR, this.apiKmsKey);
            }
        });
    }
}
exports.Transport = Transport;
//# sourceMappingURL=data:application/json;base64,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