"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateLayerId = exports.generateExtensionLayerId = exports.generateLambdaLayerId = exports.getMissingLayerVersionErrorMsg = exports.getExtensionLayerArn = exports.getLambdaLayerArn = void 0;
const crypto = require("crypto");
const loglevel_1 = require("loglevel");
const constants_1 = require("./constants");
function getLambdaLayerArn(region, version, runtime, isArm, isNode) {
    const baseLayerName = constants_1.runtimeToLayerName[runtime];
    const layerName = isArm && !isNode ? `${baseLayerName}-ARM` : baseLayerName;
    // TODO: edge case where gov cloud is the region, but they are using a token so we can't resolve it.
    const isGovCloud = constants_1.govCloudRegions.includes(region);
    // if this is a GovCloud region, use the GovCloud lambda layer
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud lambda layer");
        return `arn:aws-us-gov:lambda:${region}:${constants_1.DD_GOV_ACCOUNT_ID}:layer:${layerName}:${version}`;
    }
    return `arn:aws:lambda:${region}:${constants_1.DD_ACCOUNT_ID}:layer:${layerName}:${version}`;
}
exports.getLambdaLayerArn = getLambdaLayerArn;
function getExtensionLayerArn(region, version, isArm) {
    const baseLayerName = "Datadog-Extension";
    const layerName = isArm ? `${baseLayerName}-ARM` : baseLayerName;
    const isGovCloud = constants_1.govCloudRegions.includes(region);
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud extension layer");
        return `arn:aws-us-gov:lambda:${region}:${constants_1.DD_GOV_ACCOUNT_ID}:layer:${layerName}:${version}`;
    }
    return `arn:aws:lambda:${region}:${constants_1.DD_ACCOUNT_ID}:layer:${layerName}:${version}`;
}
exports.getExtensionLayerArn = getExtensionLayerArn;
function getMissingLayerVersionErrorMsg(functionKey, formalRuntime, paramRuntime) {
    return (`Resource ${functionKey} has a ${formalRuntime} runtime, but no ${formalRuntime} Lambda Library version was provided. ` +
        `Please add the '${paramRuntime}LayerVersion' parameter for the Datadog serverless macro.`);
}
exports.getMissingLayerVersionErrorMsg = getMissingLayerVersionErrorMsg;
function generateLambdaLayerId(lambdaFunctionArn, runtime) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return `${constants_1.LAYER_PREFIX}-${runtime}-${layerValue}`;
}
exports.generateLambdaLayerId = generateLambdaLayerId;
function generateExtensionLayerId(lambdaFunctionArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Extension layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return `${constants_1.EXTENSION_LAYER_PREFIX}-${layerValue}`;
}
exports.generateExtensionLayerId = generateExtensionLayerId;
function generateLayerId(isExtensionLayer, functionArn, runtime) {
    if (isExtensionLayer) {
        return generateExtensionLayerId(functionArn);
    }
    return generateLambdaLayerId(functionArn, runtime);
}
exports.generateLayerId = generateLayerId;
//# sourceMappingURL=data:application/json;base64,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