"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setDDEnvVariables = exports.applyEnvVariables = exports.setGitEnvironmentVariables = exports.DD_DECODE_AUTHORIZER_CONTEXT = exports.DD_ENCODE_AUTHORIZER_CONTEXT = exports.DD_PROFILING_ENABLED = exports.DD_COLD_START_TRACE_SKIP_LIB = exports.DD_MIN_COLD_START_DURATION = exports.DD_COLD_START_TRACING = exports.DD_TAGS = exports.DD_VERSION_ENV_VAR = exports.DD_SERVICE_ENV_VAR = exports.DD_ENV_ENV_VAR = exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR = exports.ENABLE_DD_LOGS_ENV_VAR = exports.LOG_LEVEL_ENV_VAR = exports.INJECT_LOG_CONTEXT_ENV_VAR = exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR = exports.ENABLE_DD_TRACING_ENV_VAR = void 0;
const loglevel_1 = require("loglevel");
exports.ENABLE_DD_TRACING_ENV_VAR = "DD_TRACE_ENABLED";
exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR = "DD_MERGE_XRAY_TRACES";
exports.INJECT_LOG_CONTEXT_ENV_VAR = "DD_LOGS_INJECTION";
exports.LOG_LEVEL_ENV_VAR = "DD_LOG_LEVEL";
exports.ENABLE_DD_LOGS_ENV_VAR = "DD_SERVERLESS_LOGS_ENABLED";
exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR = "DD_CAPTURE_LAMBDA_PAYLOAD";
exports.DD_ENV_ENV_VAR = "DD_ENV";
exports.DD_SERVICE_ENV_VAR = "DD_SERVICE";
exports.DD_VERSION_ENV_VAR = "DD_VERSION";
exports.DD_TAGS = "DD_TAGS";
exports.DD_COLD_START_TRACING = "DD_COLD_START_TRACING";
exports.DD_MIN_COLD_START_DURATION = "DD_MIN_COLD_START_DURATION";
exports.DD_COLD_START_TRACE_SKIP_LIB = "DD_COLD_START_TRACE_SKIP_LIB";
exports.DD_PROFILING_ENABLED = "DD_PROFILING_ENABLED";
exports.DD_ENCODE_AUTHORIZER_CONTEXT = "DD_ENCODE_AUTHORIZER_CONTEXT";
exports.DD_DECODE_AUTHORIZER_CONTEXT = "DD_DECODE_AUTHORIZER_CONTEXT";
const execSync = require("child_process").execSync;
const URL = require("url").URL;
function setGitEnvironmentVariables(lambdas) {
    loglevel_1.default.debug("Adding source code integration...");
    const { hash, gitRepoUrl } = getGitData();
    if (hash == "" || gitRepoUrl == "")
        return;
    // We're using an any type here because AWS does not expose the `environment` field in their type
    lambdas.forEach((lambda) => {
        if (lambda.environment[exports.DD_TAGS] !== undefined) {
            lambda.environment[exports.DD_TAGS].value += `,git.commit.sha:${hash}`;
        }
        else {
            lambda.addEnvironment(exports.DD_TAGS, `git.commit.sha:${hash}`);
        }
        lambda.environment[exports.DD_TAGS].value += `,git.repository_url:${gitRepoUrl}`;
    });
}
exports.setGitEnvironmentVariables = setGitEnvironmentVariables;
function getGitData() {
    let hash;
    let gitRepoUrl;
    try {
        hash = execSync("git rev-parse HEAD").toString().trim();
        gitRepoUrl = execSync("git config --get remote.origin.url").toString().trim();
    }
    catch (e) {
        loglevel_1.default.debug(`Failed to add source code integration. Error: ${e}`);
        return { hash: "", gitRepoUrl: "" };
    }
    return { hash, gitRepoUrl: filterAndFormatGithubRemote(gitRepoUrl) };
}
// Removes sensitive info from the given git remote url and normalizes the url prefix.
// "git@github.com:" and "https://github.com/" prefixes will be normalized into "github.com/"
function filterAndFormatGithubRemote(rawRemote) {
    rawRemote = filterSensitiveInfoFromRepository(rawRemote);
    if (!rawRemote) {
        return rawRemote;
    }
    rawRemote = rawRemote.replace(/git@github\.com:|https:\/\/github\.com\//, "github.com/");
    return rawRemote;
}
function filterSensitiveInfoFromRepository(repositoryUrl) {
    try {
        if (!repositoryUrl) {
            return repositoryUrl;
        }
        if (repositoryUrl.startsWith("git@")) {
            return repositoryUrl;
        }
        const { protocol, hostname, pathname } = new URL(repositoryUrl);
        if (!protocol || !hostname) {
            return repositoryUrl;
        }
        return `${protocol}//${hostname}${pathname}`;
    }
    catch (e) {
        return repositoryUrl;
    }
}
function applyEnvVariables(lambdas, baseProps) {
    loglevel_1.default.debug(`Setting environment variables...`);
    lambdas.forEach((lam) => {
        lam.addEnvironment(exports.ENABLE_DD_TRACING_ENV_VAR, baseProps.enableDatadogTracing.toString().toLowerCase());
        lam.addEnvironment(exports.ENABLE_XRAY_TRACE_MERGING_ENV_VAR, baseProps.enableMergeXrayTraces.toString().toLowerCase());
        lam.addEnvironment(exports.INJECT_LOG_CONTEXT_ENV_VAR, baseProps.injectLogContext.toString().toLowerCase());
        lam.addEnvironment(exports.ENABLE_DD_LOGS_ENV_VAR, baseProps.enableDatadogLogs.toString().toLowerCase());
        lam.addEnvironment(exports.CAPTURE_LAMBDA_PAYLOAD_ENV_VAR, baseProps.captureLambdaPayload.toString().toLowerCase());
        if (baseProps.logLevel) {
            lam.addEnvironment(exports.LOG_LEVEL_ENV_VAR, baseProps.logLevel);
        }
    });
}
exports.applyEnvVariables = applyEnvVariables;
function setDDEnvVariables(lambdas, props) {
    lambdas.forEach((lam) => {
        if (props.extensionLayerVersion) {
            if (props.env) {
                lam.addEnvironment(exports.DD_ENV_ENV_VAR, props.env);
            }
            if (props.service) {
                lam.addEnvironment(exports.DD_SERVICE_ENV_VAR, props.service);
            }
            if (props.version) {
                lam.addEnvironment(exports.DD_VERSION_ENV_VAR, props.version);
            }
            if (props.tags) {
                lam.addEnvironment(exports.DD_TAGS, props.tags);
            }
        }
        if (props.enableColdStartTracing !== undefined) {
            lam.addEnvironment(exports.DD_COLD_START_TRACING, props.enableColdStartTracing.toString().toLowerCase());
        }
        if (props.minColdStartTraceDuration !== undefined) {
            lam.addEnvironment(exports.DD_MIN_COLD_START_DURATION, props.minColdStartTraceDuration.toString().toLowerCase());
        }
        if (props.coldStartTraceSkipLibs !== undefined) {
            lam.addEnvironment(exports.DD_COLD_START_TRACE_SKIP_LIB, props.coldStartTraceSkipLibs);
        }
        if (props.enableProfiling !== undefined) {
            lam.addEnvironment(exports.DD_PROFILING_ENABLED, props.enableProfiling.toString().toLowerCase());
        }
        if (props.encodeAuthorizerContext !== undefined) {
            lam.addEnvironment(exports.DD_ENCODE_AUTHORIZER_CONTEXT, props.encodeAuthorizerContext.toString().toLowerCase());
        }
        if (props.decodeAuthorizerContext !== undefined) {
            lam.addEnvironment(exports.DD_DECODE_AUTHORIZER_CONTEXT, props.decodeAuthorizerContext.toString().toLowerCase());
        }
    });
}
exports.setDDEnvVariables = setDDEnvVariables;
//# sourceMappingURL=data:application/json;base64,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