from exploits.regex.regex_common import RegexParser
from delayed_assert import expect, assert_expectations
import pytest

parser_test_cases = [
    ('ab', ['ab'], ['a', 'b', '', 'aba', 'aaaaaaaaaab']),
    ('a*', ['a', 'aaaaaa', ''], ['aba', 'b', 'aaaaaaaaaab']),
    ('a*b', ['ab', 'aaaaaab', 'b'], ['aba', 'ba', 'aaaaaaacaab']),
    ('[abc]', ['a', 'b', 'c'], ['ac', 'ba', '']),
    ('[a-c]', ['a', 'b', 'c'], ['ac', 'ba', '']),
    ('a|b', ['a', 'b'], ['c', 'ac', 'ba', '']),
    ('a|b|c', ['a', 'b', 'c'], ['ac', 'ba', '']),
    ('(a|b|c)d', ['ad', 'bd', 'cd'], ['ac', 'ba', 'd']),
    ('(a|b|ab)*c', ['ac', 'bc', 'c', 'abc', 'ababc', 'aaaaaaaac', 'bbbbbbbc', 'aaaaabababbbbbc'], ['', 'ba', 'd']),
    ('a?', ['a', ''], ['ac', 'ba', 'd']),
    ('a+', ['aa', 'a', 'aaaaaaa'], ['', 'ac', 'ba', 'd']),
    ('a+b', ['aab', 'ab', 'aaaaaaab'], ['a', '', 'ac', 'ba', 'd']),
    ('[^abc]', ['d', 'e', '1', '.', '[', '*', '?'], ['', 'a', 'b', 'c', 'dd']),
    ('[^a-c]', ['d', 'e', '1', '.', '[', '*', '?'], ['', 'a', 'b', 'c', 'dd']),
    ('[-ab]', ['-', 'a', 'b'], ['', 'd', 'ab', '-a']),
    ('[ab-]', ['-', 'a', 'b'], ['', 'd', 'ab', '-a']),
    ('[^-ab]', ['3', 'd', 'h'], ['', 'a', 'b', '-', '-a']),
    ('[^ab-]', ['3', 'd', 'h'], ['', 'a', 'b', '-', '-a']),
    ('\+', ['+'], ['', 'a', 'b', '-', '-a']),
    ('.', ['a', 'b', 'c', 'd'], ['', 'ab', 'aaa']),
    ('[.+]', ['.', '+'], ['', 'a', 'aaa']),
    ('\d', ['0', '1', '9'], ['', 'a', 'b']),
    ('\D', ['a', 'b', 'A', 'D'], ['', '0', '1', '9']),
    ('\s', [' ', '\t', '\n'], ['', 'a', 'b', 'e', '1', 'aa']),
    ('\S', ['a', 'b', 'e', '1'], ['', 'aa', ' ', '\t', '\n']),
    ('a{5}', ['aaaaa'], ['', 'aaaa', 'aaaaaa', 'bbbbb']),
    ('.{5}', ['aaaaa', 'abcdf', 'e421d'], ['', 'aaaa', 'aaaaaa']),
    ('(ab){5}', ['ababababab'], ['', 'aaaaaaaaaa', 'abababab', 'abababababab']),
    ('[ab]{5}', ['aaaaa', 'bbbbb', 'ababa', 'aabaa'], ['', 'aaaa', 'bbbb', 'aaaac']),
    ('\d{5}', ['12345', '54783'], ['1111', '111111', 'aaaaa', '1111a', 'c2345']),
    ('\D{5}', ['AbDeF', 'hjdsf'], ['11111', 'aaaa', 'aaaa1', 'aaaacs']),
    ('a{2,}', ['aa', 'aaa', 'aaaa', 'aaaaa', 'aaaaaa'], ['', 'a', 'aaaab', 'bbbbb']),
    ('a{2}', ['aa'], ['', 'a', 'aaa', 'aaaa', 'aaaaa', 'bb']),
    ('a{2,4}', ['aa', 'aaa', 'aaaa'], ['', 'a', 'aaaaa', 'bb']),
    ('a{1,3}', ['a', 'aa', 'aaa'], ['', 'aaaa', 'aaaaa', 'bb']),
    ('a+5', ['a5', 'aa5', 'aaa5', 'aaaa5'], ['', '5', 'a', 'aaaaa', 'bb']),
    ('5', ['5'], ['', 'a', 'aaaaa', 'bb']),
    ('51', ['51'], ['5', 'a', 'aaaaa', 'bb']),
    ('\w', ['a', 'b', 'c', '1', '0', '9', 'A', 'Z', 'M', '_'], ['', '!', '@', '&', '%', ' ']),
    ('\W', ['!', '@', '&', '%', ' '], ['', 'a', 'b', 'c', '1', '0', '9', 'A', 'Z', 'M', '_']),
    ('(a*[0-9]+)|(a*[0-9]*)', ['aaa0000333', '52345'], ['b', 'b431324']),
    ('^(-(([0-9]+\\.[0-9]*[1-9][0-9]*)|([0-9]*[1-9][0-9]*\\.[0-9]+)|([0-9]*[1-9][0-9]*)))$',
     ['-432.432'], ['asdf', '4324.5432', '88']),
    ('A.*?B', ['AB', 'AsdfasB', 'AdfB'], ['A', 'B', 'AD']),
    ('A\\p{Z}B', ['A B', 'A\tB', 'A\nB'], ['AB', 'ABB', 'ABC']),
    ('A\\p{^Z}B', ['ABB', 'ACB'], ['A B', 'A\tB', 'A\nB', 'AB', 'ABC']),
    ('\\\\p{javaJavaIdentifierPart}', ['\\A', '\\1'], ['\\ ', '1']),
    ('\\\\', ['\\\\'], ['\\']),
    ('\\x{0420}', [u'\u0420'], ['a']),
]


def test_nfa_parser():
    for test_case in parser_test_cases:
        parser = RegexParser(False)
        nfa = parser.run(test_case[0])

        for success in test_case[1]:
            expect(nfa.match_string(success, True),
                   '"{}" should match "{}"'.format(success, test_case[0]))
        for failure in test_case[2]:
            expect(not nfa.match_string(failure, True),
                   '"{}" should not match "{}"'.format(failure, test_case[0]))
    assert_expectations()


def test_nfa_parser_no_epsilon():
    for test_case in parser_test_cases:
        parser = RegexParser(True)
        nfa = parser.run(test_case[0])

        for success in test_case[1]:
            expect(nfa.match_string(success, True),
                   '"{}" should match "{}"'.format(success, test_case[0]))
        for failure in test_case[2]:
            expect(not nfa.match_string(failure, True),
                   '"{}" should not match "{}"'.format(failure, test_case[0]))
    assert_expectations()


def test_dfa():
    for test_case in parser_test_cases:
        parser = RegexParser(dfa=True)
        dfa = parser.run(test_case[0])

        for success in test_case[1]:
            expect(dfa.match_string(success, True),
                   '"{}" should match "{}"'.format(success, test_case[0]))
        for failure in test_case[2]:
            expect(not dfa.match_string(failure, True),
                   '"{}" should not match "{}"'.format(failure, test_case[0]))
    assert_expectations()


def test_empty_character_class():
    parser = RegexParser()
    with pytest.raises(ValueError):
        parser.run('[]')


def test_empty_negation_character_class():
    parser = RegexParser()
    with pytest.raises(ValueError):
        parser.run('[^]')


def test_dangling_plus():
    parser = RegexParser()
    with pytest.raises(ValueError):
        parser.run('+')


def test_dangling_star():
    parser = RegexParser()
    with pytest.raises(ValueError):
        parser.run('*')


def test_dangling_question():
    parser = RegexParser()
    with pytest.raises(ValueError):
        parser.run('?')
