"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const ec2 = require("@aws-cdk/aws-ec2");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test("Test minimal deployment with no properties", () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [], []);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "0.0.0.0/0",
                Description: "Allow all outbound traffic by default",
                IpProtocol: "-1",
            },
        ],
    });
});
test("Test deployment with ingress rules", () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) }], []);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupIngress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
        ],
    });
});
test("Test deployment with egress rule", () => {
    // Stack
    const stack = new core_1.Stack();
    const vpc = new ec2.Vpc(stack, "test-vpc", {});
    // Helper declaration
    defaults.buildSecurityGroup(stack, "primary-queue", {
        vpc,
        allowAllOutbound: false,
    }, [], [
        { peer: ec2.Peer.ipv4("1.1.1.1/16"), connection: ec2.Port.tcp(100) },
        { peer: ec2.Peer.ipv4("2.2.2.2/24"), connection: ec2.Port.tcp(200) },
    ]);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        SecurityGroupEgress: [
            {
                CidrIp: "1.1.1.1/16",
                Description: "from 1.1.1.1/16:100",
                FromPort: 100,
                IpProtocol: "tcp",
                ToPort: 100,
            },
            {
                CidrIp: "2.2.2.2/24",
                Description: "from 2.2.2.2/24:200",
                FromPort: 200,
                IpProtocol: "tcp",
                ToPort: 200,
            },
        ],
    });
});
//# sourceMappingURL=data:application/json;base64,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