"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const defaults = require("..");
const _fieldSchema = [{
        name: "id",
        type: "int",
        comment: "Identifier for the record"
    }, {
        name: "name",
        type: "string",
        comment: "The name of the record"
    }, {
        name: "type",
        type: "string",
        comment: "The type of the record"
    }, {
        name: "numericvalue",
        type: "int",
        comment: "Some value associated with the record"
    }];
// --------------------------------------------------------------
// Create database by passing props
// --------------------------------------------------------------
test('create default CfnTable with default props', () => {
    const stack = new core_1.Stack();
    defaults.DefaultGlueTable(stack, {
        catalogId: 'fakecatalogfortest',
        databaseName: 'fakedatabase',
        tableInput: {
            parameters: {
                classification: 'json'
            },
            storageDescriptor: {
                parameters: {
                    endpointUrl: `https://kinesis.${core_1.Aws.REGION}.amazonaws.com`,
                    streamName: 'testStream',
                    typeOfData: 'kinesis'
                }
            }
        }
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Type: "AWS::Glue::Table",
        Properties: {
            CatalogId: "fakecatalogfortest",
            DatabaseName: "fakedatabase",
            TableInput: {
                Parameters: {
                    classification: "json"
                },
                StorageDescriptor: {
                    Parameters: {
                        endpointUrl: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://kinesis.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".amazonaws.com"
                                ]
                            ]
                        },
                        streamName: "testStream",
                        typeOfData: "kinesis"
                    }
                }
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Create database by passing no props to database (default database)
// --------------------------------------------------------------
test('Create table', () => {
    const stack = new core_1.Stack();
    defaults.createGlueTable(stack, defaults.createGlueDatabase(stack), undefined, _fieldSchema, 'kinesis', {
        STREAM_NAME: 'testStream'
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResourceLike('AWS::Glue::Database', {
        Type: "AWS::Glue::Database",
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseInput: {
                Description: "An AWS Glue database generated by AWS Solutions Construct"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResourceLike('AWS::Glue::Table', {
        Properties: {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                Ref: "GlueDatabase"
            },
            TableInput: {
                Parameters: {
                    classification: "json"
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Comment: "Identifier for the record",
                            Name: "id",
                            Type: "int"
                        },
                        {
                            Comment: "The name of the record",
                            Name: "name",
                            Type: "string"
                        },
                        {
                            Comment: "The type of the record",
                            Name: "type",
                            Type: "string"
                        },
                        {
                            Comment: "Some value associated with the record",
                            Name: "numericvalue",
                            Type: "int"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: "testStream",
                    NumberOfBuckets: -1,
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    Parameters: {
                        endpointUrl: {
                            "Fn::Join": [
                                "",
                                [
                                    "https://kinesis.",
                                    {
                                        Ref: "AWS::Region"
                                    },
                                    ".amazonaws.com"
                                ]
                            ]
                        },
                        streamName: "testStream",
                        typeOfData: "kinesis"
                    },
                    SerdeInfo: {
                        Parameters: {
                            paths: "id,name,type,numericvalue"
                        },
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    }
                },
                TableType: "EXTERNAL_TABLE"
            }
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an unsupported source type, it should throw an error
// --------------------------------------------------------------
test('error condition', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.DefaultGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
        defaults.DefaultGlueTable(_database, defaults.DefaultGlueTableProps(_database, _fieldSchema, 'SomeSource', { STREAM_NAME: 'somefakestream' }));
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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