"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployES(stack, domainName, cfnDomainProps, lambdaRoleARN) {
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    const cognitoAuthorizedRole = defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    if (lambdaRoleARN) {
        return defaults.buildElasticSearch(stack, domainName, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn,
            serviceRoleARN: lambdaRoleARN
        }, cfnDomainProps);
    }
    else {
        return defaults.buildElasticSearch(stack, domainName, {
            userpool,
            identitypool,
            cognitoAuthorizedRoleARN: cognitoAuthorizedRole.roleArn
        }, cfnDomainProps);
    }
}
test('snapshot test buildElasticSearch default params', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain');
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override SnapshotOptions for buildElasticSearch', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain', {
        snapshotOptions: {
            automatedSnapshotStartHour: 5
        }
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 5
        }
    });
});
test('Test override ES version for buildElasticSearch', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain', {
        elasticsearchVersion: '7.0'
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: {
                            "Fn::GetAtt": [
                                "CognitoAuthorizedRole14E74FE0",
                                "Arn"
                            ]
                        }
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "7.0",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Test ES with lambdaRoleARN', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain', {}, 'arn:aws:us-east-1:mylambdaRoleARN');
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        AccessPolicies: {
            Statement: [
                {
                    Action: "es:ESHttp*",
                    Effect: "Allow",
                    Principal: {
                        AWS: [
                            {
                                "Fn::GetAtt": [
                                    "CognitoAuthorizedRole14E74FE0",
                                    "Arn"
                                ]
                            },
                            "arn:aws:us-east-1:mylambdaRoleARN"
                        ]
                    },
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:aws:es:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":domain/test-domain/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        },
        CognitoOptions: {
            Enabled: true,
            IdentityPoolId: {
                Ref: "CognitoIdentityPool"
            },
            RoleArn: {
                "Fn::GetAtt": [
                    "CognitoKibanaConfigureRole62CCE76A",
                    "Arn"
                ]
            },
            UserPoolId: {
                Ref: "CognitoUserPool53E37E69"
            }
        },
        DomainName: "test-domain",
        EBSOptions: {
            EBSEnabled: true,
            VolumeSize: 10
        },
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 3
            },
            ZoneAwarenessEnabled: true
        },
        ElasticsearchVersion: "6.3",
        EncryptionAtRestOptions: {
            Enabled: true
        },
        NodeToNodeEncryptionOptions: {
            Enabled: true
        },
        SnapshotOptions: {
            AutomatedSnapshotStartHour: 1
        }
    });
});
test('Count ES CW Alarms', () => {
    const stack = new core_1.Stack();
    deployES(stack, 'test-domain');
    const cwList = defaults.buildElasticSearchCWAlarms(stack);
    expect(cwList.length).toEqual(9);
});
//# sourceMappingURL=data:application/json;base64,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