"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
// Imports
const sqs = require("@aws-cdk/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_helper_1 = require("./kms-helper");
function buildQueue(scope, id, props) {
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Setup the queue
        let queueProps;
        if (props.queueProps) {
            // If property overrides have been provided, incorporate them and deploy
            queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), props.queueProps);
        }
        else {
            // If no property overrides, deploy using the default configuration
            queueProps = defaults.DefaultQueueProps();
        }
        // Determine whether a DLQ property should be added
        if (props.deadLetterQueue) {
            queueProps.deadLetterQueue = props.deadLetterQueue;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey) {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                queueProps.encryptionMasterKey = props.encryptionKey;
            }
            else {
                queueProps.encryptionMasterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        const queue = new sqs.Queue(scope, id, queueProps);
        applySecureQueuePolicy(queue);
        // Return the queue
        return [queue, queue.encryptionMasterKey];
    }
    else {
        // If an existingQueueObj is specified, return that object as the queue to be used
        return [props.existingQueueObj];
    }
}
exports.buildQueue = buildQueue;
function buildDeadLetterQueue(scope, props) {
    if (!props.existingQueueObj && (props.deployDeadLetterQueue || props.deployDeadLetterQueue === undefined)) {
        // Create the Dead Letter Queue
        const [dlq] = buildQueue(scope, 'deadLetterQueue', {
            queueProps: props.deadLetterQueueProps
        });
        const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
        // Setup the Dead Letter Queue interface
        const dlqInterface = {
            maxReceiveCount: mrc,
            queue: dlq
        };
        // Return the dead letter queue interface
        return dlqInterface;
    }
    return;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply Topic policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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