"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSagemakerEndpoint = exports.createSagemakerEndpointConfig = exports.createSagemakerModel = exports.deploySagemakerEndpoint = exports.BuildSagemakerEndpoint = exports.buildSagemakerNotebook = void 0;
const sagemaker = require("@aws-cdk/aws-sagemaker");
const ec2 = require("@aws-cdk/aws-ec2");
const kms_helper_1 = require("./kms-helper");
const sagemaker_defaults_1 = require("./sagemaker-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const vpc_helper_1 = require("./vpc-helper");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const vpc_defaults_1 = require("./vpc-defaults");
const security_group_helper_1 = require("./security-group-helper");
function addPermissions(_role, props) {
    var _a, _b;
    // Grant permissions to NoteBookInstance for creating and training the model
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${core_1.Aws.PARTITION}:sagemaker:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:*`],
        actions: [
            'sagemaker:CreateTrainingJob',
            'sagemaker:DescribeTrainingJob',
            'sagemaker:CreateModel',
            'sagemaker:DescribeModel',
            'sagemaker:DeleteModel',
            'sagemaker:CreateEndpoint',
            'sagemaker:CreateEndpointConfig',
            'sagemaker:DescribeEndpoint',
            'sagemaker:DescribeEndpointConfig',
            'sagemaker:DeleteEndpoint',
            'sagemaker:DeleteEndpointConfig',
            'sagemaker:InvokeEndpoint',
        ],
    }));
    // Grant CloudWatch Logging permissions
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/sagemaker/*`],
        actions: [
            'logs:CreateLogGroup',
            'logs:CreateLogStream',
            'logs:DescribeLogStreams',
            'logs:GetLogEvents',
            'logs:PutLogEvents',
        ],
    }));
    // To place the Sagemaker endpoint in a VPC
    if (props && props.vpc) {
        _role.addToPolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: [
                'ec2:CreateNetworkInterface',
                'ec2:CreateNetworkInterfacePermission',
                'ec2:DeleteNetworkInterface',
                'ec2:DeleteNetworkInterfacePermission',
                'ec2:DescribeNetworkInterfaces',
                'ec2:AssignPrivateIpAddresses',
                'ec2:UnassignPrivateIpAddresses',
                'ec2:DescribeVpcs',
                'ec2:DescribeDhcpOptions',
                'ec2:DescribeSubnets',
                'ec2:DescribeSecurityGroups',
            ],
        }));
    }
    // To create a Sagemaker model using Bring-Your-Own-Model (BYOM) algorith image
    // The image URL is specified in the modelProps
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [`arn:${cdk.Aws.PARTITION}:ecr:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:repository/*`],
        actions: [
            'ecr:BatchCheckLayerAvailability',
            'ecr:GetDownloadUrlForLayer',
            'ecr:DescribeRepositories',
            'ecr:DescribeImages',
            'ecr:BatchGetImage',
        ],
    }));
    // Add GetAuthorizationToken (it can not be bound to resources other than *)
    _role.addToPolicy(new iam.PolicyStatement({
        resources: ['*'],
        actions: ['ecr:GetAuthorizationToken'],
    }));
    // add permission to use Elastic Inference accelerator
    if (props && props.endpointConfigProps) {
        // Get the acceleratorType, if any
        const acceleratorType = ((_a = props.endpointConfigProps) === null || _a === void 0 ? void 0 : _a.productionVariants)[0].acceleratorType;
        if (acceleratorType !== undefined) {
            _role.addToPolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['elastic-inference:Connect'],
            }));
        }
    }
    // add kms permissions
    _role.addToPolicy(new iam.PolicyStatement({
        // the kmsKeyId in the endpointConfigProps can be any of the following formats:
        // Key ID: 1234abcd-12ab-34cd-56ef-1234567890ab
        // Key ARN: arn:aws:kms:<region>:<accountID>:key/1234abcd-12ab-34cd-56ef-1234567890ab
        // Alias name: alias/ExampleAlias
        // Alias name ARN: arn:aws:kms:<region>:<accountID>:alias/ExampleAlias
        // the key is used to encrypt/decrypt data captured by the Sagemaker endpoint and stored in S3 bucket
        resources: [
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:key/*`,
            `arn:${cdk.Aws.PARTITION}:kms:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:alias/*`,
        ],
        actions: ['kms:Encrypt', 'kms:Decrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*', 'kms:DescribeKey'],
    }));
    // Add S3 permissions to get Model artifact, put data capture files, etc.
    _role.addToPolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject', 's3:PutObject', 's3:DeleteObject', 's3:ListBucket'],
        resources: ['arn:aws:s3:::*'],
    }));
    // Grant GetRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:GetRole'],
    }));
    // Grant PassRole permissions to the Sagemaker service
    _role.addToPolicy(new iam.PolicyStatement({
        resources: [_role.roleArn],
        actions: ['iam:PassRole'],
        conditions: {
            StringLike: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
        },
    }));
    // Add CFN NAG uppress to allow for "Resource": "*" for ENI access in VPC,
    // ECR authorization token for custom model images, and elastic inference
    // Add CFN NAG for Complex Role because Sagmaker needs permissions to access several services
    const roleDefualtPolicy = (_b = _role.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
    roleDefualtPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [
                {
                    id: 'W12',
                    reason: `Sagemaker needs the following minimum required permissions to access ENIs in a VPC, ECR for custom model images, and elastic inference.`,
                },
                {
                    id: 'W76',
                    reason: 'Complex role becuase Sagemaker needs permissions to access several services',
                },
            ],
        },
    };
}
function buildSagemakerNotebook(scope, props) {
    var _a, _b, _c, _d, _e, _f, _g, _h, _j;
    // Setup the notebook properties
    let sagemakerNotebookProps;
    let vpcInstance;
    let securityGroup;
    let kmsKeyId;
    let subnetId;
    // Conditional Sagemaker Notebook creation
    if (!props.existingNotebookObj) {
        if ((((_a = props.sagemakerNotebookProps) === null || _a === void 0 ? void 0 : _a.subnetId) && ((_b = props.sagemakerNotebookProps) === null || _b === void 0 ? void 0 : _b.securityGroupIds) === undefined) ||
            (((_c = props.sagemakerNotebookProps) === null || _c === void 0 ? void 0 : _c.subnetId) === undefined && ((_d = props.sagemakerNotebookProps) === null || _d === void 0 ? void 0 : _d.securityGroupIds))) {
            throw new Error('Must define both sagemakerNotebookProps.subnetId and sagemakerNotebookProps.securityGroupIds');
        }
        addPermissions(props.role);
        if (((_e = props.sagemakerNotebookProps) === null || _e === void 0 ? void 0 : _e.kmsKeyId) === undefined) {
            kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
        }
        else {
            kmsKeyId = props.sagemakerNotebookProps.kmsKeyId;
        }
        if (props.deployInsideVpc === undefined || props.deployInsideVpc) {
            if (((_f = props.sagemakerNotebookProps) === null || _f === void 0 ? void 0 : _f.subnetId) === undefined &&
                ((_g = props.sagemakerNotebookProps) === null || _g === void 0 ? void 0 : _g.securityGroupIds) === undefined) {
                vpcInstance = vpc_helper_1.buildVpc(scope, {
                    defaultVpcProps: vpc_defaults_1.DefaultPublicPrivateVpcProps(),
                });
                securityGroup = security_group_helper_1.buildSecurityGroup(scope, 'SecurityGroup', {
                    vpc: vpcInstance,
                    allowAllOutbound: false,
                }, [], [{ peer: ec2.Peer.anyIpv4(), connection: ec2.Port.tcp(443) }]);
                subnetId = vpcInstance.privateSubnets[0].subnetId;
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, subnetId, [
                    securityGroup.securityGroupId,
                ]);
            }
            else {
                sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId, (_h = props.sagemakerNotebookProps) === null || _h === void 0 ? void 0 : _h.subnetId, (_j = props.sagemakerNotebookProps) === null || _j === void 0 ? void 0 : _j.securityGroupIds);
            }
        }
        else {
            sagemakerNotebookProps = sagemaker_defaults_1.DefaultSagemakerNotebookProps(props.role.roleArn, kmsKeyId);
        }
        if (props.sagemakerNotebookProps) {
            sagemakerNotebookProps = utils_1.overrideProps(sagemakerNotebookProps, props.sagemakerNotebookProps);
        }
        // Create the notebook
        const sagemakerInstance = new sagemaker.CfnNotebookInstance(scope, 'SagemakerNotebook', sagemakerNotebookProps);
        if (vpcInstance) {
            return [sagemakerInstance, vpcInstance, securityGroup];
        }
        else {
            return [sagemakerInstance];
        }
    }
    else {
        // Return existing notebook object
        return [props.existingNotebookObj];
    }
}
exports.buildSagemakerNotebook = buildSagemakerNotebook;
function BuildSagemakerEndpoint(scope, props) {
    /** Conditional Sagemaker endpoint creation */
    if (!props.existingSagemakerEndpointObj) {
        if (props.modelProps) {
            /** return [endpoint, endpointConfig, model] */
            return deploySagemakerEndpoint(scope, props);
        }
        else {
            throw Error('Either existingSagemakerEndpointObj or at least modelProps is required');
        }
    }
    else {
        /** Otherwise, return [endpoint] */
        return [props.existingSagemakerEndpointObj];
    }
}
exports.BuildSagemakerEndpoint = BuildSagemakerEndpoint;
function deploySagemakerEndpoint(scope, props) {
    let model;
    let endpointConfig;
    let endpoint;
    let sagemakerRole;
    // Create Sagemaker's model, endpointConfig, and endpoint
    if (props.modelProps) {
        // Check if the client has provided executionRoleArn
        if (props.modelProps.executionRoleArn) {
            sagemakerRole = iam.Role.fromRoleArn(scope, 'SagemakerRoleCustomer', props.modelProps.executionRoleArn);
        }
        else {
            // Create the Sagemaker Role
            sagemakerRole = new iam.Role(scope, 'SagemakerRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
            });
            // Add required permissions
            addPermissions(sagemakerRole, props);
        }
        // Create Sagemaker Model
        model = createSagemakerModel(scope, props.modelProps, sagemakerRole, props.vpc);
        // Create Sagemaker EndpointConfig
        endpointConfig = createSagemakerEndpointConfig(scope, model.attrModelName, props.endpointConfigProps);
        // Add dependency on model
        endpointConfig.addDependsOn(model);
        // Create Sagemaker Endpoint
        endpoint = createSagemakerEndpoint(scope, endpointConfig.attrEndpointConfigName, props.endpointProps);
        // Add dependency on EndpointConfig
        endpoint.addDependsOn(endpointConfig);
        return [endpoint, endpointConfig, model];
    }
    else {
        throw Error('You need to provide at least modelProps to create Sagemaker Endpoint');
    }
}
exports.deploySagemakerEndpoint = deploySagemakerEndpoint;
function createSagemakerModel(scope, modelProps, role, vpc) {
    let finalModelProps;
    let primaryContainer;
    let vpcConfig;
    let model;
    if (vpc) {
        const modelDefaultSecurityGroup = new ec2.SecurityGroup(scope, 'ReplaceModelDefaultSecurityGroup', {
            vpc,
            allowAllOutbound: true,
        });
        // Allow https traffic from within the VPC
        modelDefaultSecurityGroup.addIngressRule(ec2.Peer.ipv4(vpc.vpcCidrBlock), ec2.Port.tcp(443));
        const cfnSecurityGroup = modelDefaultSecurityGroup.node.findChild('Resource');
        cfnSecurityGroup.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [
                    {
                        id: 'W5',
                        reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
                    },
                    {
                        id: 'W40',
                        reason: 'Egress IPProtocol of -1 is default and generally considered OK',
                    },
                ],
            },
        };
        // Throw an error if the VPC does not contain private or isolated subnets
        if (vpc.privateSubnets.length === 0 && vpc.isolatedSubnets.length === 0) {
            throw Error('VPC must contain private or isolated subnets to deploy the Sagemaker endpoint in a vpc');
        }
        vpcConfig = {
            // default SubnetType.PRIVATE (or ISOLATED or PUBLIC if there are no PRIVATE subnets)
            // So, private subnets will be used if provided by customer. Otherwise, use the default isolated subnets,
            subnets: vpc.selectSubnets({
                onePerAz: true,
            }).subnetIds,
            securityGroupIds: [modelDefaultSecurityGroup.securityGroupId],
        };
    }
    if (modelProps.primaryContainer) {
        // Get user provided Model's primary container
        primaryContainer = modelProps.primaryContainer;
        // Get default Model props
        finalModelProps = sagemaker_defaults_1.DefaultSagemakerModelProps(role.roleArn, primaryContainer, vpcConfig);
        // Override default model properties
        finalModelProps = utils_1.overrideProps(finalModelProps, modelProps);
        // Create the Sagemaker's Model
        model = new sagemaker.CfnModel(scope, 'SagemakerModel', finalModelProps);
        // Add dependency on the Sagemaker's role
        model.node.addDependency(role);
        return model;
    }
    else {
        throw Error('You need to provide at least primaryContainer to create Sagemaker Model');
    }
}
exports.createSagemakerModel = createSagemakerModel;
function createSagemakerEndpointConfig(scope, modelName, endpointConfigProps) {
    let finalEndpointConfigProps;
    let kmsKeyId;
    let endpointConfig;
    // Create encryption key if one is not provided
    if (endpointConfigProps && endpointConfigProps.kmsKeyId) {
        kmsKeyId = endpointConfigProps.kmsKeyId;
    }
    else {
        kmsKeyId = kms_helper_1.buildEncryptionKey(scope).keyId;
    }
    // Get default EndpointConfig props
    finalEndpointConfigProps = sagemaker_defaults_1.DefaultSagemakerEndpointConfigProps(modelName, kmsKeyId);
    // Overwrite default EndpointConfig properties
    if (endpointConfigProps) {
        finalEndpointConfigProps = utils_1.overrideProps(finalEndpointConfigProps, endpointConfigProps);
    }
    // Create the Sagemaker's EndpointConfig
    endpointConfig = new sagemaker.CfnEndpointConfig(scope, 'SagemakerEndpointConfig', finalEndpointConfigProps);
    return endpointConfig;
}
exports.createSagemakerEndpointConfig = createSagemakerEndpointConfig;
function createSagemakerEndpoint(scope, endpointConfigName, endpointProps) {
    let finalEndpointProps;
    let endpoint;
    // Get default Endpoint props
    finalEndpointProps = sagemaker_defaults_1.DefaultSagemakerEndpointProps(endpointConfigName);
    // Overwrite default Endpoint properties
    if (endpointProps) {
        finalEndpointProps = utils_1.overrideProps(finalEndpointProps, endpointProps);
    }
    // Create the Sagemaker's Endpoint
    endpoint = new sagemaker.CfnEndpoint(scope, 'SagemakerEndpoint', finalEndpointProps);
    return endpoint;
}
exports.createSagemakerEndpoint = createSagemakerEndpoint;
//# sourceMappingURL=data:application/json;base64,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