"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.createLoggingBucket = exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const rule_1 = require("@aws-cdk/aws-s3/lib/rule");
const duration_1 = require("@aws-cdk/core/lib/duration");
function buildS3Bucket(scope, props, bucketId) {
    if (props.bucketProps) {
        return s3BucketWithLogging(scope, props.bucketProps, bucketId);
    }
    else {
        return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
    }
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function createLoggingBucket(scope, bucketId, removalPolicy) {
    let loggingBucketProps;
    if (removalPolicy) {
        loggingBucketProps = utils_1.overrideProps(s3_bucket_defaults_1.DefaultS3Props(), { removalPolicy });
    }
    else {
        loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
    }
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    loggingBucketResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W35',
                    reason: _reason
                }]
        }
    };
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: rule_1.StorageClass.GLACIER,
                    transitionAfter: duration_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if (s3BucketProps.versioned === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(undefined, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props();
        }
    }
    else {
        // Create the Logging Bucket
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.removalPolicy);
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if ((s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.versioned) === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
        }
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    fnResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
                },
                {
                    id: 'W89',
                    reason: `This is not a rule for the general case, just for specific use cases/industries`
                },
                {
                    id: 'W92',
                    reason: `Impossible for us to define the correct concurrency for clients`
                }]
        }
    };
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    policyResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
                }]
        }
    };
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,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